/**********************************************************************************************
 Copyright (c) 2014 DisplayModule. All rights reserved.

 Redistribution and use of this source code, part of this source code or any compiled binary
 based on this source code is permitted as long as the above copyright notice and following
 disclaimer is retained.

 DISCLAIMER:
 THIS SOFTWARE IS SUPPLIED "AS IS" WITHOUT ANY WARRANTIES AND SUPPORT. DISPLAYMODULE ASSUMES
 NO RESPONSIBILITY OR LIABILITY FOR THE USE OF THE SOFTWARE.
 ********************************************************************************************/

/******************************************************************************
 * Includes
 *****************************************************************************/

#include "mbed.h"

#include "DmTftHX8353C.h"
#include "DmTftS6D0164.h"
#include "DmTftIli9325.h"
#include "DmTftIli9341.h"
#include "DmTftSsd2119.h"
#include "DmTouch.h"

#include "Canvas.h"

/******************************************************************************
 * Typedefs and defines
 *****************************************************************************/

typedef enum {
  Idle,
  SelectingColor,
  Drawing,
} paintMode_t;

typedef struct {
  uint16_t color;
  Canvas* canvas;
} colorButton_t;

/* Note that there are restrictions on which platforms that can use printf
   in combinations with the DmTftLibrary. Some platforms (e.g. LPC1549 LPCXpresso)
   use the same pins for USBRX/USBTX and display control. Printing will
   cause the display to not work. Read more about this on the display's notebook
   page. */
//#define log(...) printf(__VA_ARGS__)
#define log(...)

#define MARGIN       5
#define BOX_W       30
#define BOX_H       30
#define POINT_SIZE   2

#define NUM_BUTTONS  (sizeof(buttons)/sizeof(buttons[0]))

#if 0
  /* Displays without adapter */
  #define DM_PIN_SPI_MOSI   D11
  #define DM_PIN_SPI_MISO   D12
  #define DM_PIN_SPI_SCLK   D13
  #define DM_PIN_CS_TOUCH   D4
  #define DM_PIN_CS_TFT     D10
  #define DM_PIN_CS_SDCARD  D8
  #define DM_PIN_CS_FLASH   D6
#else
  /* Displays with adapter */
  #define DM_PIN_SPI_MOSI   A0
  #define DM_PIN_SPI_MISO   D9
  #define DM_PIN_SPI_SCLK   A1
  #define DM_PIN_CS_TOUCH   D8
  #define DM_PIN_CS_TFT     A3
  #define DM_PIN_CS_SDCARD  D10
#endif

/******************************************************************************
 * Local variables
 *****************************************************************************/

DmTftIli9325 tft;  /* DM_TFT28_103 and DM_TFT24_104 */
//DmTftIli9341 tft;  /* DM_TFT28_105 */
//DmTftSsd2119 tft;   /* DM_TFT35_107 */

DmTouch touch(DmTouch::DM_TFT28_103, DmTouch::Software); /* For LPC4088 QuickStart Board */
//DmTouch touch(DmTouch::DM_TFT28_103);
//DmTouch touch(DmTouch::DM_TFT24_104, DmTouch::Software); /* For LPC4088 QuickStart Board */
//DmTouch touch(DmTouch::DM_TFT24_104);
//DmTouch touch(DmTouch::DM_TFT28_105);
//DmTouch touch(DmTouch::DM_TFT35_107);

DigitalInOut csTouch(DM_PIN_CS_TOUCH, PIN_OUTPUT, PullUp, 1);
DigitalInOut csDisplay(DM_PIN_CS_TFT, PIN_OUTPUT, PullUp, 1);
DigitalInOut csSDCard(DM_PIN_CS_SDCARD, PIN_OUTPUT, PullUp, 1);
#ifdef DM_PIN_CS_FLASH
  DigitalInOut csFlash(DM_PIN_CS_FLASH, PIN_OUTPUT, PullUp, 1);
#endif  

/******************************************************************************
 * Global variables
 *****************************************************************************/


/******************************************************************************
 * Main
 *****************************************************************************/

int main() {
  uint16_t x, y, lastX, lastY;
  bool penDown;
  paintMode_t mode = Idle;
  uint16_t color = WHITE;
  int activeIdx = 4;
  int selIdx = 0;

  colorButton_t buttons[] = {
    { RED, new Canvas(MARGIN, MARGIN, BOX_W, BOX_H, BLACK, RED, 3, GRAY1) },
    { GREEN, new Canvas(MARGIN+(BOX_W + MARGIN), MARGIN, BOX_W, BOX_H, BLACK, GREEN, 3, GRAY1) },
    { BLUE, new Canvas(MARGIN+(BOX_W + MARGIN)*2, MARGIN, BOX_W, BOX_H, BLACK, BLUE, 3, GRAY1) },
    { YELLOW, new Canvas(MARGIN+(BOX_W + MARGIN)*3, MARGIN, BOX_W, BOX_H, BLACK, YELLOW, 3, GRAY1) },
    { WHITE, new Canvas(MARGIN+(BOX_W + MARGIN)*4, MARGIN, BOX_W, BOX_H, BLACK, WHITE, 3, GRAY1) },
  };
  
  tft.init();
  
  for (int i = 0; i < NUM_BUTTONS; i++) {
    if (color != buttons[i].color) {
      buttons[i].canvas->enableBorder(false);
    }
    buttons[i].canvas->draw(&tft);
  }

  Canvas drawArea(MARGIN, MARGIN+(BOX_H + MARGIN), tft.width(), tft.height() - MARGIN+(BOX_H + MARGIN));
  Canvas clearButton(tft.width() - 50 - MARGIN, MARGIN, 50, BOX_H, WHITE, BLACK, 1, WHITE);

  drawArea.draw(&tft);
  clearButton.draw(&tft);
  clearButton.drawString(&tft, "clr");
  
  touch.init();

  lastX = lastY = 0;  
  while(true)
  {
    touch.readTouchData(x, y, penDown);
    switch (mode)
    {
      case Idle:
        if (penDown)
        {
          if (drawArea.isInside(x, y))
          {
            tft.drawPoint(x, y, POINT_SIZE);
            mode = Drawing;
            log("start draw\n");
          } else if (clearButton.isInside(x, y)) {
            log("Pressed clear!\n");
            drawArea.draw(&tft);
          } else {
            for (int i = 0; i < NUM_BUTTONS; i++) {
              if (buttons[i].canvas->isInside(x, y)) {
                mode = SelectingColor;
                selIdx = i;
                log("Pen down on button %d\n", i);
              }
            }
          }
        }
        break;
        
      case SelectingColor:
        if (!penDown)
        {
          if (buttons[selIdx].canvas->isInside(lastX, lastY)) {
            buttons[activeIdx].canvas->enableBorder(false);
            buttons[activeIdx].canvas->draw(&tft);

            log("Pen up => actual click on color %d!\n", selIdx);
            
            color = buttons[selIdx].color;
            tft.setTextColor(BLACK, color);

            activeIdx = selIdx;
            buttons[activeIdx].canvas->enableBorder(true);
            buttons[activeIdx].canvas->draw(&tft);
          }
          
          mode = Idle;
        }
        break;
        
      case Drawing:
        if (penDown && drawArea.isInside(x, y)) {
          tft.drawPoint(x, y, POINT_SIZE);
          log("painting\n");
        } else if (!penDown) {
          mode = Idle;
          log("end draw\n");
        }
        break;
    }
    lastX = x;
    lastY = y;
    wait(0.002);
  }
}
