#include "mbed.h"
#include "callbacks.h"

/** A class which can communicate with a BlueGiga BLE112(3) over a UART Connection. Note that all communication is asynchronous. Your class will have to supply callbacks for each command it intends to use.
* 
* Example:
* @code
* #include "mbed.h"
* #include "BGLib.h"
* 
* BGLib ble112(p9, p10, p7, p8);
* 
* void helloCallback() {
*    printf("BLE112 said hello!\r\n");
* }
* 
* int main() {
*     ble112.set_ble_rsp_system_hello(&helloCallback);
*     ble112.ble_cmd_system_hello();
* }
* @endcode
*/

class BGLib {


public:
    /** Create a BGLib instance
    * @param tx Pin to use for UART transmission
    * @param rx Pin to use for UART reception
    * @param rts Flow control pin used for RTS.
    * @param cts Flow control pin used for CTS.
    */
    BGLib(PinName tx, PinName rx, PinName rts, PinName cts);
    
    /** GAP Discoverable modes */
    enum gap_discover_mode {
        gap_discover_limited = 0, ///< Discover only limited discoverable devices, that is, Slaves which have the LE Limited Discoverable Mode bit set in the Flags AD type of their advertisement packets.
        gap_discover_generic = 1, ///<  Discover limited and generic discoverable devices, that is, Slaves which have the LE Limited Discoverable Mode or the LE General Discoverable Mode bit set in the Flags AD type of their advertisement packets.
        gap_discover_observation = 2 ///< Discover all devices regardless of the Flags AD type, so also devices in non-discoverable mode will be reported to host.
    };
    
    /** Send a Hello command to the device. */
    void ble_cmd_system_hello();
    
    /** Set the callback for a Hello command. Invoked when a response arrives.
    * @param pCallback Function pointer to the desired callback.
    */
    void set_ble_rsp_system_hello(hello_callback_t pCallback); 
    
    /** Send a Get Info command to the device. */
    void ble_cmd_system_get_info();
    
    /** Set the callback for a Get Info command. Invoked when a response arrives.
    * @param pCallback Function pointer to the desired callback.
    */
    void set_ble_rsp_system_get_info(get_info_callback_t pCallback);
    
    /** Send a Reset command to the device.
    * @param args A struct containing the DFU flag.  
    */
    void ble_cmd_system_reset(ble_msg_system_reset_t args);
    
    /** Set the callback for a Boot event. Invoked when the BLE112 is powered on or reset. 
    * @param pCallback Function pointer to the desired callback.
    */
    void set_ble_evt_system_boot(boot_callback_t pCallback);
    
    /** Set a "timestamping" callback. This is invoked whenever the mbed receives <em>any</em> data on the UART interface.
    * @param pCallback Function pointer to the desired callback.
    */
    void set_timestamp_callback(timestamp_callback_t pCallback);
    
    /** Send a Discover command to the device. 
    * @param mode The mode that the device should do discovery in.
    */
    void ble_cmd_gap_discover(gap_discover_mode mode);
    
    /** Set the callback for a Discover command.
    * @param pCallback Function pointer to the desired callback.
    */
    void set_ble_rsp_gap_discover(discover_callback_t pCallback);
    
    /** Set the callback for Scan Result events.
    * @param pCallback Function pointer to the desired callback.
    */
    void set_ble_evt_gap_scan_result(scan_result_callback_t pCallback);
        
    
private:
    /** Function that handles parsing responses on the UART port. */
    void parse();
    
    /** Function that sends a specific number of bytes via the UART port.
    * @attention This function doesn't do any bounds checking. Be careful with what you put in!
    */
    void send_bytes(uint8_t bytes[], int length);
    
    /** Serial Port object. */
    Serial mSerial;
    
    
    // Callbacks
    hello_callback_t mHelloCallback;
    get_info_callback_t mGetInfoCallback;
    boot_callback_t mBootCallback;
    timestamp_callback_t mTimestampCallback;
    discover_callback_t mDiscoverCallback;
    scan_result_callback_t mScanResultCallback;
    
};