#include "BGLib.h"

BGLib::BGLib(PinName tx, PinName rx, PinName rts, PinName cts) :
    mSerial(tx, rx) {
    mSerial.set_flow_control(SerialBase::RTSCTS, rts, cts);
    mSerial.baud(57600);
    mSerial.attach(this, &BGLib::parse);
}

void BGLib::set_ble_rsp_system_hello(hello_callback_t pCallback) {
    mHelloCallback = pCallback;
}

void BGLib::ble_cmd_system_hello() {
    uint8_t bytes[] = {0x00, 0x00, 0x00, 0x01};
    send_bytes(bytes, 4);
}

void BGLib::ble_cmd_system_get_info() {
    uint8_t bytes[] = {0x00, 0x00, 0x00, 0x08};
    send_bytes(bytes, 4);
}

void BGLib::set_ble_rsp_system_get_info(get_info_callback_t pCallback) {
    mGetInfoCallback = pCallback;
}

void BGLib::set_ble_evt_system_boot(boot_callback_t pCallback) {
    mBootCallback = pCallback;
}

void BGLib::set_ble_rsp_gap_discover(discover_callback_t pCallback) {
    mDiscoverCallback = pCallback;
}

void BGLib::ble_cmd_system_reset(ble_msg_system_reset_t args) {
    uint8_t bytes[] = {0x00, 0x01, 0x00, 0x00, 0x00};
    bytes[4] = args.boot_in_dfu;
    send_bytes(bytes, 5);
}

void BGLib::set_timestamp_callback(timestamp_callback_t pCallback) {
    mTimestampCallback = pCallback;
}

void BGLib::set_ble_evt_gap_scan_result(scan_result_callback_t pCallback) {
    mScanResultCallback = pCallback;
}

void BGLib::ble_cmd_gap_discover(gap_discover_mode mode) {
    uint8_t bytes[] = {0x00, 0x01, 0x06, 0x02, 0x00};
    bytes[4] = (uint8_t) mode;
    send_bytes(bytes, 5);
}

void BGLib::parse() {
    mTimestampCallback();
    
    #ifdef DEBUG
    printf("Got data from port!\r\n");
    #endif
    
    uint8_t hilen = mSerial.getc();
    uint8_t lolen = mSerial.getc();
    uint8_t msg_class = mSerial.getc();
    uint8_t msg_id = mSerial.getc();
    
    #ifdef DEBUG
    printf("Message header: %x %x %x %x\r\n", hilen, lolen, msg_class, msg_id);
    #endif
    
    if (hilen == 0x00) { // response
        if (msg_class == 0x00) { // system_class
            if (msg_id == 0x01) { // system_hello
                mHelloCallback();
            } else if (msg_id == 0x08) { //system_get_info
                
                #ifdef DEBUG
                printf("Get Info Response\r\n");
                #endif
                
                ble_msg_system_get_info_rsp_t result;
                result.major = mSerial.getc() | (mSerial.getc() << 8);
                result.minor = mSerial.getc() | (mSerial.getc() << 8);
                result.patch = mSerial.getc() | (mSerial.getc() << 8);
                result.build = mSerial.getc() | (mSerial.getc() << 8);
                result.ll_version = mSerial.getc() | (mSerial.getc() << 8);
                result.protocol_version = mSerial.getc();
                result.hw = mSerial.getc();
                mGetInfoCallback(result);
            }
        } else if (msg_class == 0x06) { // gap_class
            if (msg_id == 0x02) { // gap_discover
                ble_msg_gap_discover_rsp_t result; 
                result.result = mSerial.getc() | (mSerial.getc() << 8);
                mDiscoverCallback(result);
            }
        }         
    } else if (hilen == 0x80) { // event
        if (msg_class == 0x00) { //system_class
            if (msg_id == 0x00) { //system_boot
                #ifdef DEBUG
                printf("Boot Event\r\n");
                #endif
                
                ble_msg_system_boot_evt_t result; 
                result.major = mSerial.getc() | (mSerial.getc() << 8);
                result.minor = mSerial.getc() | (mSerial.getc() << 8);
                result.patch = mSerial.getc() | (mSerial.getc() << 8);
                result.build = mSerial.getc() | (mSerial.getc() << 8);
                result.ll_version = mSerial.getc() | (mSerial.getc() << 8);
                result.protocol_version = mSerial.getc();
                result.hw = mSerial.getc();
                mBootCallback(result);
            }
        } else if (msg_class == 0x06) { // gap_class
            if (msg_id == 0x00) { // gap_scan_result
                #ifdef DEBUG
                printf("Scan Result Event\r\n");
                #endif
                
                ble_msg_gap_scan_result_evt_t result;
                
                result.rssi = mSerial.getc();
                
                result.packet_type = mSerial.getc();
                
                for (int i = 5; i >= 0; i--) {
                    result.hw_addr[i] = mSerial.getc();
                }
                
                result.address_type = mSerial.getc();
                
                result.bond = mSerial.getc();
                
                result.data_len = mSerial.getc();
                
                result.data = new uint8_t[result.data_len];
                for (int i = result.data_len - 1; i >= 0; i--) {
                    result.data[i] = mSerial.getc();
                }
                mScanResultCallback(result);
            }
        }
    }
        
    
    //safety valve: if there are bytes remaining
}

void BGLib::send_bytes(uint8_t bytes[], int length) {
    for (int i = 0; i < length; i++) {
        mSerial.putc(bytes[i]);
    }
}