#ifndef MICROCHIP_23LCXXX_SPI_H
#define MICROCHIP_23LCXXX_SPI_H

#include <mbed.h>

/** Microchip SPI SRAM access library for 23LC1024, 23LC512
 *
 * Example:
 * @code
 * #include <mbed.h>
 * #include "23LCxx_SPI.h"
 * 
 * Microchip23LCxx sram(SPI_MOSI, SPI_MISO, SPI_SCK, SPI_CS, 20 * 1000 * 1000);
 * 
 * uint8_t buf[256] = {};
 * 
 * void main() {
 *     const uint32_t address = 0x2000;
 *     const uint32_t size = 256;
 * 
 *     sram.change_mode(Microchip23LCxx::SEQUENTIAL);
 *     
 *     // write
 *     for (uint32_t i = 0; i < size; ++i) {
 *         buf[i] = i;
 *     }
 *     sram.write_bytes(address, buf, size);
 * 
 *     // read
 *     for (uint32_t i = 0; i < size; ++i) {
 *         buf[i] = 0;
 *     }
 *     sram.read_bytes(address, buf, size);
 * }
 * @endcode
 */
class Microchip23LCxx {
public:
    /** SPI COMMAND for Microchip 23LC1024, 23LC512 */
    enum Microchip23LCxxCommnd {
        READ  = 0x03u,  ///< Read data from memory array beginning at selected address
        WRITE = 0x02u,  ///< Write data to memory array beginning at selected address
        EDIO  = 0x3bu,  ///< Enter Dual I/O access (enter SDI bus mode)
        EQIO  = 0x38u,  ///< Enter Quad I/O access (enter QDI bus mode)
        RSTIO = 0xffu,  ///< Reset Dual and Quad I/O access (revert to SPI bus mode)
        RDMR  = 0x05u,  ///< Read Mode Register
        WRMR  = 0x01u,  ///< Write Mode Register
    };

    /** Access mode for Microchip 23LC1024, 23LC512 */
    enum Microchip23LCxxMode {
        MODE_MASK  = 0xc0u, ///< mode bits [7:6] in mode register
        BYTE       = 0x00u, ///< 00: Byte mode
        SEQUENTIAL = 0x40u, ///< 01: Sequential mode (default)
        PAGE       = 0x80u, ///< 10: Page mode
        RESERVED   = 0xc0u, ///< 11: Reserved
    };

    /** Constructor.
     *
     * @param[in] mosi SPI MOSI pin name
     * @param[in] miso SPI MISO pin name
     * @param[in] sck SPI SCK pin name
     * @param[in] cs SPI CS pin name
     * @param[in] hz SPI Frequency
     */
    Microchip23LCxx(const PinName mosi, const PinName miso, const PinName sck, const PinName cs, const uint32_t hz);

    /** read from mode register.
     *
     * @return register value
     */
    uint8_t read_mode_register();

    /** write to mode register.
     *
     * @param[in] mode
     */
    void write_mode_register(const uint8_t value);

    /** change mode bits.
     *
     * @param[in] mode mode
     * @return mode before change
     * @invariant other bits in register.
     */
    uint8_t change_mode(const uint8_t next_mode);

    /** 1byte read
     *
     * @param[in] address adress in 24bit
     * @return read data
     */
    uint8_t read_byte(const uint32_t address);

    /** 1byte write
     *
     * @param[in] address address in 24bit
     * @param[in] write data
     */
    void write_byte(const uint32_t address, const uint8_t data);

    /** multi-byte read
     *
     * @pre sequential mode or page mode is required.
     */
    void read_bytes(const uint32_t address, uint8_t __restrict data[], const uint32_t size);

    /** multi-byte write
     *
     * @pre sequential mode or page mode is required.
     */
    void write_bytes(const uint32_t address, const uint8_t __restrict data[], const uint32_t size);

private:
    SPI _spi;
    DigitalOut _cs;
    void _set_address(const uint32_t address);
};

#endif
