
#include <mbed.h>
#include "23LCxx_SPI.h"

Microchip23LCxx::Microchip23LCxx(const PinName mosi, const PinName miso, const PinName sck, const PinName cs, const uint32_t hz):
_spi(mosi, miso, sck), _cs(cs, 1) {
    _spi.format(8, 0);   // 8bit, mode=0
    _spi.frequency(hz);
}

uint8_t
Microchip23LCxx::read_mode_register() {
    _cs = 0;
    _spi.write(RDMR);
    const uint8_t value = _spi.write(0x00);
    _cs = 1;
    
    return value;
}

void
Microchip23LCxx::write_mode_register(const uint8_t value) {
    _cs = 0;
    _spi.write(WRMR);
    _spi.write(value);
    _cs = 1;
}

uint8_t
Microchip23LCxx::change_mode(const uint8_t next_mode) {
    const uint8_t previous_register = read_mode_register();
    const uint8_t previous_mode = previous_register & MODE_MASK;
    if (next_mode != previous_mode) {
        const uint8_t next_register = (previous_register & ~MODE_MASK) | uint8_t(next_mode);
        write_mode_register(next_register);
    }
    return previous_mode;
}

void
Microchip23LCxx::_set_address(const uint32_t address) {
    const uint8_t address_high = (address >> (8 * 2)) & 0xFFu;
    const uint8_t address_mid  = (address >> (8 * 1)) & 0xFFu;
    const uint8_t address_low  = (address >> (8 * 0)) & 0xFFu;
    _spi.write(address_high);
    _spi.write(address_mid);
    _spi.write(address_low);
}

uint8_t
Microchip23LCxx::read_byte(const uint32_t address) {
    _cs = 0;
    _spi.write(READ);
    _set_address(address);
    const uint8_t data = _spi.write(0);
    _cs = 1;

    return data;
}

void
Microchip23LCxx::write_byte(const uint32_t address, const uint8_t data) {
    _cs = 0;
    _spi.write(WRITE);
    _set_address(address);
    _spi.write(data);
    _cs = 1;
}

/**
 * multi-byte read
 * @pre sequential mode or page mode is required.
 */
void
Microchip23LCxx::read_bytes(const uint32_t address, uint8_t __restrict data[], const uint32_t size) {
    _cs = 0;
    _spi.write(READ);
    _set_address(address);
    for (uint32_t i = 0; i < size; ++i) {
        data[i] = _spi.write(0x00u);
    }
    _cs = 1;
}

/**
 * multi-byte write
 * @pre sequential mode or page mode is required.
 */
void
Microchip23LCxx::write_bytes(const uint32_t address, const uint8_t __restrict data[], const uint32_t size) {
    _cs = 0;
    _spi.write(WRITE);
    _set_address(address);
    for (uint32_t i = 0; i < size; ++i) {
        _spi.write(data[i]);
    }
    _cs = 1;
}
