/* 3pi Lab 3 PID example

(c) Dr Craig A. Evans, University of Leeds

June 2017

*/

#include "mbed.h"
#include "m3pi.h"

// API objects
m3pi robot;

BusOut leds(LED4,LED3,LED2,LED1);

DigitalIn button_A(p18);
DigitalIn button_B(p17);
DigitalIn button_X(p21);
DigitalIn button_Y(p22);
DigitalIn button_enter(p24);
DigitalIn button_back(p23);

AnalogIn pot_P(p15);
AnalogIn pot_I(p16);
AnalogIn pot_D(p19);
AnalogIn pot_S(p20);

// Function Prototypes
void init();
void welcome();
void print_menu();
void calibrate();
void line_follow();
void tune();

// Globals
const char g_song[] = "!T240 L8 a gafaeada c+adaeafa >aa>bac#ada c#adaeaf4";
const char g_song2[]= "L16 cdegreg4";

// some random default values for the PID parameters
float kp = 0.2;
float ki = 0.0;
float kd = 0.0;
float speed = 0.1;

// maximum values (may need changing)
float kp_max = 0.5;
float ki_max = 1.0;
float kd_max = 0.2;


// Main Function
int main()
{
    // initilise
    init();
    // welcome song and battery data
    welcome();

    // main loop
    while(1) {

        // print the main menu on the LCD
        print_menu();

        // check the buttons to see which mode to enter
        if (button_B.read() == 0) {
            calibrate();  // calibration
        } else if (button_Y.read() == 0) {
            line_follow();  // line-follow
        } else if (button_A.read() == 0) {
            tune();  // tuning the PID parameters
        } else if (button_X.read() == 0) {
            welcome();  // battery voltage etc.
        }

        wait(0.2);

    }
}

// Functions
void init()
{
    // initialise the robot
    robot.init();

    // turn on the pull-up resistor on the button
    // the button will read 1 when not pressed and 0 when pressed
    button_A.mode(PullUp);
    button_B.mode(PullUp);
    button_X.mode(PullUp);
    button_Y.mode(PullUp);
    button_enter.mode(PullUp);
    button_back.mode(PullUp);

    // turn off LEDs
    leds = 0b0000;
}

void welcome()
{
    // play welcome song, battery voltage and software version
    robot.lcd_clear();
    robot.play_music(g_song,sizeof(g_song));
    robot.display_battery_voltage(0,0);
    robot.display_signature(0,1);
    wait(3.0);

}

void print_menu()
{
    // text for the main menu
    robot.lcd_clear();
    robot.lcd_goto_xy(0,0);
    robot.lcd_print("pid inf",7);
    robot.lcd_goto_xy(0,1);
    robot.lcd_print("cal  go",7);
}

// button A
void tune()
{
    // Each mode has different LED pattern
    leds = 0b1000;
    wait(1.0);

    // keep tuning until back is pressed
    while (button_back.read() == 1) {

        robot.lcd_clear();
        char buffer[8];

        // if A is pressed, read P pot to get kp value
        if (button_A.read() == 0) {  // P
            kp = kp_max*pot_P.read();
            sprintf(buffer,"p=%.3f",kp);
            // if B is pressed, read I pot to get ki value
        } else if (button_B.read() == 0) {  // I
            ki = ki_max*pot_I.read();
            sprintf(buffer,"i=%.3f",ki);
            // if X is pressed, read P pot to get kd value
        } else if (button_X.read() == 0) {  // D
            kd = kd_max*pot_D.read();
            sprintf(buffer,"d=%.3f",kd);
            // if Y is pressed, read Speed pot to get speed value
        } else if (button_Y.read() == 0) {  // Speed
            speed = 1.0*pot_S.read();
            sprintf(buffer,"s=%.3f",speed);
        } else {
            memset(&buffer[0], ' ', sizeof(buffer));
        }

        // print new value on LCD
        robot.lcd_goto_xy(0,1);
        robot.lcd_print(buffer,7);

        wait(0.1);

    }

    robot.lcd_clear();
    leds = 0x00;
    wait(1.0);
    //go back to main menu
}

// button B
void calibrate()
{
    // Each mode has different LED pattern
    leds = 0b0100;
    robot.reset_calibration();

    // display message on LCD
    robot.lcd_clear();
    robot.lcd_goto_xy(0,0);
    robot.lcd_print(" Press",6);
    robot.lcd_goto_xy(0,1);
    robot.lcd_print(" Enter",6);
    wait(1.0);

    while (button_enter.read() == 1) {  // wait for button press

    }
    robot.lcd_clear();

    wait(2.0);
    robot.auto_calibrate();

    robot.lcd_goto_xy(0,0);
    robot.lcd_print(" Press",6);
    robot.lcd_goto_xy(0,1);
    robot.lcd_print("  Back",6);

    while (button_back.read() == 1) {  // wait for button press

    }
    robot.lcd_clear();

    leds = 0x00;
    wait(1.0);
    //go back to main menu
}

// button Y
void line_follow()
{
    // Each mode has different LED pattern
    leds = 0b0001;
    robot.lcd_clear();
    robot.lcd_goto_xy(0,0);
    robot.lcd_print(" Press",6);
    robot.lcd_goto_xy(0,1);
    robot.lcd_print(" Enter",6);
    wait(1.0);

    while (button_enter.read() == 1) {  // wait for button press

    }
    robot.lcd_clear();

    wait(2.0);  // give chance to move hand away
    leds = 0b0000;

    // array to store sensor values
    unsigned int values[5]= {0};

    // frequency
    float dt = 1.0/100.0;
    
    // stop looping when button is pressed
    while( button_back.read() == 1) {

        // get calibrated sensor values
        robot.get_calibrated_values(values);

        // in range -1 to 1, no error when line is at 0.0
        // number is negative when the line is to the left
        // number is positive when the line is to the right
        float error = robot.calc_line_position(values);
        
        // ADD in code here for I and P error

        float power = kp*error;

        robot.motors(speed+power,speed-power);

        wait(dt);

    }  // end loop

    robot.stop();
    robot.lcd_clear();

    leds = 0x00;
    wait(1.0);
    //go back to main menu

}



