#ifndef LP8_SERVICE_H
#define LP8_SERVICE_H

/* LP8 Gatt Service Class */

class LP8_Service 
{
public:

    //UUID descriptors
    const static uint16_t LP8_SERVICE_UUID                      = 0xA000;           // service identifier
    const static uint16_t LP8_STATE_CHARACTERISTIC_UUID         = 0xA001;           // CO2 concentration characteristic
    const static uint16_t LP8_READ_TEMP_UUID                    = 0x2A1F;           // temp (Celsius) characteristic
    const static uint16_t LP8_VCAP_UUID                         = 0xA010;           // Vcap characteristic (mV)
    const static uint16_t LP8_ERROR_UUID                        = 0xA011;           // LP8 Error bits characteristic  
    const static uint16_t LP8_WRITE_UUID                        = 0xA002;           // Write calculation control from app to lp8
    const static uint16_t LP8_WRITTEN_UUID                      = 0xB001;           // display what was written into lp8 calculation control
    
    const static uint16_t LP8_reWrite_uuid                      = 0xB111;
    
    
//constructor setup for Gatt Service
    LP8_Service(BLE         &_ble,                                                  /* Pass variables to characteristics and ble  */
                int         co2ValueInitial, 
                float       tempValueInitial, 
                int         initialVcap, 
                uint32_t    initialError, 
                uint8_t     initCC, 
                uint8_t     initSentCC):                                            
        ble(_ble), 
        lp8State(LP8_STATE_CHARACTERISTIC_UUID, &co2ValueInitial, GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY),
        lp8Temp(LP8_READ_TEMP_UUID, &tempValueInitial, GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY),
        lp8Vcap(LP8_VCAP_UUID, &initialVcap, GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY),
        lp8Error(LP8_ERROR_UUID, &initialError, GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY),
        lp8WriteCC(LP8_WRITE_UUID, &initCC),
        lp8Written(LP8_WRITTEN_UUID, &initSentCC, GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY),
        lp8reWriteCC(LP8_reWrite_uuid, &initCC, GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY)
        {
            //characterisitics 
            GattCharacteristic *charTable[] = {&lp8State, &lp8Temp, &lp8Vcap, &lp8Error, &lp8WriteCC, &lp8Written, &lp8reWriteCC };
             
            //Service, Setup for LP8 GattService
            GattService        lp8Service(LP8_SERVICE_UUID, charTable, sizeof(charTable) / sizeof(GattCharacteristic *));       
            
            //add gatt service to ble stack
            ble.addService(lp8Service);                                                                                         
        };


//update Gattserver with new values to characteristics
    void updateCo2Value(int co2Value)
    {
        ble.gattServer().write(lp8State.getValueAttribute().getHandle(),(uint8_t *) &co2Value, sizeof(co2Value));   
    };

    void updateTempValue(float tempValue)
    {
        ble.gattServer().write(lp8Temp.getValueAttribute().getHandle(),(uint8_t *) &tempValue, sizeof(tempValue));              
    };
    
    void updateVcapValue(int Vcap)
    {
        ble.gattServer().write(lp8Vcap.getValueAttribute().getHandle(),(uint8_t *) &Vcap, sizeof(Vcap));                        
    };
    
    void updateError(uint32_t lp8ErrorValue)
    {
        ble.gattServer().write(lp8Error.getValueAttribute().getHandle(),(uint8_t *) &lp8ErrorValue, sizeof(lp8ErrorValue));                        
    };

//     
    void updateDataWritten(uint8_t lp8WrittenValue)
    {
        ble.gattServer().write(lp8Written.getValueAttribute().getHandle(),(uint8_t *) &lp8WrittenValue, sizeof(lp8WrittenValue));
    }
    
    void updateReWrite ( uint8_t lp8ReWrite )
    {
        ble.gattServer().write(lp8reWriteCC.getValueAttribute().getHandle(),(uint8_t *) &lp8ReWrite, sizeof(lp8ReWrite));    
    }


private:

    BLE                                     &ble;       //
    
    //Service Characteristics
    
    //Read sensor values
    ReadOnlyGattCharacteristic<int>         lp8State;   //
    ReadOnlyGattCharacteristic<float>       lp8Temp;
    ReadOnlyGattCharacteristic<int>         lp8Vcap;
    ReadOnlyGattCharacteristic<uint32_t>    lp8Error;
    
    //Write Value
    WriteOnlyGattCharacteristic<uint8_t>    lp8WriteCC;
    
    //read back written and cc byte(flag)
    ReadOnlyGattCharacteristic<uint8_t>     lp8Written;
    ReadOnlyGattCharacteristic<uint8_t>     lp8reWriteCC;
    

};
#endif