/*
Copyright 2019 (c) Analog Devices, Inc.

All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:
  - Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.
  - Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in
    the documentation and/or other materials provided with the
    distribution.
  - Neither the name of Analog Devices, Inc. nor the names of its
    contributors may be used to endorse or promote products derived
    from this software without specific prior written permission.
  - The use of this software may or may not infringe the patent rights
    of one or more patent holders. This license does not release you
    from the requirement that you obtain separate licenses from these
    patent holders to use this software.
  - Use of the software either in source or binary form, must be run
    on or directly connected to an Analog Devices Inc. component.

THIS SOFTWARE IS PROVIDED BY ANALOG DEVICES "AS IS" AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, NON-INFRINGEMENT,
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL ANALOG DEVICES BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, INTELLECTUAL PROPERTY RIGHTS, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/*!
 ******************************************************************************
 * @file:
 * @brief:  Sensor type definitions for ADMW1001.
 *-----------------------------------------------------------------------------
 */

#ifndef __ADMW1001_SENSOR_TYPES_H__
#define __ADMW1001_SENSOR_TYPES_H__

/*! @addtogroup ADMW1001_Api
 *  @{
 */

#ifdef __cplusplus
extern "C" {
#endif

/*! ADMW1001 measurement channel identifiers */
typedef enum
{
    ADMW1001_CH_ID_NONE  = -1,
    /*!< Used to indicate when no channel is selected */

    ADMW1001_CH_ID_ANLG_1_UNIVERSAL = 0,
    /*!< Universal channel 1 universal channel */
    ADMW1001_CH_ID_ANLG_2_UNIVERSAL,
    /*!< Universal channel 2 universal channel */   
    ADMW1001_CH_ID_ANLG_1_DIFFERENTIAL,
    /*!< Universal channel 1 differential channel */
    ADMW1001_CH_ID_ANLG_2_DIFFERENTIAL,
    /*!< Universal channel 2 differential channel */ 
    
    ADMW1001_CH_ID_DIG_I2C_0,
    /*!< Digital I2C Sensor channel #0 */
    ADMW1001_CH_ID_DIG_I2C_1,
    /*!< Digital I2C Sensor channel #1 */
    ADMW1001_CH_ID_DIG_I2C_2,
    /*!< Digital I2C Sensor channel #2 */
    ADMW1001_CH_ID_DIG_I2C_3,
    /*!< Digital I2C Sensor channel #3 */
    ADMW1001_CH_ID_DIG_I2C_4,
    /*!< Digital I2C Sensor channel #4 */
    ADMW1001_CH_ID_DIG_I2C_5,
    /*!< Digital I2C Sensor channel #5 */
    
    ADMW1001_CH_ID_DIG_SPI_0,
    /*!< Digital SPI Sensor channel #0 */
    ADMW1001_CH_ID_DIG_SPI_1,
    /*!< Digital SPI Sensor channel #1 */
    ADMW1001_CH_ID_DIG_SPI_2,
    /*!< Digital SPI Sensor channel #2 */

    ADMW1001_MAX_CHANNELS,
    /*!< Maximum number of measurement channels on ADMW1001 */
    
}   ADMW1001_CH_ID;

/*! ADMW1001 analog sensor type options
 *
 *  Select the sensor type that is connected to an ADC analog measurement
 *  channel.
 *
 *  @note Some channels may only support a subset of the available sensor types
 *        below.
 *
 */
typedef enum
{
    ADMW1001_ADC_SENSOR_THERMOCOUPLE_T = 0,
    /*!< Standard T-type Thermocouple temperature sensor 
     *
     * @note For use with Analog Sensor channels only
     */
    ADMW1001_ADC_SENSOR_THERMOCOUPLE_J = 1,
    /*!< Standard J-type Thermocouple temperature sensor 
     *
     * @note For use with Analog Sensor channels only
     */
    ADMW1001_ADC_SENSOR_THERMOCOUPLE_K = 2,
    /*!< Standard K-type Thermocouple temperature sensor 
     *
     * @note For use with Analog Sensor channels only
     */
    ADMW1001_ADC_SENSOR_THERMOCOUPLE_1 = 8,
    /*!< Standard thermocouple temperature sensor 
     *
     * @note For use with Analog Sensor channels only
     */
    ADMW1001_ADC_SENSOR_THERMOCOUPLE_2 = 9,
    /*!< Standard thermocouple temperature sensor 
     *
     * @note For use with Analog Sensor channels only
     */
    ADMW1001_ADC_SENSOR_THERMOCOUPLE_3 = 10,
    /*!< Standard thermocouple temperature sensor 
     *
     * @note For use with Analog Sensor channels only
     */
    ADMW1001_ADC_SENSOR_THERMOCOUPLE_4 = 11,
    /*!< Standard thermocouple temperature sensor 
     *
     * @note For use with Analog Sensor channels only
     */
    ADMW1001_ADC_SENSOR_RTD_2WIRE_PT100 = 32,
    /*!< Standard 2-wire PT100 RTD temperature sensor 
     *
     * @note For use with Cold-Juction Compensation and Analog Sensor channels
     *       only
     */
    ADMW1001_ADC_SENSOR_RTD_2WIRE_PT1000 = 33,
    /*!< Standard 2-wire PT1000 RTD temperature 
     *
     * @note For use with Cold-Juction Compensation and Analog Sensor channels
     *       only
     */
    ADMW1001_ADC_SENSOR_RTD_2WIRE_1 = 40,
    /*!< 2-wire RTD temperature sensor 
     *
     * @note For use with Cold-Juction Compensation and Analog Sensor channels
     *       only
     */
    ADMW1001_ADC_SENSOR_RTD_2WIRE_2 = 41,
    /*!< 2-wire RTD temperature sensor 
     *
     * @note For use with Cold-Juction Compensation and Analog Sensor channels
     *       only
     */
    ADMW1001_ADC_SENSOR_RTD_2WIRE_3 = 42,
    /*!< 2-wire RTD temperature sensor 
     *
     * @note For use with Cold-Juction Compensation and Analog Sensor channels
     *       only
     */
    ADMW1001_ADC_SENSOR_RTD_2WIRE_4 = 43,
    /*!< 2-wire RTD temperature sensor 
     *
     * @note For use with Cold-Juction Compensation and Analog Sensor channels
     *       only
     */
    ADMW1001_ADC_SENSOR_RTD_3WIRE_PT100 = 64,
    /*!< Standard 3-wire PT100 RTD temperature sensor 
     *
     * @note For use with Analog Sensor channels only
     */
    ADMW1001_ADC_SENSOR_RTD_3WIRE_PT1000 = 65,
    /*!< Standard 3-wire PT1000 RTD temperature sensor 
     *
     * @note For use with Analog Sensor channels only
     */
    ADMW1001_ADC_SENSOR_RTD_3WIRE_1 = 72,
    /*!< 3-wire RTD temperature sensor 
     *
     * @note For use with Analog Sensor channels only
     */
    ADMW1001_ADC_SENSOR_RTD_3WIRE_2 = 73,
    /*!< 3-wire RTD temperature sensor 
     *
     * @note For use with Analog Sensor channels only
     */
    ADMW1001_ADC_SENSOR_RTD_3WIRE_3 = 74,
    /*!< 3-wire RTD temperature sensor 
     *
     * @note For use with Analog Sensor channels only
     */
    ADMW1001_ADC_SENSOR_RTD_3WIRE_4 = 75,
    /*!< 3-wire RTD temperature sensor 
     *
     * @note For use with Analog Sensor channels only
     */
    ADMW1001_ADC_SENSOR_RTD_4WIRE_PT100 = 96,
    /*!< Standard 4-wire PT100 RTD temperature sensor 
     *
     * @note For use with Analog Sensor channels only
     */
    ADMW1001_ADC_SENSOR_RTD_4WIRE_PT1000 = 97,
    /*!< Standard 4-wire PT1000 RTD temperature sensor 
     *
     * @note For use with Analog Sensor channels only
     */
    ADMW1001_ADC_SENSOR_RTD_4WIRE_1 = 104,
    /*!< 4-wire RTD temperature sensor 
     *
     * @note For use with Analog Sensor channels only
     */
    ADMW1001_ADC_SENSOR_RTD_4WIRE_2 = 105,
    /*!< 4-wire RTD temperature sensor 
     *
     * @note For use with Analog Sensor channels only
     */
    ADMW1001_ADC_SENSOR_RTD_4WIRE_3 = 106,
    /*!< 4-wire RTD temperature sensor 
     *
     * @note For use with Analog Sensor channels only
     */
    ADMW1001_ADC_SENSOR_RTD_4WIRE_4 = 107,
    /*!< 4-wire RTD temperature sensor 
     *
     * @note For use with Analog Sensor channels only
     */
    ADMW1001_ADC_SENSOR_THERMISTOR_A_10K = 128,
    /*!< Standard 10kOhm NTC Thermistor
     *
     * @note For use with Analog Sensor channels only
     */
    ADMW1001_ADC_SENSOR_THERMISTOR_B_10K = 129,
    /*!< Standard 10kOhm NTC Thermistor 
     *
     * @note For use with Analog Sensor channels only
     */
    ADMW1001_ADC_SENSOR_THERMISTOR_1 = 136,
    /*!< Thermistor sensor 
     *
     * @note For use with Analog Sensor channels only
     */
    ADMW1001_ADC_SENSOR_THERMISTOR_2 = 137,
    /*!< Thermistor sensor with user-defined linearisation and
     *   default configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADMW1001_ADC_SENSOR_THERMISTOR_3 = 138,
    /*!< Thermistor sensor 
     *
     * @note For use with Analog Sensor channels only
     */
    ADMW1001_ADC_SENSOR_THERMISTOR_4 = 139,
    /*!< Thermistor sensor 
     *
     * @note For use with Analog Sensor channels only
     */
    ADMW1001_ADC_SENSOR_BRIDGE_4WIRE_1 = 168,
    /*!< Standard 4-wire Bridge Transducer sensor 
     *
     * @note For use with Analog Sensor channels only
     * @note Bridge Excitation Voltage must be selected as reference
     */
    ADMW1001_ADC_SENSOR_BRIDGE_4WIRE_2 = 169,
    /*!< Standard 4-wire Bridge Transducer sensor 
     *
     * @note For use with Analog Sensor channels only
     * @note Bridge Excitation Voltage must be selected as reference
     */
    ADMW1001_ADC_SENSOR_BRIDGE_4WIRE_3 = 170,
    /*!< Standard 4-wire Bridge Transducer sensor 
     *
     * @note For use with Analog Sensor channels only
     * @note Bridge Excitation Voltage must be selected as reference
     */
    ADMW1001_ADC_SENSOR_BRIDGE_4WIRE_4 = 171,
    /*!< Standard 4-wire Bridge Transducer sensor 
     *
     * @note For use with Analog Sensor channels only
     * @note Bridge Excitation Voltage must be selected as reference
     */
    ADMW1001_ADC_SENSOR_BRIDGE_6WIRE_1 = 200,
    /*!< Standard 6-wire Bridge Transducer sensor 
     *
     * @note For use with Analog Sensor channels only
     * @note Bridge Excitation Voltage must be selected as reference
     */
    ADMW1001_ADC_SENSOR_BRIDGE_6WIRE_2 = 201,
    /*!< Standard 6-wire Bridge Transducer sensor 
     *
     * @note For use with Analog Sensor channels only
     * @note Bridge Excitation Voltage must be selected as reference
     */
    ADMW1001_ADC_SENSOR_BRIDGE_6WIRE_3 = 202,
    /*!< Standard 6-wire Bridge Transducer sensor 
     *
     * @note For use with Analog Sensor channels only
     * @note Bridge Excitation Voltage must be selected as reference
     */
    ADMW1001_ADC_SENSOR_BRIDGE_6WIRE_4 = 203,
    /*!< Standard 6-wire Bridge Transducer sensor 
     *
     * @note For use with Analog Sensor channels only
     * @note Bridge Excitation Voltage must be selected as reference
     */
    ADMW1001_ADC_SENSOR_DIODE_2C_TYPEA = 224,
    /*!< Standard Diode two current temperature sensor 
     *
     * @note For use with  Analog Sensor channels only
     */
    ADMW1001_ADC_SENSOR_DIODE_3C_TYPEA = 225,
    /*!< Standard Diode three current temperature sensor
     *
     * @note For use with  Analog Sensor channels only
     */
    ADMW1001_ADC_SENSOR_DIODE_2C_1 = 232,
    /*!< Standard Diode two current sensor
     *
     * @note For use with  Analog Sensor channels only
     */
    ADMW1001_ADC_SENSOR_DIODE_3C_1 = 233,
    /*!< Standard Diode three current sensor
     *
     * @note For use with  Analog Sensor channels only
     */
    ADMW1001_ADC_SENSOR_MICROPHONE_A = 256,
    /*!< Generic microphone sensor without external amplifier
     *
     * @note For use with Analog Sensor channels only
     */
    ADMW1001_ADC_SENSOR_MICROPHONE_B = 257,
    /*!< Generic microphone sensor with external amplifier and bias
     *
     * @note For use with Analog Sensor channels only
     */
    ADMW1001_ADC_SENSOR_MICROPHONE_1 = 264,
    /*!< Generic microphone sensor without external amplifier
     *
     * @note For use with Analog Sensor channels only
     */
    ADMW1001_ADC_SENSOR_MICROPHONE_2 = 265,
    /*!< Generic microphone sensor with external amplifier and bias
     *
     * @note For use with Analog Sensor channels only
     */
    ADMW1001_ADC_SENSOR_VOLTAGE = 512,
    /*!< Generic voltage sensor 
     *
     * @note For use with Analog 0-10V Voltage Sensor channels only
     */
    ADMW1001_ADC_SENSOR_VOLTAGE_PRESSURE_A = 544,
    /*!< Honeywell Pressure voltage sensor (HSCMRNN1.6BAAA3) with default
     *   linearisation and default configuration options
     *
     * @note For use with Analog 0-10V Voltage Sensor channels only
     */
    ADMW1001_ADC_SENSOR_VOLTAGE_PRESSURE_B = 545,
    /*!< Amphenol Pressure voltage sensor (NPA-300B-015A) 
     *
     * @note For use with Analog 0-10V Voltage Sensor channels only
     */
    ADMW1001_ADC_SENSOR_VOLTAGE_PRESSURE_1 = 552,
    /*!< Generic pressure voltage sensor 
     *
     * @note For use with Analog 0-10V Voltage Sensor channels only
     */
    ADMW1001_ADC_SENSOR_VOLTAGE_PRESSURE_2 = 553,
    /*!< Generic pressure voltage sensor 
     *
     * @note For use with Analog 0-10V Voltage Sensor channels only
     */
    ADMW1001_ADC_SENSOR_CURRENT = 768,
    /*!< Generic current sensor with no linearisation applied
     *
     * @note For use with Analog 4-20mA Current Sensor channels only
     */
    ADMW1001_ADC_SENSOR_CURRENT_PRESSURE_A = 800,
    /*!< Honeywell Pressure current sensor (PX2CN2XX100PACH)
     *
     * @note For use with Analog 4-20mA Current Sensor channels only
     */
    ADMW1001_ADC_SENSOR_CURRENT_PRESSURE_1 = 808,
    /*!< Generic pressure current sensor
     *
     * @note For use with Analog 4-20mA Current Sensor channels only
     */
    ADMW1001_ADC_SENSOR_CURRENT_PRESSURE_2 = 809,
    /*!< Generic pressure current sensor
     *
     * @note For use with Analog 4-20mA Current Sensor channels only
     */

}   ADMW1001_ADC_SENSOR_TYPE;

/*! ADMW1001 I2C digital sensor type options
 *
 *  Select the sensor type that is connected to an I2C digital measurement
 *  channel.
 *
 *  @note These are pre-defined sensors using built-in linearisation data
 */
typedef enum
{
    ADMW1001_I2C_SENSOR_HUMIDITY_A = 2112,
    /*!< Honeywell HiH9000-series humidity sensor 
     *
     * @note For use with I2C Digital Sensor channels only
     */
    ADMW1001_I2C_SENSOR_HUMIDITY_B = 2113,
    /*!< Sensirion SHT35-DIS-B humidity sensor 
     *
     * @note For use with I2C Digital Sensor channels only
     */
    ADMW1001_I2C_SENSOR_HUMIDITY_C = 2114,
    /*!< TE MS8607 humidity sensor 
     *   Note: the MS8607 is a combined Pressure & Temperature, Humidity sensor.
     *   The humidity sensor is treated as a virtual device, i.e. standalone.
     *
     * @note For use with I2C Digital Sensor channels only
     */
    ADMW1001_I2C_SENSOR_HUMIDITY_D = 2115,
    /*!< Sensirion SCD30 humidity sensor 
     *   Note: the SCD30 is a combined CO2, Humidity and Temperature sensor.
     *   The humidity sensor is treated as a virtual device, i.e. standalone.
     *
     * @note For use with I2C Digital Sensor channels only
     */
    ADMW1001_I2C_SENSOR_AMBIENTLIGHT_A = 2176,
    /*!< ON-Semiconductor NOA1305 ambient light sensor 
     *
     * @note For use with I2C Digital Sensor channels only
     */
    ADMW1001_I2C_SENSOR_PRESSURE_A = 2208,
    /*!< TE MS8607 pressure sensor 
     *   Note: the MS8607 is a combined Pressure & Temperature, Humidity sensor.
     *   The pressure sensor is treated as a virtual device, i.e. standalone.
     *
     * @note For use with I2C Digital Sensor channels only
     */
    ADMW1001_I2C_SENSOR_TEMPERATURE_A = 2216,
    /*!< TE MS8607 temperature sensor 
     *   Note: the MS8607 is a combined Pressure & Temperature, Humidity sensor.
     *   The temperature sensor is treated as a virtual device, i.e. standalone.
     *
     * @note For use with I2C Digital Sensor channels only
     */
    ADMW1001_I2C_SENSOR_TEMPERATURE_B = 2217,
    /*!< Sensirion SCD30 temperature sensor 
     *   Note: the SCD30 is a combined CO2, Humidity and Temperature sensor.
     *   The temperature sensor is treated as a virtual device, i.e. standalone.
     *
     * @note For use with I2C Digital Sensor channels only
     */
    ADMW1001_I2C_SENSOR_CO2_A = 2224,
    /*!< Sensirion SCD30 C02 sensor 
     *   Note: the SCD30 is a combined CO2, Humidity and Temperature sensor.
     *   The CO2 sensor is treated as a virtual device, i.e. standalone.
     *
     * @note For use with I2C Digital Sensor channels only
     */
     
}   ADMW1001_I2C_SENSOR_TYPE;

/*! ADMW1001 SPI digital sensor type options
 *
 *  Select the sensor type that is connected to an SPI digital measurement
 *  channel.
 *
 *  @note These are pre-defined sensors using built-in linearisation data
 */
typedef enum
{
    ADMW1001_SPI_SENSOR_PRESSURE_A = 3072,
    /*!< Honeywell HSCDRNN1.6BASA3 pressure sensor 
     *
     * @note For use with SPI Digital Sensor channels only
     */
    ADMW1001_SPI_SENSOR_ACCELEROMETER_A = 3200,
    /*!< Analog Devices ADxL362 3-axis accelerometer sensor
     *
     * @note For use with SPI Digital Sensor channels only
     *
     * @note This sensor requires the use of 3 SPI Digital Sensor channels, with
     *       the sensor measurements from the X/Y/Z axes each output on a
     *       seperate dedicated channel (SPI#0/SPI#1/SPI#2, respectively)
     */
    ADMW1001_SPI_SENSOR_ACCELEROMETER_B = 3201,
    /*!< Analog Devices ADxL355 3-axis accelerometer sensor
     *
     * @note For use with SPI Digital Sensor channels only
     *
     * @note This sensor requires the use of 3 SPI Digital Sensor channels, with
     *       the sensor measurements from the X/Y/Z axes each output on a
     *       seperate dedicated channel (SPI#0/SPI#1/SPI#2, respectively)
     */

}   ADMW1001_SPI_SENSOR_TYPE;

#ifdef __cplusplus
}
#endif

/*!
 * @}
 */

#endif /* __ADMW1001_SENSOR_TYPES_H__ */
