/******************************************************************************
Copyright 2017 (c) Analog Devices, Inc.

All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:
  - Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.
  - Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in
    the documentation and/or other materials provided with the
    distribution.
  - Neither the name of Analog Devices, Inc. nor the names of its
    contributors may be used to endorse or promote products derived
    from this software without specific prior written permission.
  - The use of this software may or may not infringe the patent rights
    of one or more patent holders. This license does not release you
    from the requirement that you obtain separate licenses from these
    patent holders to use this software.
  - Use of the software either in source or binary form, must be run
    on or directly connected to an Analog Devices Inc. component.

THIS SOFTWARE IS PROVIDED BY ANALOG DEVICES "AS IS" AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, NON-INFRINGEMENT,
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL ANALOG DEVICES BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, INTELLECTUAL PROPERTY RIGHTS, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *****************************************************************************/

/*!
 ******************************************************************************
 * @file:
 * @brief:  ADMW OS-dependent wrapper layer for log functions
 *-----------------------------------------------------------------------------
 */

#include <mbed.h>
#include "inc/admw_log.h"

static Serial *gpUartDevice;

static ADMW_LOG_LEVEL gLogLevel = ADMW_LOG_LEVEL_DEBUG;

#ifdef __cplusplus
extern "C" {
#endif

/*
 * Open the Log interface and allocate resources
 */
ADMW_RESULT admw_LogOpen(
    ADMW_PLATFORM_LOG_CONFIG *pConfig)
{
    if (pConfig->disableLogs)
        return ADMW_SUCCESS;

    gpUartDevice = new Serial((PinName)pConfig->txPin,
                              (PinName)pConfig->rxPin,
                              pConfig->baudRate);
    if (!gpUartDevice) {
        ADMW_LOG_ERROR("Failed to allocate memory for Log UART context");
        return ADMW_NO_MEM;
    }

    return ADMW_SUCCESS;
}

/*
 * Close the Log interface and free resources
 */
void admw_LogClose(void)
{
    if (gpUartDevice) {
        delete gpUartDevice;
        gpUartDevice = 0;
    }
}

/*
 * Set the log level for uart communication
 */
void admw_LogLevel(ADMW_LOG_LEVEL maxLevel)
{
    gLogLevel = maxLevel;
}

/*
 * Print a log message to the platform log interface
 */
void admw_Log(ADMW_LOG_LEVEL level, const char* format, ...)
{
    char buffer[256];
    va_list va_args;

    if (!gpUartDevice)
        return;

    if (level > gLogLevel)
        return;

    va_start(va_args, format);
    vsnprintf(buffer, sizeof(buffer), format, va_args);
    va_end(va_args);

    gpUartDevice->printf("%s\r\n", buffer);
}

#ifdef __cplusplus
}
#endif
