/*
Copyright 2019 (c) Analog Devices, Inc.

All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:
  - Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.
  - Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in
    the documentation and/or other materials provided with the
    distribution.
  - Neither the name of Analog Devices, Inc. nor the names of its
    contributors may be used to endorse or promote products derived
    from this software without specific prior written permission.
  - The use of this software may or may not infringe the patent rights
    of one or more patent holders. This license does not release you
    from the requirement that you obtain separate licenses from these
    patent holders to use this software.
  - Use of the software either in source or binary form, must be run
    on or directly connected to an Analog Devices Inc. component.

THIS SOFTWARE IS PROVIDED BY ANALOG DEVICES "AS IS" AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, NON-INFRINGEMENT,
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL ANALOG DEVICES BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, INTELLECTUAL PROPERTY RIGHTS, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/*!
 ******************************************************************************
 * @file:   admw_api.h
 * @brief:  ADMW Host Library Application Programming Interface (API)
 *-----------------------------------------------------------------------------
 */

#ifndef _ADMW_API_H__
#define _ADMW_API_H__

#include "inc/admw_types.h"
#include "inc/admw_config_types.h"
#include "inc/admw_platform.h"
#include "inc/admw_gpio.h"
#include "inc/admw_spi.h"
#include "inc/admw_log.h"
#include "inc/admw_time.h"

/*! @defgroup ADMW_Api ADMW Host Library API
 *  Host library API common to the ADMW product family.
 *  @{
 */

#ifdef __cplusplus
extern "C" {
#endif

/*! The maximum number of channels supported by this API
 * @note Specific ADMW products may implement a lower number of channels */
#define ADMW_MAX_CHANNELS 16

/*! A handle used in all API functions to identify the ADMW device. */

typedef void* ADMW_DEVICE_HANDLE;

#define ADMW_FORMATTED_VERSION_SIZE     11u
#define ADMW_VERSION_REG_VAL_SIZE       4u
#define ADMW1001_FIRMWARE_VERSION_DEFAULT "00.00.0000"
#define SPI_BUFFER_SIZE                        100u

#define SFL_READ_STATUS_HDR_SIZE               14u

#define SFL_READ_STATUS_RESPONSE_SIZE          42u

/*! Supported connection types for communication with the ADMW device. */
typedef enum {
    ADMW_CONNECTION_TYPE_SPI = 1,
    /*!< Serial Peripheral Interface (SPI) connection type */
} ADMW_CONNECTION_TYPE;

/*! Connection details for communication with a ADMW device instance. */
typedef struct {
    ADMW_CONNECTION_TYPE      type;
    /*!< Connection type selection */
    ADMW_PLATFORM_SPI_CONFIG  spi;
    /*!< SPI connection parameters, required if SPI connection type is used */
    ADMW_PLATFORM_GPIO_CONFIG gpio;
    /*!< GPIO connection parameters, for device reset and status I/O signals */
    ADMW_PLATFORM_LOG_CONFIG  log;
    /*!< Log interface connection parameters, for display/routing of log messages */
} ADMW_CONNECTION;

/*! Bit masks (flags) for the different device status indicators. */
typedef enum {
    ADMW_DEVICE_STATUS_BUSY         = (1 << 0),
    /*!< Indicates that a command is currently running on the device */
    ADMW_DEVICE_STATUS_DATAREADY    = (1 << 1),
    /*!< Indicates the availability of measurement data for retrieval */
    ADMW_DEVICE_STATUS_ERROR        = (1 << 2),
    /*!< Indicates that an error condition has been detected by the device */
    ADMW_DEVICE_STATUS_ALERT        = (1 << 3),
    /*!< Indicates that an alert condition has been detected by the device */
    ADMW_DEVICE_STATUS_FIFO_ERROR   = (1 << 4),
    /*!< Indicates that a FIFO error condition has been detected by the device */
    ADMW_DEVICE_STATUS_CONFIG_ERROR = (1 << 5),
    /*!< Indicates that a configuration error condition has been detected by the device */
    ADMW_DEVICE_STATUS_LUT_ERROR    = (1 << 6),
    /*!< Indicates that a look-up table error condition has been detected by the device */
} ADMW_DEVICE_STATUS_FLAGS;

/*! Bit masks (flags) for the different diagnostics status indicators. */
typedef enum {
    ADMW_DIAGNOSTICS_STATUS_CHECKSUM_ERROR       = (1 << 0),
    /*!< Indicates Error on Internal Checksum Calculations */
    ADMW_DIAGNOSTICS_STATUS_CONVERSION_ERROR     = (1 << 8),
    /*!< Indicates Error During Internal ADC Conversions */
    ADMW_DIAGNOSTICS_STATUS_CALIBRATION_ERROR    = (1 << 9),
    /*!< Indicates Error During Internal Device Calibrations */
} ADMW_DIAGNOSTICS_STATUS_FLAGS;

/*! Bit masks (flags) for the different channel alert indicators. */
typedef enum {
    ADMW_CHANNEL_ALERT_TIMEOUT          = (1 << 0),
    /*!< Indicates timeout condition detected on the channel */
    ADMW_CHANNEL_ALERT_UNDER_RANGE      = (1 << 1),
    /*!< Indicates raw sample under valid input range, possibly clamped */
    ADMW_CHANNEL_ALERT_OVER_RANGE       = (1 << 2),
    /*!< Indicates raw sample over valid input range, possibly clamped */
    ADMW_CHANNEL_ALERT_LOW_LIMIT        = (1 << 3),
    /*!< Indicates measurement result was below configured minimum threshold */
    ADMW_CHANNEL_ALERT_HIGH_LIMIT       = (1 << 4),
    /*!< Indicates measurement result was above configured maximum threshold */
    ADMW_CHANNEL_ALERT_SENSOR_OPEN      = (1 << 5),
    /*!< Indicates open circuit or mis-wire condition detected on the channel */
    ADMW_CHANNEL_ALERT_REF_DETECT       = (1 << 6),
    /*!< Indicates reference-detect error condition detected on the channel */
    ADMW_CHANNEL_ALERT_CONFIG_ERR       = (1 << 7),
    /*!< Indicates configuration error condition detected on the channel */
    ADMW_CHANNEL_ALERT_LUT_ERR          = (1 << 8),
    /*!< Indicates look-up table error condition detected on the channel */
    ADMW_CHANNEL_ALERT_SENSOR_NOT_READY = (1 << 9),
    /*!< Indicates digital sensor not-ready error condition detected on the channel */
    ADMW_CHANNEL_ALERT_COMP_NOT_READY   = (1 << 10),
    /*!< Indicates compensation channel not-ready error condition detected on the channel */
    ADMW_CHANNEL_ALERT_LUT_UNDER_RANGE  = (1 << 13),
    /*!< Indicates raw sample was under the available LUT/equation range */
    ADMW_CHANNEL_ALERT_LUT_OVER_RANGE   = (1 << 14),
    /*!< Indicates raw sample was over the available LUT/equation range */
} ADMW_CHANNEL_ALERT_FLAGS;

/*! Status details retreived from the ADMW device. */
typedef struct {
    ADMW_DEVICE_STATUS_FLAGS      deviceStatus;
    /*!< General summary status information from the device  */
    ADMW_DIAGNOSTICS_STATUS_FLAGS diagnosticsStatus;
    /*!< Diagnostic error status information from the device  */
    ADMW_CHANNEL_ALERT_FLAGS      channelAlerts[ADMW_MAX_CHANNELS];
    /*!< Per-channel alert status information from the device  */
    uint32_t                           errorCode;
    /*!< Code identifying the last error signalled by the device  */
    uint32_t                           alertCode;
    /*!< Code identifying the last alert signalled by the device  */
    uint32_t                           debugCode;
    /*!< Supplementary code related to the last error/alert  */
    uint32_t                           channelAlertCodes[ADMW_MAX_CHANNELS];
    /*!< Per-channel code identifying the last alert signalled for each channel */
} ADMW_STATUS;

/*! Data sample details retreived from the ADMW device. */
typedef struct {
    ADMW_DEVICE_STATUS_FLAGS      status;
    /*!< Device summary status snapshot when the sample was recorded */
    uint32_t                           channelId;
    /*!< The measurement channel from which this sample was obtained */
    uint32_t                           rawValue;
    /*!< The raw (unprocessed) value obtained directly from the measurement
     *   channel, if available
     */
    float32_t                          processedValue;
    /*!< The processed value obtained from the measurement channel, as a final
     *   measurement value, following calibration and linearisation correction,
     *   and conversion into an appropriate unit of measurement.
     */
} ADMW_DATA_SAMPLE;

/*! Measurement mode options for the ADMW device.
 *  @ref admw_StartMeasurement
 */
typedef enum {
    ADMW_MEASUREMENT_MODE_NORMAL = 0,
    /*!< In this mode, normal measurement cycle(s) are executed and data samples
     *   are returned with raw measurement values included. */
    ADMW_MEASUREMENT_MODE_OMIT_RAW,
    /*!< In this mode, normal measurement cycle(s) are executed and data samples
     *   are returned with raw measurement values omitted for efficiency. */
} ADMW_MEASUREMENT_MODE;

/*! Identifiers for the user configuration slots in persistent memory. */
typedef enum {
    ADMW_FLASH_CONFIG_1,
} ADMW_USER_CONFIG_SLOT;


/******************************************************************************
 * ADMW High-Level API function prototypes
 *****************************************************************************/

/*!
 * @brief Open ADMW device handle and set up communication interface.
 *
 * @param[in]  nDeviceIndex    Zero-based index number identifying this device
 *                             instance.  Note that this will be used to
 *                             retrieve a specific device configuration for
 *                             this device (see @ref admw_SetConfig
 *                             and @ref ADMW_CONFIG)
 * @param[in]  pConnectionInfo Host-specific connection details (e.g. SPI, GPIO)
 * @param[out] phDevice        Pointer to return an ADMW device handle
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 *         - #ADMW_NO_MEM  Failed to allocate memory resources.
 *         - #ADMW_INVALID_DEVICE_NUM Invalid device index specified
 *
 * @details Configure and initialise the Log interface and the SPI/GPIO
 *          communication interface to the ADMW1001 module.
 */
ADMW_RESULT admw_Open(
    unsigned                   const nDeviceIndex,
    ADMW_CONNECTION     * const pConnectionInfo,
    ADMW_DEVICE_HANDLE  * const phDevice);

/*!
 * @brief Close ADMW device context and free resources.
 *
 * @param[in] hDevice ADMW device context handle
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 */
ADMW_RESULT admw_Close(
    ADMW_DEVICE_HANDLE    const hDevice);

/*!
 * @brief Get the current state of the specified GPIO input signal.
 *
 * @param[in]  hDevice    ADMW device context handle
 * @param[in]  ePinId     GPIO pin to query
 * @param[out] pbAsserted Pointer to return the state of the status signal GPIO pin
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 *         - #ADMW_INVALID_DEVICE_NUM Invalid GPIO pin specified.
 *
 * @details Sets *pbAsserted to true if the status signal is asserted, or false
 *          otherwise.
 */
ADMW_RESULT admw_GetGpioState(
    ADMW_DEVICE_HANDLE   const hDevice,
    ADMW_GPIO_PIN        const ePinId,
    bool                  * const pbAsserted);

/*!
 * @brief Register an application-defined callback function for GPIO interrupts
 *
 * @param[in] hDevice          ADMW context handle (@ref admw_Open)
 * @param[in] ePinId           GPIO pin on which to enable/disable interrupts
 * @param[in] callbackFunction Function to be called when an interrupt occurs.
 *                             Specify NULL here to disable interrupts.
 * @param[in] pCallbackParam   Optional opaque parameter passed to the callback
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 *         - #ADMW_INVALID_DEVICE_NUM Invalid GPIO pin specified.
 */
ADMW_RESULT admw_RegisterGpioCallback(
    ADMW_DEVICE_HANDLE          const hDevice,
    ADMW_GPIO_PIN               const ePinId,
    ADMW_GPIO_CALLBACK          const callbackFunction,
    void                           * const pCallbackParam);

/*!
 * @brief Trigger a shut down of the device.
 *
 * @param[in] hDevice ADMW device context handle
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 *
 * @details Instructs the ADMW device to initiate a shut down,
 *          typically used to conserve power when the device is not
 *          in use.  The device may be restarted by calling
 *          @ref admw_Reset().  Note that active configuration
 *          settings are not preserved during shutdown and must be
 *          reloaded after the device has become ready again.
 *
 * @note No other command must be running when this is called.
 */
ADMW_RESULT admw_Shutdown(
    ADMW_DEVICE_HANDLE    const hDevice);

/*!
 * @brief Reset the ADMW device.
 *
 * @param[in] hDevice ADMW device context handle
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 *
 * @details Trigger a hardware-reset of the ADMW device.
 *
 * @note The device may require several seconds before it is ready for use
 *       again. @ref admw_GetDeviceReadyState may be used to check if
 *       the device is ready.
 */
ADMW_RESULT admw_Reset(
    ADMW_DEVICE_HANDLE    const hDevice);

/*!
 * @brief Check if the device is ready, following power-up or a reset.
 *
 * @param[in]  hDevice ADMW device context handle
 * @param[out] pbReady Pointer to return true if the device is ready, or false
 *                     otherwise
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 *
 * @details This function attempts to read a fixed-value device register via
 *           the communication interface.
 */
ADMW_RESULT admw_GetDeviceReadyState(
    ADMW_DEVICE_HANDLE    const hDevice,
    bool                   * const pbReady);

/*!
 * @brief Obtain the product ID from the device.
 *
 * @param[in]  hDevice    ADMW device context handle
 * @param[out] pProductId Pointer to return the product ID value
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 *
 * @details Reads the product ID registers on the device and returns the value.
 */
ADMW_RESULT admw_GetProductID(
    ADMW_DEVICE_HANDLE    const hDevice,
    ADMW_PRODUCT_ID     * const pProductId);

/*!
 * @brief Write full configuration settings to the device registers.
 *
 * @param[in]  hDevice ADMW device context handle
 * @param[out] pConfig Pointer to the configuration data structure
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 *
 * @details Translates configuration details provided into device-specific
 *          register settings and updates device configuration registers.
 *
 * @note Settings are not applied until admw_ApplyConfigUpdates() is called
 */
ADMW_RESULT admw_SetConfig(
    ADMW_DEVICE_HANDLE   const hDevice,
    ADMW_CONFIG        * const pConfig);

/*!
 * @brief Apply the configuration settings currently stored in device registers
 *
 * @param[in] hDevice ADMW device context handle
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 *
 * @details Instructs the ADMW device to reload and apply configuration
 *          from the device configuration registers.  Changes to configuration
 *          registers are ignored by the device until this function is called.
 *
 * @note No other command must be running when this is called.
 */
ADMW_RESULT admw_ApplyConfigUpdates(
    ADMW_DEVICE_HANDLE    const hDevice);

/*!
 * @brief Store the configuration settings to persistent memory on the device.
 *
 * @param[in] hDevice ADMW device context handle
 * @param[in] eSlotId User configuration slot in persistent memory
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 *
 * @details Instructs the ADMW device to save the current contents of its
 *          device configuration registers to non-volatile memory.
 *
 * @note No other command must be running when this is called.
 * @note Do not power down the device while this command is running.
 */
ADMW_RESULT admw_SaveConfig(
    ADMW_DEVICE_HANDLE    const hDevice,
    ADMW_USER_CONFIG_SLOT const eSlotId);

/*!
 * @brief Restore configuration settings from persistent memory on the device.
 *
 * @param[in] hDevice ADMW device context handle
 * @param[in] eSlotId User configuration slot in persistent memory
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 *
 * @details Instructs the ADMW device to restore the contents of its
 *          device configuration registers from non-volatile memory.
 *
 * @note No other command must be running when this is called.
 */
ADMW_RESULT admw_RestoreConfig(
    ADMW_DEVICE_HANDLE    const hDevice,
    ADMW_USER_CONFIG_SLOT const eSlotId);

/*!
 * @brief Erases the external flash memory.
 *
 * @param[in] hDevice ADMW device context handle
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 *
 * @details Sends the bulk erase instruction to the external flash
 *          device. All stored samples are deleted.
 *          It is a blocking operation and takes tens of seconds to
 *          complete.
 *
 * @note No other command must be running when this is called.
 */
ADMW_RESULT admw_EraseExternalFlash(
    ADMW_DEVICE_HANDLE    const hDevice);

/*!
 * @brief Gets the number of samples stored in the external flash
 *        memory.
 *
 * @param[in] hDevice ADMW device context handle
 * @param[in] pSampleCount Address of the return value.
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 *
 * @note No other command must be running when this is called.
 */
ADMW_RESULT admw_GetExternalFlashSampleCount(
    ADMW_DEVICE_HANDLE    const hDevice,
    uint32_t * nSampleCount);

// DEBUG - TO BE DELETED
ADMW_RESULT admw_SetExternalFlashIndex(
    ADMW_DEVICE_HANDLE    const hDevice,
    uint32_t nStartIndex);

/*!
 * @brief Read measurement samples stored in the the external flash memory.
 *
 * @param[in]  hDevice          ADMW device context handle
 * @param[out] pSamples         Pointer to return a set of requested data
 *                              samples.
 * @param[in]  nStartIndex      Index of first sample to retrieve.
 * @param[in]  nBytesPerSample  The size, in bytes, of each sample.
 * @param[in]  nRequested       Number of requested data samples.
 * @param[out] pnReturned       Number of valid data samples successfully
 *                              retrieved.
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 *
 * @details Reads the status registers and extracts the relevant information
 *          to return to the caller.
 *
 */
ADMW_RESULT admw_GetExternalFlashData(
    ADMW_DEVICE_HANDLE    const hDevice,
    ADMW_DATA_SAMPLE    * const pSamples,
    uint32_t                   const nIndex,
    uint32_t                   const nRequested,
    uint32_t                 * const pnReturned);

/*!
 * @brief Store the LUT data to persistent memory on the device.
 *
 * @param[in] hDevice ADMW device context handle
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 *
 * @details Instructs the ADMW device to save the current contents of its
 *          LUT data buffer, set using @ref admw_SetLutData, to
 *          non-volatile memory.
 *
 * @note No other command must be running when this is called.
 * @note Do not power down the device while this command is running.
 */
ADMW_RESULT admw_SaveLutData(
    ADMW_DEVICE_HANDLE    const hDevice);

/*!
 * @brief Restore LUT data from persistent memory on the device.
 *
 * @param[in] hDevice ADMW device context handle
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 *
 * @details Instructs the ADMW device to restore the contents of its
 *          LUT data, previously stored with @ref admw_SaveLutData, from
 *          non-volatile memory.
 *
 * @note No other command must be running when this is called.
 */
ADMW_RESULT admw_RestoreLutData(
    ADMW_DEVICE_HANDLE    const hDevice);

/*!
 * @brief Start the measurement cycles on the device.
 *
 * @param[in] hDevice          ADMW device context handle
 * @param[in] eMeasurementMode Allows a choice of special modes for the
 *                             measurement.  See @ref ADMW_MEASUREMENT_MODE
 *                             for further information.
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 *
 * @details Instructs the ADMW device to start executing measurement cycles
 *          according to the current applied configuration settings.  The
 *          DATAREADY status signal will be asserted whenever new measurement
 *          data is published, according to selected settings.
 *          Measurement cycles may be stopped by calling @ref
 *          admw_StopMeasurement.
 *
 * @note No other command must be running when this is called.
 */
ADMW_RESULT admw_StartMeasurement(
    ADMW_DEVICE_HANDLE    const hDevice,
    ADMW_MEASUREMENT_MODE const eMeasurementMode);

/*!
 * @brief Stop the measurement cycles on the device.
 *
 * @param[in] hDevice ADMW device context handle
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 *
 * @details Instructs the ADMW device to stop executing measurement cycles.
 *          The command may be delayed until the current conversion, if any, has
 *          been completed and published.
 *
 * @note To be used only if a measurement command is currently running.
 */
ADMW_RESULT admw_StopMeasurement(
    ADMW_DEVICE_HANDLE    const hDevice);

/*!
 * @brief Run built-in diagnostic checks on the device.
 *
 * @param[in] hDevice ADMW device context handle
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 *
 * @details Instructs the ADMW device to execute its built-in diagnostic
 *          tests, on any enabled measurement channels, according to the current
 *          applied configuration settings.  Device status registers will be
 *          updated to indicate if any errors were detected by the diagnostics.
 *
 * @note No other command must be running when this is called.
 */
ADMW_RESULT admw_RunDiagnostics(
    ADMW_DEVICE_HANDLE    const hDevice);

/*!
 * @brief Run built-in calibration on the device.
 *
 * @param[in] hDevice ADMW device context handle
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 *
 * @details Instructs the ADMW device to execute its self-calibration
 *          routines, on any enabled measurement channels, according to the
 *          current applied configuration settings.  Device status registers
 *          will be updated to indicate if any errors were detected.
 *
 * @note No other command must be running when this is called.
 */
ADMW_RESULT admw_RunCalibration(
    ADMW_DEVICE_HANDLE    const hDevice);

/*!
 * @brief Run built-in digital calibration on the device.
 *
 * @param[in] hDevice ADMW device context handle
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 *
 * @details Instructs the ADMW device to execute its calibration
 *          routines, on any enabled digital channels, according to the
 *          current applied configuration settings.  Device status registers
 *          will be updated to indicate if any errors were detected.
 *
 * @note No other command must be running when this is called.
 */
ADMW_RESULT admw_RunDigitalCalibration(
    ADMW_DEVICE_HANDLE    const hDevice);

/*!
 * @brief Read the current status from the device registers.
 *
 * @param[in]  hDevice ADMW device context handle
 * @param[out] pStatus Pointer to return the status summary obtained from the
 *                     device.
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 *
 * @details Reads the status registers and extracts the relevant information
 *          to return to the caller.
 *
 * @note This may be called at any time, assuming the device is ready.
 */
ADMW_RESULT admw_GetStatus(
    ADMW_DEVICE_HANDLE    const hDevice,
    ADMW_STATUS         * const pStatus);

/*!
 * @brief Read measurement data samples from the device registers.
 *
 * @param[in]  hDevice          ADMW device context handle
 * @param[in]  eMeasurementMode Must be set to the same value used for @ref
 *                              admw_StartMeasurement().
 * @param[out] pSamples         Pointer to return a set of requested data samples.
 * @param[in]  nBytesPerSample  The size, in bytes, of each sample.
 * @param[in]  nRequested       Number of requested data samples.
 * @param[out] pnReturned       Number of valid data samples successfully retrieved.
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 *
 * @details Reads the status registers and extracts the relevant information
 *          to return to the caller.
 *
 * @note This is intended to be called only when the DATAREADY status signal
 *       is asserted.
 */
ADMW_RESULT admw_GetData(
    ADMW_DEVICE_HANDLE    const hDevice,
    ADMW_MEASUREMENT_MODE const eMeasurementMode,
    ADMW_DATA_SAMPLE    * const pSamples,
    uint8_t                    const nBytesPerSample,
    uint32_t                   const nRequested,
    uint32_t                 * const pnReturned);

/*!
 * @brief Check if a command is currently running on the device.
 *
 * @param[in]  hDevice          ADMW device context handle
 * @param[out] pbCommandRunning Pointer to return the command running status
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 *
 * @details Reads the device status register to check if a command is running.
 */
ADMW_RESULT admw_GetCommandRunningState(
    ADMW_DEVICE_HANDLE hDevice,
    bool *pbCommandRunning);

ADMW_RESULT deviceInformation(ADMW_DEVICE_HANDLE hDevice);
#ifdef __cplusplus
}
#endif

/*!
 * @}
 */

#endif /* _ADMW_API_H__ */
