/** MARMEX_OB OLED screen drawing library with Japanese font
 *
 *  @class   MARMEX_OB_oled_J
 *  @author  dia
 *  @version 0.1
 *  @date    01-Feb-2013
 *
 *  Released under the MIT License: http://mbed.org/license/mit
 *
 *  MARMEX_OB_oled OLED screen drawing library with Japanese font for mbed
 *
 * special thanks
 *  ref http://mbed.org/users/nxpfan/notebook/MARMEX_OB_oled_lib/
 *  ref http://mbed.org/users/s_s_s/notebook/mary-ob%E7%94%A8%E3%81%AE%E3%83%A9%E3%82%A4%E3%83%96%E3%83%A9%E3%83%AA%E3%83%BC/
 *  ref http://mbed.org/users/kanpapa/notebook/mary-oled-board/
 *  ref http://www.geocities.jp/littlimi/misaki.htm
 */

#ifndef     MBED_MARMEX_OB_OLED_J
#define     MBED_MARMEX_OB_OLED_J

#define     countof(x) ( sizeof(x) / sizeof(x[0]) ) //for Japanese

#include    "mbed.h"
#include    "MARMEX_OB_oled.h"
#include    "small_font.h"

/** MARMEX_OB_oled OLED screen drawing with Japanese class
 *
 *  This is a driver code for MARMEX_OB_oled board OLED screen with Japanese.
 *  This class inherits MARMEX_OB_oled.h of mbed.org.
 *  To use this class, import MARMEX_OB_oled.h from here..
 *   http://mbed.org/users/nxpfan/notebook/MARMEX_OB_oled_lib/
 *
 *  Example:
 *  @code
 * #include "mbed.h"
 * #include "MARMEX_OB_oled_J.h"
 * 
 * MARMEX_OB_oled_J oled( p5, p7,  p27, p24, p26 ); // mosi, sclk, cs, rst, power_control
 * 
 * LocalFileSystem local("local");
 * 
 * void lfTrim(char* text) {
 *     char *p;
 *     p = strchr( text, '\n' );
 *     if ( p != NULL )
 *     {
 *         *p = '\0';
 *     }
 * }
 * 
 * int main() {
 *     FILE *fp;
 *     char *filename = "/local/load.txt";
 *     if((fp = fopen(filename, "r")) == NULL){
 *         printf("Error: %s could not open.", filename);
 *         return 1;
 *     }
 *     
 *     char text[20];
 * 
 *     oled.cls();
 *     oled.locate(0,0); oled.fontsize(1,1);oled.foreground(0x00FFFFFF); oled.printf("ABC:123");
 *     oled.locate(0,1); oled.fontsize(2,1);oled.foreground(0x00FF0000); oled.printf("ABC:123");
 *     if(fgets(text, 20, fp ) != NULL) {
 *         lfTrim(text);
 *         oled.locate(0,1); oled.fontsize(1,2);oled.foreground(0x0000FF00); oled.printf("%s", text);
 *     }
 *     if(fgets(text, 20, fp ) != NULL) {
 *         lfTrim(text);
 *         oled.locate(0,4); oled.fontsize(2,1);oled.foreground(0x000000FF); oled.printf("%s", text);
 *     }
 *     if(fgets(text, 20, fp ) != NULL) {
 *         lfTrim(text);
 *         oled.locate(0,3); oled.fontsize(2,2);oled.foreground(0x0000FFFF); oled.printf("%s", text);
 *     }
 *     if(fgets(text, 20, fp ) != NULL) {
 *         lfTrim(text);
 *         oled.locate(0,3); oled.fontsize(3,3);oled.foreground(0x00FF00FF); oled.printf("%s", text);
 *     }
 *     fclose(fp);
 * }
 */

class MARMEX_OB_oled_J : public MARMEX_OB_oled {

public:


    /** Create a MARMEX_OB_oled_J object connected to specified SPI and DigitalOut pins
     *
     *  @param mosi SPI-MOSI pin (for MAPLE board, use p5)
     *  @param sclk SPI-SCLK pin (for MAPLE board, use p8)
     *  @param cs   chip select signal (for MAPLE board, use p8(slot1), p26(slot2))
     *  @param rst  reset signal (for MAPLE board, use p30(slot1), p21(slot2))
     *  @param power_pin backlight power control signal (for MAPLE board, use p11(slot1), p17(slot2))
     *
     *  Example of MARMEX_OB_oled on MAPLE board:
     *  @code
     *  #include "mbed.h"
     *  #include "MARMEX_OB_oled.h"
     *
     *  MARMEX_OB_oled_J   oled( p5, p7,  p8, p30, p11 ); // mosi, sclk, cs, rst, power_control
     *  MARMEX_OB_oled_J   oled( p5, p7, p26, p21, p17 ); // mosi, sclk, cs, rst, power_control
     *  ...
     *  ..
     *  @endcode
     */

    MARMEX_OB_oled_J( PinName mosi, PinName sclk, PinName cs, PinName rst, PinName power_pin ) : MARMEX_OB_oled( mosi, sclk, cs, rst, power_pin ) {
        kstate = false;    // for Japanese
        _font_width_size = 1;     //1:8x8 2:16x16 ... for Japanese
        _font_height_size = 1;    //1:8x8 2:16x16 ... for Japanese
        power( ON );
        reset();
    }

#if DOXYGEN_ONLY
    /** Write a character to the LCD
     *
     * @param c The character to write to the display
     */
    int putc( int c );

    /** Write a formated string to the LCD
     *
     * @param format A printf-style format string, followed by the
     *               variables to use in formating the string.
     *
     *  !!! 16th character in the string will be disappeared
     *  !!! This problem is due to difference of screen size NokiaLCD library and its internal mechanism...
     */
    int printf( const char* format, ... );

    /** Set the foreground colour
     *
     * @param c 24-bit colour
     */
    void foreground(int c);

    /** Set the background colour
     *
     * @param c 24-bit colour
     */
    void background(int c);

    /** Locate to a screen column and row
     *
     * @param column  The horizontal position from the left, indexed from 0
     * @param row     The vertical position from the top, indexed from 0
     */
    void locate(int column, int row);

    /** reset MARMEX_OB_oled
     *
     *  Executes hardware reset and initialize.
     *  See MARMEX_OB_oled manual for the initialization sequence and values
     *  For gamma correction table, using math function to make the code simple
     */
    void reset( void );

    /** Clear the screen and locate to 0,0 */
    void cls( void );

    /** Set a pixel on te screen
     *
     * @param x horizontal position from left
     * @param y vertical position from top
     * @param colour 24-bit colour in format 0x00RRGGBB
     */
    virtual void pixel( int x, int y, int colour );

    /** Fill an area of the screen
     *
     * @param x horizontal position from left
     * @param y vertical position from top
     * @param width width in pixels
     * @param height height in pixels
     * @param colour 24-bit colour in format 0x00RRGGBB
     */
    void fill( int x, int y, int width, int height, int colour );

    void blit( int x, int y, int width, int height, const int* colour );


    /** Screen width
     *
     *  @return screen width [pixel]
     */
    int width();

    /** Screen height
     *
     *  @return screen height [pixel]
     */
    int height();

    /** Columns
     *
     *  @return screen columns
     */
    int columns();

    /** Rows
     *
     *  @return screen rows
     */
    int rows();

    /** Power switch for OLED backlight
     *
     * @param sw argument can be MARMEX_OB_oled::ON or MARMEX_OB_oled::OFF
     */
    void power( unsigned char sw );

#endif

   /** font size
    * void Moled::fontsize(int yoko,int tate){
    * 
    * @param _font_width   The vertical multiply factor
    * @param _font_height   The horizotal multiply factor
    */
    void fontsize(int fontsize_width,int fontsize_height){
        _font_width_size=fontsize_width;
        _font_height_size=fontsize_height;
    }
    
    unsigned int findface(unsigned short c) {
        unsigned int p = 0;
        int i, sum;
        for (sum = i = 0; i < countof(font8table); i++) {
            if (font8table[i].start <= c && c <= font8table[i].end) {
                p = (sum + c - font8table[i].start);
                break;
            }
            sum += font8table[i].end - font8table[i].start + 1;
        }
        return p;
    }
    
    virtual int _putc(int value) {
        int x = _column * 8 ;  // FIXME: Char sizes
        int y = _row * 8;
    
        if(value == '\n') {
            newline();
        } else if (kstate) { // 2nd byte of shift-jis
            kstate = false;
            int p = findface(kbuf << 8 | value);
            bitblit(x + 1, y + 1, 8,8, (char*)&(FontLookup[p][0]));
    
               //printf("%x %x",( kbuf << 8 | value),p); //for debug
            _column++;
            if (_column >= COLS) {
                _row++;
                _column = 0;
            }
            if (_row >= ROWS) {
                _row = 0;
            }
        } else if ((0x81 <= value && value <= 0x9f) || (0xe0 <= value && value <= 0xfc)) { // 1st byte of shift-jis
            kstate = true;
            kbuf = value;
        } else {
            bitblit(x + 1, y + 1, 8, 8, (char*)&(FontLookup_ABC[value-32][0]));
            _column++;
            if (_column >= COLS) {
                _row++;
                _column = 0;
            }
            if (_row >= ROWS) {
                _row = 0;
            }
        }
    
        return value;
    }

    void bitblit( int x, int y, int width, int height, const char* bitstream ) {
        _cs = 0;
        int ii=0;
        _window(x*_font_width_size, y*_font_height_size, width*_font_width_size, height*_font_height_size);  //add *2
        for (int i=0; i<height*_font_height_size; i++) {
            ii=i/_font_height_size;
            for (int j=0;j<width;j++){
                int byte = (ii*width+j) / 8;
                int bit =  (ii*width+j) % 8;
                int colour = ((bitstream[byte] << bit) & 0x80) ? _foreground : _background;
                for (int jj=0;jj<_font_width_size;jj++){
                    _putp(colour);
                } 
            }
        }
    }


private:
    bool kstate;                           //for Japnaese
    unsigned int kbuf;                     //for Japanese
    int _font_width_size, _font_height_size; //for Japanese
}
;

#endif  //  MBED_MARMEX_OB_OLED_J
