/*
 * main.c - sample application to switch to AP mode and ping client
 *
 * Copyright (C) 2014 Texas Instruments Incorporated - http://www.ti.com/
 *
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/*
 * Application Name     -   Getting started with Wi-Fi Access-Point mode
 * Application Overview -   This sample application demonstrates how
 *                          to configure CC3100 in Access-Point mode. Any
 *                          WLAN station in its range can connect/communicate
 *                          to/with it as per the standard networking protocols.
 *                          On a successful connection, the device ping's the
 *                          connected station.
 * Application Details  -   http://processors.wiki.ti.com/index.php/CC31xx_Getting_Started_with_WLAN_AP
 *                          doc\examples\getting_started_with_wlan_ap.pdf
 */

#include "cc3100_simplelink.h"
#include "cc3100_sl_common.h"

#include "fPtr_func.h"
#include "cc3100.h"
#include "cc3100_spi.h"
#include "myBoardInit.h"

using namespace mbed_cc3100;

#if (THIS_BOARD == MBED_BOARD_LPC1768)
//cc3100 _cc3100(p9, p10, p8, SPI(p5, p6, p7));//LPC1768  irq, nHib, cs, mosi, miso, sck
cc3100 _cc3100(p9, p10, p8, SPI(p11, p12, p13));//LPC1768  irq, nHib, cs, mosi, miso, sck
Serial pc(USBTX, USBRX);//lpc1768
#elif (THIS_BOARD == ST_MBED_NUCLEOF411)
cc3100 _cc3100(PA_9, PC_7, PB_6, SPI(PA_7, PA_6, PA_5));//nucleoF411  irq, nHib, cs, mosi, miso, sck
Serial pc(SERIAL_TX, SERIAL_RX);//nucleoF411
#elif (THIS_BOARD == ST_MBED_NUCLEOF401)
cc3100 _cc3100(PA_9, PC_7, PB_6, SPI(PA_7, PA_6, PA_5));//nucleoF401  irq, nHib, cs, mosi, miso, sck
Serial pc(SERIAL_TX, SERIAL_RX);//nucleoF401
#elif (THIS_BOARD == EA_MBED_LPC4088)
cc3100 _cc3100(p9, p10, p8, SPI(p5, p6, p7));//LPC4088  irq, nHib, cs, mosi, miso, sck
Serial pc(USBTX, USBRX);//EA_lpc4088
#elif (THIS_BOARD == ST_MBED_NUCLEOF103)
cc3100 _cc3100(PA_9, PC_7, PB_6, SPI(PA_7, PA_6, PA_5));//nucleoF103  irq, nHib, cs, mosi, miso, sck
Serial pc(SERIAL_TX, SERIAL_RX);
#elif (THIS_BOARD == Seeed_Arch_Max)
cc3100 _cc3100(PD_12, PD_13, PD_11, SPI(PB_5, PB_4, PB_3));//Seeed_Arch_Max  irq, nHib, cs, mosi, miso, sck
Serial pc(USBTX, USBRX);
#else

#endif

#define APPLICATION_VERSION "1.2.0"

/*
 * GLOBAL VARIABLES -- Start
 */
int32_t demo = 0;

/*
 * GLOBAL VARIABLES -- End
 */


/*
 * STATIC FUNCTION DEFINITIONS -- Start
 */

static void displayBanner();
/*
 * STATIC FUNCTION DEFINITIONS -- End
 */
 
void station_app(void);
void AP_app(void);

/*
 * Application's entry point
 */
  
 
int main(void) {

    pc.baud(115200);

    int32_t retVal = -1;

    retVal = _cc3100.initializeAppVariables();
    ASSERT_ON_ERROR(retVal);

    displayBanner();

    _cc3100.CLR_STATUS_BIT(g_Status, STATUS_BIT_PING_DONE);
    g_PingPacketsRecv = 0;
    
    /*
     * Following function configures the device to default state by cleaning
     * the persistent settings stored in NVMEM (viz. connection profiles &
     * policies, power policy etc)
     *
     * Applications may choose to skip this step if the developer is sure
     * that the device is in its default state at start of application
     *
     * Note that all profiles and persistent settings that were done on the
     * device will be lost
     */
    retVal = _cc3100.configureSimpleLinkToDefaultState();
    
    if(retVal < 0) {
        if (DEVICE_NOT_IN_STATION_MODE == retVal)
            printf(" Failed to configure the device to its default state \n\r");

        LOOP_FOREVER();
    }

    printf(" Device is configured in it's default state \n\r");
    
    /*
     * Assumption is that the device is configured in station mode already
     * and it is in its default state
     */
    /* Initializing the CC3100 device */
    
    if(!(demo ==1)){
        retVal = _cc3100.sl_Start(0, 0, 0);
    
        if ((retVal < 0) || (ROLE_STA != retVal) ){
        printf(" Failed to start the device \n\r");
        LOOP_FOREVER();
        }

        printf(" Device started as STATION \n\r");
    }
    
    if(demo == 0 ){
       station_app();
    }else{
       AP_app(); 
    }    
    return 0;
}    

/*!
    \brief This function displays the application's banner

    \param      None

    \return     None
*/
static void displayBanner()
{
    if(!demo){
        printf("\n\r\n\r");
        printf(" Getting started with WLAN access-point application - Version ");
        printf("%s",APPLICATION_VERSION);
        printf("\n\r*******************************************************************************\n\r");
    
    }else{
         printf("\n\r\n\r");
         printf(" Getting started with station application - Version ");
         printf(" %s", APPLICATION_VERSION);
         printf("\n\r*******************************************************************************\n\r");
    }
}

void AP_app(void){
    
    SlPingStartCommand_t PingParams = {0};
    SlPingReport_t Report = {0};
//    SlNetCfgIpV4Args_t ipV4 = {0};
//    SlNetAppDhcpServerBasicOpt_t dhcpParams = {0};

    uint8_t SecType = 0;
    int32_t role = ROLE_STA;
    int32_t retVal = -1;
    
    /*
     * Assumption is that the device is configured in station mode already
     * and it is in its default state
     */
    role = _cc3100.sl_Start(0, 0, 0);
    if (ROLE_AP == role) {
        /* If the device is in AP mode, we need to wait for this event before doing anything */
        while(!_cc3100.IS_IP_ACQUIRED(g_Status,STATUS_BIT_IP_ACQUIRED)) {
            _cc3100._nonos._SlNonOsMainLoopTask();
        }
    } else {
        /* Configure CC3100 to start in AP mode */
        retVal = _cc3100._wlan.sl_WlanSetMode(ROLE_AP);
        if(retVal < 0)
            LOOP_FOREVER();
            
        /* Configure the SSID of the CC3100 */
        retVal = _cc3100._wlan.sl_WlanSet(SL_WLAN_CFG_AP_ID, WLAN_AP_OPT_SSID,
                strlen(SSID_AP_MODE), (uint8_t *)SSID_AP_MODE);
        if(retVal < 0)
            LOOP_FOREVER();

        SecType = SEC_TYPE_AP_MODE;
        /* Configure the Security parameter the AP mode */
        retVal = _cc3100._wlan.sl_WlanSet(SL_WLAN_CFG_AP_ID, WLAN_AP_OPT_SECURITY_TYPE, 1, (uint8_t *)&SecType);
        if(retVal < 0)
            LOOP_FOREVER();

        retVal = _cc3100._wlan.sl_WlanSet(SL_WLAN_CFG_AP_ID, WLAN_AP_OPT_PASSWORD, strlen(PASSWORD_AP_MODE), (uint8_t *)PASSWORD_AP_MODE);
        if(retVal < 0)
            LOOP_FOREVER();    

        retVal = _cc3100.sl_Stop(SL_STOP_TIMEOUT);
        if(retVal < 0)
            LOOP_FOREVER();

        _cc3100.CLR_STATUS_BIT(g_Status, STATUS_BIT_IP_ACQUIRED);

        role = _cc3100.sl_Start(0, 0, 0);
        if (ROLE_AP == role) {
            /* If the device is in AP mode, we need to wait for this event before doing anything */
            while(!_cc3100.IS_IP_ACQUIRED(g_Status,STATUS_BIT_IP_ACQUIRED)) {
                _cc3100._nonos._SlNonOsMainLoopTask();
            }
        } else {
            printf(" Device couldn't be configured in AP mode \n\r");
            LOOP_FOREVER();
        }
    }
    printf(" Device started as Access Point\n\r");

    /* Wait */
    printf(" Waiting for clients to connect...!\n\r");
    while((!_cc3100.IS_IP_LEASED(g_Status,STATUS_BIT_IP_LEASED)) || (!_cc3100.IS_STA_CONNECTED(g_Status,STATUS_BIT_STA_CONNECTED))) {
        _cc3100._nonos._SlNonOsMainLoopTask();
    }
    printf(" Client connected to the device \n\r");
    printf(" Pinging...! \n\r");

    /* Set the ping parameters */
    PingParams.PingIntervalTime = PING_INTERVAL;
    PingParams.PingSize = PING_SIZE;
    PingParams.PingRequestTimeout = PING_TIMEOUT;
    PingParams.TotalNumberOfAttempts = PING_ATTEMPTS;
    PingParams.Flags = 0;
    PingParams.Ip = g_StationIP; /* Fill the station IP address connected to CC3100 */

    /* Ping client connected to CC3100 */
    retVal = _cc3100._netapp.sl_NetAppPingStart((SlPingStartCommand_t*)&PingParams, SL_AF_INET, (SlPingReport_t*)&Report, &SimpleLinkPingReport);
    if(retVal < 0)
        LOOP_FOREVER();

    /* Wait */
    while(!_cc3100.IS_PING_DONE(g_Status,STATUS_BIT_PING_DONE)) {
        _cc3100._nonos._SlNonOsMainLoopTask();
    }

    if (0 == g_PingPacketsRecv) {
        printf(" A STATION couldn't connect to the device \n\r");
        //ASSERT_ON_ERROR(LAN_CONNECTION_FAILED);
        printf(" ERROR code %d\n\r", LAN_CONNECTION_FAILED);
    }

    printf(" Device and the station are successfully connected \n\r");
    //return SUCCESS;
}

void station_app(void){
    
    int32_t retVal = -1;
    
    /* Connecting to WLAN AP */
    retVal = _cc3100.establishConnectionWithAP();
    if(retVal < 0)
    {
        printf(" Failed to establish connection w/ an AP \n\r");
        LOOP_FOREVER();
    }    
    printf(" Connection established w/ AP and IP is acquired \n\r");
    
    printf(" Pinging...! \n\r");
    retVal = _cc3100.checkLanConnection();
    if(retVal < 0)
    {
        printf(" Device couldn't connect to LAN \n\r");
        LOOP_FOREVER();
    }
    printf(" Device successfully connected to the LAN\r\n");
    
    retVal = _cc3100.checkInternetConnection();
    if(retVal < 0)
    {
        printf(" Device couldn't connect to the internet \n\r");
        LOOP_FOREVER();
    }

    printf(" Device successfully connected to the internet \n\r");
}              


