/* mbed Microcontroller Library
 * Copyright (c) 2006-2013 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
#include "SemihostedFileSystem.h"
#include "mbed_semihost_api_shim.h"
#include <string.h>
#include <stdio.h>
#include <errno.h>
#include <cstdarg>

#define STRING_STACK_LIMIT    120

#define OPEN_R          0
#define OPEN_B          1
#define OPEN_PLUS       2
#define OPEN_W          4
#define OPEN_A          8
#define OPEN_INVALID   -1

static int posix_to_semihost_open_flags(int flags) {
    /* POSIX flags -> semihosting open mode */
    int openmode;
    if (flags & O_RDWR) {
        /* a plus mode */
        openmode = OPEN_PLUS;
        if (flags & O_APPEND) {
            openmode |= OPEN_A;
        } else if (flags & O_TRUNC) {
            openmode |= OPEN_W;
        } else {
            openmode |= OPEN_R;
        }
    } else if (flags & O_WRONLY) {
        /* write or append */
        if (flags & O_APPEND) {
            openmode = OPEN_A;
        } else {
            openmode = OPEN_W;
        }
    } else if (flags == O_RDONLY) {
        /* read mode */
        openmode = OPEN_R;
    } else {
        /* invalid flags */
        openmode = OPEN_INVALID;
    }

    return openmode;
}

FILEHANDLE semihosted_file_open(const char* name, int flags) {
    int openmode = posix_to_semihost_open_flags(flags);
    if (openmode == OPEN_INVALID) {
        return (FILEHANDLE)NULL;
    }

    FILEHANDLE fh = semihost_open(name, openmode);
    if (fh == -1) {
        return (FILEHANDLE)NULL;
    }

    return fh;
}

SemihostedFileHandle::SemihostedFileHandle(FILEHANDLE fh) : _fh(fh), pos(0) {
    // No lock needed in constructor
}

SemihostedFileHandle::SemihostedFileHandle(const char* name, int flags)
    : _fh(semihosted_file_open(name, flags)), pos(0) {
    // No lock needed in constructor
}

int SemihostedFileHandle::close() {
    int retval = semihost_close(_fh);
    delete this;
    return retval;
}

ssize_t SemihostedFileHandle::write(const void *buffer, size_t length) {
    lock();
    ssize_t n = semihost_write(_fh, (const unsigned char*)buffer, length, 0); // number of characters not written
    n = length - n; // number of characters written
    pos += n;
    unlock();
    return n;
}

ssize_t SemihostedFileHandle::read(void *buffer, size_t length) {
    lock();
    ssize_t n = semihost_read(_fh, (unsigned char*)buffer, length, 0); // number of characters not read
    n = length - n; // number of characters read
    pos += n;
    unlock();
    return n;
}

int SemihostedFileHandle::isatty() {
    lock();
    int ret = semihost_istty(_fh);
    unlock();
    return ret;
}

off_t SemihostedFileHandle::seek(off_t position, int whence) {
    lock();
    if (whence == SEEK_CUR) {
        position += pos;
    } else if (whence == SEEK_END) {
        position += semihost_flen(_fh);
    } /* otherwise SEEK_SET, so position is fine */

    /* Always seems to return -1, so just ignore for now. */
    semihost_seek(_fh, position);
    pos = position;
    unlock();
    return position;
}

int SemihostedFileHandle::sync() {
    lock();
    int ret = semihost_ensure(_fh);
    unlock();
    return ret;
}

off_t SemihostedFileHandle::size() {
    lock();
    off_t off = semihost_flen(_fh);
    unlock();
    return off;
}

void SemihostedFileHandle::lock() {
    _mutex.lock();
}

void SemihostedFileHandle::unlock() {
    _mutex.unlock();
}

// Experimental support for printf in RawSerial. No Stream inheritance
// means we can't call printf() directly, so we use sprintf() instead.
// We only call malloc() for the sprintf() buffer if the buffer
// length is above a certain threshold, otherwise we use just the stack.
int SemihostedFileHandle::printf(const char *format, ...) {
    lock();
    std::va_list arg;
    va_start(arg, format);
    // ARMCC microlib does not properly handle a size of 0.
    // As a workaround supply a dummy buffer with a size of 1.
    char dummy_buf[1];
    int len = vsnprintf(dummy_buf, sizeof(dummy_buf), format, arg);
    if (len < STRING_STACK_LIMIT) {
        char temp[STRING_STACK_LIMIT];
        vsprintf(temp, format, arg);
        write(temp, len);
    } else {
        char *temp = new char[len + 1];
        vsprintf(temp, format, arg);
        write(temp, len);
        delete[] temp;
    }
    va_end(arg);
    unlock();
    return len;
}