#include "XL320_Driver.h"

XL320::XL320(uint32_t baud, PinName tx, PinName rx): sXL320(tx, rx)
{
    baudrate = baud;
    sXL320.baud(baudrate);

    iPacket[0] = 0xFF; //Header
    iPacket[1] = 0xFF; //Header
    iPacket[2] = 0xFD; //Header
    iPacket[3] = 0x00; //Reserved
}

/***** XL320 Instruction Method *****/
void XL320::iRead(uint8_t id, uint8_t length, uint16_t address)
{
    //Creates instruction packet to read data
    iPacket[4] = id;
    iPacket[5] = 0x07;
    iPacket[6] = 0x00;
    iPacket[7] = READ_DATA; //Instruction
    iPacket[8] = SHIFT_TO_LSB(address);
    iPacket[9] = SHIFT_TO_MSB(address);
    iPacket[10] = SHIFT_TO_LSB(length);
    iPacket[11] = SHIFT_TO_MSB(length);

    uint16_t CRC = update_crc(0, iPacket, 12);
    iPacket[12] = SHIFT_TO_LSB(CRC);
    iPacket[13] = SHIFT_TO_MSB(CRC);

    iPacketLength = 14;
}

void XL320::iWrite(uint8_t id, uint16_t address, uint8_t param[], uint8_t paramLength)
{
    //Creates instruction packet to write action
    int i;

    iPacket[4] = id;            //PacketID
    iPacket[5] = (5 + paramLength);  //Packet Lenght LEN_L
    iPacket[6] = 0x00;          //Packet Lenght LEN_H
    iPacket[7] = WRITE_DATA;    //Instruction
    iPacket[8] = SHIFT_TO_LSB(address);
    iPacket[9] = SHIFT_TO_MSB(address);
    for(i = 10; i < 10 + paramLength ; i++) iPacket[i] = param[i - 10];

    uint16_t CRC = update_crc(0, iPacket, 10 + paramLength);
    iPacket[i] = SHIFT_TO_LSB(CRC);
    iPacket[i + 1] = SHIFT_TO_MSB(CRC);

    iPacketLength = i + 2;
}

/***** Dynamixel Protocol 2.0 Methods *****/
void XL320::sendIPacket()
{
    float timeOut = 12.0 * (float) ((float) iPacketLength) / ((float) baudrate); //12 = 10 (start + 8 bits + stop) + 2 (gives a bit more time)

    //Send packet
    for(int i = 0; i < iPacketLength; i++) sXL320.putc(iPacket[i]);

    //Waits for sending before doing something else
    wait(timeOut);
}

void XL320::getRPacket()
{
    int i = 0;
    float timeOut = 12.0 * (float) ((float) rPacketLength) / ((float) baudrate); //12 = 10 (start + 8 bits + stop) + 2 (gives a bit more time)
    Timer t;

    t.start();
    while((i < rPacketLength) && (t.read() <= timeOut)) {
        if (sXL320.readable()) {
            rPacket[i] = sXL320.getc();
            //pc.putc(rPacket[i]); //Prints packets
            i++;
        }
    }
    t.stop();

    if (t.read() >= timeOut) rPacket[8] = 0x80; //Creates an error code
}

/***** Generic functions *****/
void XL320::SetSomething(uint8_t id, uint16_t address, uint8_t param[], uint8_t paramLength)
{
    iWrite(id, address, param, paramLength);
    rPacketLength = 11;

    sendIPacket();
    if (id != 0xFE) getRPacket(); //Broadcast does not returns packets
    else rPacket[8] = 0x00; //No error because no rPacket in broadcast

    switch(rPacket[8]) {
        case 0x00 : //No error
            break;

        case 0x80 :
            pc.printf("Missed status packet\r\n");
            break;

        default :   //Error occurred
            pc.printf("Error %d in status packet\r\n", rPacket[8]);
            break;
    }
}

uint8_t XL320::GetSomething(uint8_t id, uint16_t address)
{
    iRead(id, 1, address);
    rPacketLength = 12;

    sendIPacket();
    getRPacket();

    switch(rPacket[8]) {
        case 0x00 : //No error
            return rPacket[9];

        case 0x80 :
            pc.printf("Missed status packet\r\n");
            return rPacket[8];

        default :   //Error occurred
            pc.printf("Error %d in status packet\r\n", rPacket[8]);
            return rPacket[8]; //Return error status
    }
}

uint16_t XL320::GetSomethingElse(uint8_t id, uint16_t address)
{
    iRead(id, 2, address);
    rPacketLength = 13;

    sendIPacket();
    getRPacket();

    switch(rPacket[8]) {
        case 0x00 : //No error
            return (uint16_t)rPacket[9] | (((uint16_t)rPacket[10]<<8)&0xFF00);

        case 0x80 :
            pc.printf("Missed status packet\r\n");
            return (uint16_t)rPacket[8];

        default :   //Error occurred
            pc.printf("Error %d in status packet\r\n", rPacket[8]);
            return (uint16_t)rPacket[8]; //Return error status
    }
}

////// EEPROM ACCESS METHODS //////

/***** XL320 Network Parameter *****/
void XL320::SetID(uint8_t id, uint8_t newId)
{
    uint8_t parameter[1];
    parameter[0] = newId;

    SetSomething(id, XL_ID, parameter, 1);
    wait_ms(200);   //Waits for the value to be written in EEPROM
}

uint8_t XL320::GetID(uint8_t id)
{
    return GetSomething(id, XL_ID);
}

void XL320::SetBaudRate(uint8_t id, uint8_t baudrt)
{
    uint8_t parameter[1];
    parameter[0] = baudrt;

    SetSomething(id, BAUD_RATE, parameter, 1);

    switch (baudrt) {
        case 0:
            sXL320.baud(9600);
            break;
        case 1:
            sXL320.baud(57600);
            break;
        case 2:
            sXL320.baud(115200);
            break;
        case 3:
            sXL320.baud(1000000);
            break;
    }

    wait_ms(200);   //Waits for the value to be written in EEPROM
}

uint8_t XL320::GetBaudRate(uint8_t id)
{
    return GetSomething(id, BAUD_RATE);
}

void XL320::SetRetDelTime(uint8_t id,uint8_t time)
{
    uint8_t parameter[1];
    parameter[0] = time;

    SetSomething(id, RETURN_DELAY_TIME, parameter, 1);
    wait_ms(200);   //Waits for the value to be written in EEPROM
}

uint8_t XL320::GetRetDelTime(uint8_t id)
{
    return GetSomething(id, RETURN_DELAY_TIME);
}

/***** XL320 Motor Setting *****/
void XL320::SetCWAngLim(uint8_t id, uint16_t angle)
{
    uint8_t parameter[2];
    parameter[0] = SHIFT_TO_LSB(angle);
    parameter[1] = SHIFT_TO_MSB(angle);

    SetSomething(id, CW_ANGLE_LIMIT, parameter, 2);
    wait_ms(200);   //Waits for the value to be written in EEPROM
}

uint16_t XL320::GetCWAngLim(uint8_t id)
{
    return GetSomethingElse(id, CW_ANGLE_LIMIT);
}

void XL320::SetCCWAngLim(uint8_t id, uint16_t angle)
{
    uint8_t parameter[2];
    parameter[0] = SHIFT_TO_LSB(angle);
    parameter[1] = SHIFT_TO_MSB(angle);

    SetSomething(id, CCW_ANGLE_LIMIT, parameter, 2);
    wait_ms(200);   //Waits for the value to be written in EEPROM
}

uint16_t XL320::GetCCWAngLim(uint8_t id)
{
    return GetSomethingElse(id, CCW_ANGLE_LIMIT);
}

void XL320::SetContMode(uint8_t id, uint8_t mode)
{
    uint8_t parameter[1];
    parameter[0] = mode;

    SetSomething(id, CONTROL_MODE, parameter, 1);
    wait_ms(200);   //Waits for the value to be written in EEPROM
}

uint8_t XL320::GetContMode(uint8_t id)
{
    return GetSomething(id, CONTROL_MODE);
}

void XL320::SetTempLim(uint8_t id, uint8_t temp)
{
    uint8_t parameter[1];
    parameter[0] = temp;

    SetSomething(id, LIMIT_TEMPERATURE, parameter, 1);
    wait_ms(200);   //Waits for the value to be written in EEPROM
}

uint8_t XL320::GetTempLim(uint8_t id)
{
    return GetSomething(id, LIMIT_TEMPERATURE);
}

void XL320::SetLowVoltLim(uint8_t id, uint8_t volt)
{
    uint8_t parameter[1];
    parameter[0] = volt;

    SetSomething(id, LOWER_LIMIT_VOLTAGE, parameter, 1);
    wait_ms(200);   //Waits for the value to be written in EEPROM
}

uint8_t XL320::GetLowVoltLim(uint8_t id)
{
    return GetSomething(id, LOWER_LIMIT_VOLTAGE);
}

void XL320::SetUpVoltLim(uint8_t id, uint8_t volt)
{
    uint8_t parameter[1];
    parameter[0] = volt;

    SetSomething(id, UPPER_LIMIT_VOLTAGE, parameter, 1);
    wait_ms(200);   //Waits for the value to be written in EEPROM
}

uint8_t XL320::GetUpVoltLim(uint8_t id)
{
    return GetSomething(id, UPPER_LIMIT_VOLTAGE);
}

void XL320::SetMaxTorq(uint8_t id, uint16_t torque)
{
    uint8_t parameter[2];
    parameter[0] = SHIFT_TO_LSB(torque);
    parameter[1] = SHIFT_TO_MSB(torque);

    SetSomething(id, MAX_TORQUE, parameter, 2);
    wait_ms(200);   //Waits for the value to be written in EEPROM
}

uint16_t XL320::GetMaxTorq(uint8_t id)
{
    return GetSomethingElse(id, MAX_TORQUE);
}

void XL320::SetRetLev(uint8_t id, uint8_t level)
{
    uint8_t parameter[1];
    parameter[0] = level;

    SetSomething(id, RETURN_LEVEL, parameter, 1);
    wait_ms(200);   //Waits for the value to be written in EEPROM
}

uint8_t XL320::GetRetLev(uint8_t id)
{
    return GetSomething(id, RETURN_LEVEL);
}

void XL320::SetAlarmShut(uint8_t id, uint8_t alarm)
{
    uint8_t parameter[1];
    parameter[0] = alarm;

    SetSomething(id, ALARM_SHUTDOWN, parameter, 1);
    wait_ms(200);   //Waits for the value to be written in EEPROM
}

uint8_t XL320::GetAlarmShut(uint8_t id)
{
    return GetSomething(id, ALARM_SHUTDOWN);
}

////// RAM ACCESS METHODS //////

/***** XL320 On/Off *****/
void XL320::SetTorqueEn(uint8_t id, uint8_t enable)
{
    uint8_t parameter[1];
    parameter[0] = enable;

    SetSomething(id, TORQUE_ENABLE, parameter, 1);
}

uint8_t XL320::GetTorqueEn(uint8_t id)
{
    return GetSomething(id, TORQUE_ENABLE);
}

void XL320::TurnOnLED(uint8_t id, uint8_t led)
{
    uint8_t parameter[1];
    parameter[0] = led;

    SetSomething(id, LED, parameter, 1);
}

uint8_t XL320::GetStatusLED(uint8_t id)
{
    return GetSomething(id, LED);
}

/***** XL320 Motor Control *****/
void XL320::SetDGain(uint8_t id, uint8_t d_cons)
{
    uint8_t parameter[1];
    parameter[0] = d_cons;

    SetSomething(id, D_GAIN, parameter, 1);
}

uint8_t XL320::GetDGain(uint8_t id)
{
    return GetSomething(id, D_GAIN);
}

void XL320::SetIGain(uint8_t id, uint8_t i_cons)
{
    uint8_t parameter[1];
    parameter[0] = i_cons;

    SetSomething(id, I_GAIN, parameter, 1);
}

uint8_t XL320::GetIGain(uint8_t id)
{
    return GetSomething(id, I_GAIN);
}

void XL320::SetPGain(uint8_t id, uint8_t p_cons)
{
    uint8_t parameter[1];
    parameter[0] = p_cons;

    SetSomething(id, P_GAIN, parameter, 1);
}

uint8_t XL320::GetPGain(uint8_t id)
{
    return GetSomething(id, P_GAIN);
}

void XL320::SetGoalPos(uint8_t id, uint16_t position)
{
    uint8_t parameter[2];
    parameter[0] = SHIFT_TO_LSB(position);
    parameter[1] = SHIFT_TO_MSB(position);

    SetSomething(id, GOAL_POSITION, parameter, 2);
}

uint16_t XL320::GetGoalPos(uint8_t id)
{
    return GetSomethingElse(id, GOAL_POSITION);
}

void XL320::SetGoalVel(uint8_t id, uint16_t velocity)
{
    uint8_t parameter[2];
    parameter[0] = SHIFT_TO_LSB(velocity);
    parameter[1] = SHIFT_TO_MSB(velocity);

    SetSomething(id, GOAL_SPEED, parameter, 2);
}

uint16_t XL320::GetGoalVel(uint8_t id)
{
    return GetSomethingElse(id, GOAL_SPEED);
}

void XL320::SetGoalTorq(uint8_t id, uint16_t torque)
{
    uint8_t parameter[2];
    parameter[0] = SHIFT_TO_LSB(torque);
    parameter[1] = SHIFT_TO_MSB(torque);

    SetSomething(id, GOAL_TORQUE, parameter, 2);
}

uint16_t XL320::GetGoalTorq(uint8_t id)
{
    return GetSomethingElse(id, GOAL_TORQUE);
}

/***** XL320 Feedback *****/
uint16_t XL320::GetPresentPos(uint8_t id)
{
    return GetSomethingElse(id, PRESENT_POSITION);
}

uint16_t XL320::GetPresentSpeed(uint8_t id)
{
    return GetSomethingElse(id, PRESENT_SPEED);
}

uint16_t XL320::GetPresentLoad(uint8_t id)
{
    return GetSomethingElse(id, PRESENT_LOAD);
}

uint8_t XL320::GetPresentVolt(uint8_t id)
{
    return GetSomething(id, PRESENT_VOLTAGE);
}

uint8_t XL320::GetPresentTemp(uint8_t id)
{
    return GetSomething(id, PRESENT_TEMPERATURE);
}

/***** XL320 Status *****/
uint8_t XL320::GetRegInst(uint8_t id)
{
    return GetSomething(id, REGISTERED_INSTRUCTION);
}

uint8_t XL320::GetMoving(uint8_t id)
{
    return GetSomething(id, MOVING);
}

uint8_t XL320::GetHWErr(uint8_t id)
{
    return GetSomething(id, HARDWARE_ERROR_STATUS);
}

void XL320::SetPunch(uint8_t id, uint16_t punch)
{
    uint8_t parameter[2];
    parameter[0] = SHIFT_TO_LSB(punch);
    parameter[1] = SHIFT_TO_MSB(punch);

    SetSomething(id, PUNCH, parameter, 2);
}

uint16_t XL320::GetPunch(uint8_t id)
{
    return GetSomethingElse(id, PUNCH);
}

/***** XL320 Instruction Method *****/
void XL320::FactoryReset(uint8_t id, uint8_t option)
{
    iPacket[4] = id;
    iPacket[5] = 0x04;
    iPacket[6] = 0x00;
    iPacket[7] = FACTORY_RESET; //Instruction
    iPacket[8] = option;

    uint16_t CRC = update_crc(0, iPacket, 9);
    iPacket[9] = SHIFT_TO_LSB(CRC);
    iPacket[10] = SHIFT_TO_MSB(CRC);

    iPacketLength = 11;
    rPacketLength = 11;

    sendIPacket();
    getRPacket();
}

void XL320::Ping(uint8_t id)
{
    iPacket[4] = id;
    iPacket[5] = 0x03;
    iPacket[6] = 0x00;
    iPacket[7] = PING; //Instruction

    uint16_t CRC = update_crc(0, iPacket, 9);
    iPacket[8] = SHIFT_TO_LSB(CRC);
    iPacket[9] = SHIFT_TO_MSB(CRC);

    iPacketLength = 10;
    rPacketLength = 14;

    sendIPacket();
    getRPacket();
}

/***** Calculating CRC Method *****/
unsigned short XL320::update_crc(unsigned short crc_accum, unsigned char *data_blk_ptr, unsigned short data_blk_size)
{
    unsigned short i, j;
    unsigned short crc_table[256] = {
        0x0000, 0x8005, 0x800F, 0x000A, 0x801B, 0x001E, 0x0014, 0x8011,
        0x8033, 0x0036, 0x003C, 0x8039, 0x0028, 0x802D, 0x8027, 0x0022,
        0x8063, 0x0066, 0x006C, 0x8069, 0x0078, 0x807D, 0x8077, 0x0072,
        0x0050, 0x8055, 0x805F, 0x005A, 0x804B, 0x004E, 0x0044, 0x8041,
        0x80C3, 0x00C6, 0x00CC, 0x80C9, 0x00D8, 0x80DD, 0x80D7, 0x00D2,
        0x00F0, 0x80F5, 0x80FF, 0x00FA, 0x80EB, 0x00EE, 0x00E4, 0x80E1,
        0x00A0, 0x80A5, 0x80AF, 0x00AA, 0x80BB, 0x00BE, 0x00B4, 0x80B1,
        0x8093, 0x0096, 0x009C, 0x8099, 0x0088, 0x808D, 0x8087, 0x0082,
        0x8183, 0x0186, 0x018C, 0x8189, 0x0198, 0x819D, 0x8197, 0x0192,
        0x01B0, 0x81B5, 0x81BF, 0x01BA, 0x81AB, 0x01AE, 0x01A4, 0x81A1,
        0x01E0, 0x81E5, 0x81EF, 0x01EA, 0x81FB, 0x01FE, 0x01F4, 0x81F1,
        0x81D3, 0x01D6, 0x01DC, 0x81D9, 0x01C8, 0x81CD, 0x81C7, 0x01C2,
        0x0140, 0x8145, 0x814F, 0x014A, 0x815B, 0x015E, 0x0154, 0x8151,
        0x8173, 0x0176, 0x017C, 0x8179, 0x0168, 0x816D, 0x8167, 0x0162,
        0x8123, 0x0126, 0x012C, 0x8129, 0x0138, 0x813D, 0x8137, 0x0132,
        0x0110, 0x8115, 0x811F, 0x011A, 0x810B, 0x010E, 0x0104, 0x8101,
        0x8303, 0x0306, 0x030C, 0x8309, 0x0318, 0x831D, 0x8317, 0x0312,
        0x0330, 0x8335, 0x833F, 0x033A, 0x832B, 0x032E, 0x0324, 0x8321,
        0x0360, 0x8365, 0x836F, 0x036A, 0x837B, 0x037E, 0x0374, 0x8371,
        0x8353, 0x0356, 0x035C, 0x8359, 0x0348, 0x834D, 0x8347, 0x0342,
        0x03C0, 0x83C5, 0x83CF, 0x03CA, 0x83DB, 0x03DE, 0x03D4, 0x83D1,
        0x83F3, 0x03F6, 0x03FC, 0x83F9, 0x03E8, 0x83ED, 0x83E7, 0x03E2,
        0x83A3, 0x03A6, 0x03AC, 0x83A9, 0x03B8, 0x83BD, 0x83B7, 0x03B2,
        0x0390, 0x8395, 0x839F, 0x039A, 0x838B, 0x038E, 0x0384, 0x8381,
        0x0280, 0x8285, 0x828F, 0x028A, 0x829B, 0x029E, 0x0294, 0x8291,
        0x82B3, 0x02B6, 0x02BC, 0x82B9, 0x02A8, 0x82AD, 0x82A7, 0x02A2,
        0x82E3, 0x02E6, 0x02EC, 0x82E9, 0x02F8, 0x82FD, 0x82F7, 0x02F2,
        0x02D0, 0x82D5, 0x82DF, 0x02DA, 0x82CB, 0x02CE, 0x02C4, 0x82C1,
        0x8243, 0x0246, 0x024C, 0x8249, 0x0258, 0x825D, 0x8257, 0x0252,
        0x0270, 0x8275, 0x827F, 0x027A, 0x826B, 0x026E, 0x0264, 0x8261,
        0x0220, 0x8225, 0x822F, 0x022A, 0x823B, 0x023E, 0x0234, 0x8231,
        0x8213, 0x0216, 0x021C, 0x8219, 0x0208, 0x820D, 0x8207, 0x0202
    };

    for(j = 0; j < data_blk_size; j++) {
        i = ((unsigned short)(crc_accum >> 8) ^ data_blk_ptr[j]) & 0xFF;
        crc_accum = (crc_accum << 8) ^ crc_table[i];
    }
    return crc_accum;
}

XL320::~XL320() {};