/*******************************************************************************
This library was developed at ECAM Lyon by Alexandre Simon

This code is inspired from an Mbed library for Dynamixel XL-320 developed by
Muhammad Fajar Adityo.

The XL-320 motor chain needs to be powered externally using for instance the
card "Poppy PIXL" that can be found at https://github.com/poppy-project/pixl
This board powers up your motors and provides a conversion from the 5V TTL half
duplex bus of Dynamixel motors from the 3.3V UART ports of the Mbed board
without echo of outcoming serial frames from Mbed microcontroller.

This library is based on the Dynamixel XL-320 data sheet available at
http://support.robotis.com/en/product/actuator/dynamixel_x/xl_series/xl-320.htm

This library is based on the Dynamixel communication protocol available at
http://support.robotis.com/en/product/actuator/dynamixel_pro/communication/
instruction_status_packet.htm
*******************************************************************************/

#ifndef XL320_H
#define XL320_H

#include "mbed.h"

static Serial pc(USBTX,USBRX);

////////// SHIFT BYTES ///////////
#define SHIFT_TO_LSB(w)         (uint8_t((uint16_t(w)) & 0x00ff))
#define SHIFT_TO_MSB(w)         (uint8_t((uint16_t(w) >> 8) & 0x00ff))

////////// EEPROM Address ////////      //Access    Initial Value
#define MODEL_NUMBER            0x00    //R         350     -
#define VERSION                 0x02    //R         -       -
#define XL_ID                   0x03    //RW        1       0 ~ 252(0xFC) Broadcast = 0xFE
#define BAUD_RATE               0x04    //RW        3       0 : 9600, 1 : 57600, 2 : 115200, 3 : 1Mbps
#define RETURN_DELAY_TIME       0x05    //RW        250     0 ~ 254
#define CW_ANGLE_LIMIT          0x06    //RW        0       0 ~ 1023
#define CCW_ANGLE_LIMIT         0x08    //RW        1023    0 ~ 1023
#define SYSTEM_DATA2            0x0A
#define CONTROL_MODE            0x0B    //RW        2       1 : Wheel mode (spin infinitely), 2 : Join mode
#define LIMIT_TEMPERATURE       0x0C    //RW        65      0 ~ 150
#define LOWER_LIMIT_VOLTAGE     0x0D    //RW        60      50 ~ 250. 10 = 1V
#define UPPER_LIMIT_VOLTAGE     0x0E    //RW        90      50 ~ 250. 10 = 1V
#define MAX_TORQUE              0x0F    //RW        1023    0 ~ 1023
#define RETURN_LEVEL            0x11    //RW        2       0 : No return excepted Ping, 1 : Return only for READ command, 2 : Return for all commands
#define ALARM_SHUTDOWN          0x12    //RW        3       0 ~ 7

////////// RAM Address ///////////
#define TORQUE_ENABLE           0X18    //RW        0       0 ~ 1
#define LED                     0x19    //RW        0       0 ~ 7. 0 : off, 1 : red, 2 : green...
#define D_GAIN                  0x1B    //RW        0       0 ~ 254
#define I_GAIN                  0x1C    //RW        0       0 ~ 254
#define P_GAIN                  0x1D    //RW        32      0 ~ 254
#define GOAL_POSITION           0x1E    //RW        -       0 ~ 1023. 1 unit = 0.29 degree
#define GOAL_SPEED              0x20    //RW        -       0 ~ 2047
#define GOAL_TORQUE             0x23    //RW        -       0 ~ 1023
#define PRESENT_POSITION        0x25    //R         -       -
#define PRESENT_SPEED           0x27    //R         -       -
#define PRESENT_LOAD            0x29    //R         -       -
#define PRESENT_VOLTAGE         0x2D    //R         -       -
#define PRESENT_TEMPERATURE     0x2E    //R         -       -
#define REGISTERED_INSTRUCTION  0x2F    //R         0       -
#define MOVING                  0x31    //R         0       -
#define HARDWARE_ERROR_STATUS   0x32    //R         0       -
#define PUNCH                   0x33    //RW        32      0 ~ 1023

////////// INSTRUCTION ///////////
#define PING                    0x01
#define READ_DATA               0x02
#define WRITE_DATA              0x03
#define REG_WRITE               0x04
#define ACTION                  0x05
#define FACTORY_RESET           0x06
#define REBOOT                  0x08
#define STATUS                  0x55
#define SYNC_READ               0x82
#define SYNC_WRITE              0x83
#define BULK_READ               0x92
#define BULK_WRITE              0x93

////////// ERROR ///////////
#define RESULT_FAIL             0x01
#define INS_ERR                 0x02
#define CRC_ERR                 0x03
#define DATA_RANGE_ERR          0x04
#define DATA_LENGTH_ERR         0x05
#define DATA_LIMIT_ERR          0x06
#define ACCESS_ERR              0x07

////////// RESET ///////////
#define RESET_ALL               0xFF
#define RESET_EXP_ID            0x01
#define RESET_EXP_ID_BR         0x02

class XL320
{
public:
    XL320(uint32_t baud, PinName tx, PinName rx);
    ~XL320();

    ////// EEPROM ACCESS METHODS //////

    /***** XL320 Network Parameter *****/
    void SetID(uint8_t id, uint8_t newId);
    uint8_t GetID(uint8_t id);
    void SetBaudRate(uint8_t id, uint8_t baudrate);
    uint8_t GetBaudRate(uint8_t id);
    void SetRetDelTime(uint8_t id, uint8_t time);
    uint8_t GetRetDelTime(uint8_t id);

    /***** XL320 Motor Setting *****/
    void SetCWAngLim(uint8_t id, uint16_t angle);
    uint16_t GetCWAngLim(uint8_t id);
    void SetCCWAngLim(uint8_t id, uint16_t angle);
    uint16_t GetCCWAngLim(uint8_t id);
    void SetContMode(uint8_t id, uint8_t mode);
    uint8_t GetContMode(uint8_t id);
    void SetTempLim(uint8_t id, uint8_t temp);
    uint8_t GetTempLim(uint8_t id);
    void SetLowVoltLim(uint8_t id, uint8_t volt);
    uint8_t GetLowVoltLim(uint8_t id);
    void SetUpVoltLim(uint8_t id, uint8_t volt);
    uint8_t GetUpVoltLim(uint8_t id);
    void SetMaxTorq(uint8_t id, uint16_t torque);
    uint16_t GetMaxTorq(uint8_t id);
    void SetRetLev(uint8_t id, uint8_t level);
    uint8_t GetRetLev(uint8_t id);
    void SetAlarmShut(uint8_t id, uint8_t alarm);
    uint8_t GetAlarmShut(uint8_t id);

    ////// RAM ACCESS METHODS //////

    /***** XL320 On/Off *****/
    void SetTorqueEn(uint8_t id, uint8_t enable);
    uint8_t GetTorqueEn(uint8_t id);
    void TurnOnLED(uint8_t id, uint8_t led);
    uint8_t GetStatusLED(uint8_t id);

    /***** XL320 Motor Control *****/
    void SetDGain(uint8_t id, uint8_t d_cons);
    uint8_t GetDGain(uint8_t id);
    void SetIGain(uint8_t id, uint8_t i_cons);
    uint8_t GetIGain(uint8_t id);
    void SetPGain(uint8_t id, uint8_t p_cons);
    uint8_t GetPGain(uint8_t id);
    void SetGoalPos(uint8_t id, uint16_t position);
    uint16_t GetGoalPos(uint8_t id);
    void SetGoalVel(uint8_t id, uint16_t velocity);
    uint16_t GetGoalVel(uint8_t id);
    void SetGoalTorq(uint8_t id, uint16_t torque);
    uint16_t GetGoalTorq(uint8_t id);

    /***** XL320 Feedback *****/
    uint16_t GetPresentPos(uint8_t id);
    uint16_t GetPresentSpeed(uint8_t id);
    uint16_t GetPresentLoad(uint8_t id);
    uint8_t GetPresentVolt(uint8_t id);
    uint8_t GetPresentTemp(uint8_t id);

    /***** XL320 Status *****/
    uint8_t GetRegInst(uint8_t id);
    uint8_t GetMoving(uint8_t id);
    uint8_t GetHWErr(uint8_t id);
    void SetPunch(uint8_t id, uint16_t punch);
    uint16_t GetPunch(uint8_t id);

    /***** XL320 Instruction Method *****/
    void FactoryReset(uint8_t id, uint8_t option);
    void Ping(uint8_t id);

private:
    /***** Generic functions *****/
    void SetSomething(uint8_t id, uint16_t address, uint8_t param[], uint8_t paramLength);
    uint8_t GetSomething(uint8_t id, uint16_t address);
    uint16_t GetSomethingElse(uint8_t id, uint16_t address);

    /***** XL320 Instruction Method *****/
    void iRead(uint8_t id, uint8_t length, uint16_t address);
    void iWrite(uint8_t id, uint16_t address, uint8_t param[], uint8_t paramLength);

    /***** Dynamixel Protocol 2.0 Methods *****/
    void sendIPacket();
    void getRPacket();

    /***** Calculating CRC Method *****/
    unsigned short update_crc(unsigned short crc_accum, unsigned char *data_blk_ptr, unsigned short data_blk_size);

    /***** Variables *****/
    uint32_t baudrate;      //Baudrate
    uint8_t iPacket[20];    //Instruction Packet
    uint8_t iPacketLength;  //Instruction Packet Length
    uint8_t rPacket[20];    //Return Packet
    uint8_t rPacketLength;  //Return Packet Length

    Serial sXL320;          //Serial Line for XL320
};

#endif