#include "Poppy_Arm.h"

Poppy_Arm::Poppy_Arm(uint32_t baudrate, PinName tx, PinName rx) : XL320(baudrate, tx, rx)
{
    dim[0] = 33;
    dim[1] = 22;
    dim[2] = 55;
    dim[3] = 46;
    dim[4] = 49;
    dim[5] = 128;
}

uint8_t Poppy_Arm::initNewServo(uint32_t currentBaudRate, uint32_t newBaudRate, uint8_t newId)
{
    //Sets the id and the baud rate of all connected motors
    if (currentBaudRate != 9600 && currentBaudRate != 57600 && currentBaudRate != 115200 && currentBaudRate != 1000000) {
        pc.printf("You did not enter a valid currentBaudRate");
        return 1;
    }

    if (newBaudRate != 9600 && newBaudRate != 57600 && newBaudRate != 115200 && newBaudRate != 1000000) {
        pc.printf("You did not enter a valid newBaudRate");
        return 2;
    }

    Poppy_Arm s1(currentBaudRate, p9, p10); //Creates a Poppy_Arm object at the current baud rate
    switch (newBaudRate) {  //Changes all connected motors baud rate to the new baud rate
        case 9600:
            s1.SetBaudRate(0xFE, 0);
            break;
        case 57600:
            s1.SetBaudRate(0xFE, 1);
            break;
        case 115200:
            s1.SetBaudRate(0xFE, 2);
            break;
        case 1000000:
            s1.SetBaudRate(0xFE, 3);
            break;
        default:
            return 2;
    }
    Poppy_Arm s2(newBaudRate, p9, p10);     //Creates a Poppy_Arm object at the new baud rate
    s2.SetID(0xFE, newId);                  //Changes all connected motors's id
    s2.TurnOnLED(newId , 2);                //Turns on the green led to verify that changes worked
    return 0;
}

void Poppy_Arm::ChristmasTree()
{
    //Leds blinks green and red
    for (uint8_t i = 1; i <= 6; i++) TurnOnLED(i, 2);
    for (uint8_t i = 6; i >= 1; i--) TurnOnLED(i, 1);
}

void Poppy_Arm::FindCoordinates()
{
    //Finds the current coordinates of the tip of the arm
    double angle[5];
    for (uint8_t i = 0; i <= 4; i++) angle[i] = (double) (0.29 * (GetPresentPos(i + 1) - 512)); //512 units is the 0 degree for calculation
    CalculateCoordinates(angle);
}

void Poppy_Arm::CalculateCoordinates(double angle[])
{
    //Calculates the coordinates of the tip of the arm from the motors angles in degree
    double co[5], si[5];
    for (uint8_t i = 0; i <= 4; i++) {
        co[i] = cos(angle[i] * PI / 180);
        si[i] = sin(angle[i] * PI / 180);
    }
    coord[0] = -co[0]*si[1]*dim[2] + dim[3]*(-co[0]*co[1]*si[2] - co[0]*co[2]*si[1]) + dim[4]*(co[3]*(co[0]*co[1]*co[2] - co[0]*si[1]*si[2]) - si[0]*si[3]) + dim[5]*(co[4]*(co[3]*(co[0]*co[1]*co[2] - co[0]*si[1]*si[2]) - si[0]*si[3]) + si[4]*(-co[0]*co[1]*si[2] - co[0]*co[2]*si[1]));
    coord[1] = -si[0]*si[1]*dim[2] + dim[3]*(-co[1]*si[0]*si[2] - co[2]*si[0]*si[1]) + dim[4]*(co[0]*si[3] + co[3]*(co[1]*co[2]*si[0] - si[0]*si[1]*si[2])) + dim[5]*(co[4]*(co[0]*si[3] + co[3]*(co[1]*co[2]*si[0] - si[0]*si[1]*si[2])) + si[4]*(-co[1]*si[0]*si[2] - co[2]*si[0]*si[1]));
    coord[2] = co[1]*dim[2] + co[3]*dim[4]*(co[1]*si[2] + co[2]*si[1]) + dim[0] + dim[1] + dim[3]*(co[1]*co[2] - si[1]*si[2]) + dim[5]*(co[3]*co[4]*(co[1]*si[2] + co[2]*si[1]) + si[4]*(co[1]*co[2] - si[1]*si[2]));
}

void Poppy_Arm::MoveToCoordinates(double xCoord, double yCoord, double zCoord)
{
    //Moves the tip of the arm to the desired coordinates in millimeters
    double angle[5], bestCoord[3] = {0}, bestAngleDeg[5];
    uint16_t bestAngle[5];

    if (xCoord == 0) angle[0] = (double) (90); //Moves the first motor in the right direction
    else angle[0] = (double) (atan(yCoord / xCoord) * 180 / PI);

    angle[3] = (double) (0); //Motor 4 stays in position 512 to reduce the number of possibilities (512 = 0 degree for calculation)

    for (int16_t i = -90; i <= 90; i = i + 15) { //This range is to avoid that the arm bumps itself
        for (int16_t j = -90; j <= 90; j = j + 15) {
            for (int16_t k = -90; k <= 90; k = k + 15) {
                angle[1] = (double) (i);
                angle[2] = (double) (j);
                angle[4] = (double) (k);
                CalculateCoordinates(angle);
                if (sqrt(pow(xCoord - coord[0], 2) + pow(yCoord - coord[1], 2) + pow(zCoord - coord[2], 2)) < sqrt(pow(xCoord - bestCoord[0], 2) + pow(yCoord - bestCoord[1], 2) + pow(zCoord - bestCoord[2], 2))) {
                    //If the distance between the new coordinates and the desired coordinates is smaller than between the last best coordinates and the desired coordinates then we save the new coordinates
                    for (uint8_t l = 0; l <= 2; l++) bestCoord[l] = coord[l];   //Saves the best coordinates
                    for (uint8_t l = 0; l <= 4; l++) bestAngleDeg[l] = angle[l]; //Saves the best angles in degrees
                    for (uint8_t l = 0; l <= 4; l++) bestAngle[l] = (uint16_t) (512 + angle[l] / 0.29); //Saves the best angles in motor units
                }
            }
        }
    }

    //This part is used to reach more precisely the targeted position. It can be commented
    int16_t accuracy = 7; //Accuracy can be changed but calculation time will increase
    int16_t lBound[5] = {0, bestAngleDeg[1] - accuracy, bestAngleDeg[2] - accuracy, 0, bestAngleDeg[4] - accuracy};
    int16_t uBound[5] = {0, bestAngleDeg[1] + accuracy, bestAngleDeg[2] + accuracy, 0, bestAngleDeg[4] + accuracy};

    for (int16_t i = lBound[1]; i <= uBound[1]; i++) {
        for (int16_t j = lBound[2]; j <= uBound[2]; j++) {
            for (int16_t k = lBound[4]; k <= uBound[4]; k++) {
                angle[1] = (double) (i);
                angle[2] = (double) (j);
                angle[4] = (double) (k);
                CalculateCoordinates(angle);
                if (sqrt(pow(xCoord - coord[0], 2) + pow(yCoord - coord[1], 2) + pow(zCoord - coord[2], 2)) < sqrt(pow(xCoord - bestCoord[0], 2) + pow(yCoord - bestCoord[1], 2) + pow(zCoord - bestCoord[2], 2))) {
                    //If the distance of the new coordinates to the desired coordinates is smaller than the last best coordinates then we save them
                    for (uint8_t l = 0; l <= 2; l++) bestCoord[l] = coord[l];   //Saves the best coordinates
                    for (uint8_t l = 0; l <= 4; l++) bestAngleDeg[l] = angle[l]; //Saves the best angles in degrees
                    for (uint8_t l = 0; l <= 4; l++) bestAngle[l] = (uint16_t) (512 + angle[l] / 0.29); //Saves the best angles in motor units
                }
            }
        }
    }

    pc.putc(' '); //Without this line the arm does not move...
    for (uint8_t i = 0; i <= 4; i++) SetGoalPos(i + 1, bestAngle[i]); //Moves the arm
}

double Poppy_Arm::GetCoordinates(uint8_t Coord) //0 = x, 1 = y, 2 = z
{
    //Gets the requested coordinate of the tip of the arm after the execution of FindCoordinates()
    switch(Coord) {
        case 0 : //x
            return coord[0];
        case 1 : //y
            return coord[1];
        case 2 : //z
            return coord[2];
        default : //Wrong input
            pc.printf("Error : Wrong input in GetCoordinates. Out of range (0 to 2)\r\n");
            return 32202; //Return "Error" in Leet speak
    }
}

void Poppy_Arm::OpenClamp(bool open)
{
    //Opens or closes the clamp
    if (open == true) SetGoalPos(6, (uint16_t) (512 - 90 / 0.29));
    else SetGoalPos(6, (uint16_t) (512 + 20 / 0.29));
}

Poppy_Arm::~Poppy_Arm() {};