/**********************************************************************
* Copyright (C) 2016 Maxim Integrated Products, Inc., All Rights Reserved.
*
* Permission is hereby granted, free of charge, to any person obtaining a
* copy of this software and associated documentation files (the "Software"),
* to deal in the Software without restriction, including without limitation
* the rights to use, copy, modify, merge, publish, distribute, sublicense,
* and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included
* in all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
* OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
* IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
* OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
* ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
* OTHER DEALINGS IN THE SOFTWARE.
*
* Except as contained in this notice, the name of Maxim Integrated
* Products, Inc. shall not be used except as stated in the Maxim Integrated
* Products, Inc. Branding Policy.
*
* The mere transfer of this software does not imply any licenses
* of trade secrets, proprietary technology, copyrights, patents,
* trademarks, maskwork rights, or any other form of intellectual
* property whatsoever. Maxim Integrated Products, Inc. retains all
* ownership rights.
**********************************************************************/


#include "GridEye/GridEye.h"



GridEye::GridEye(I2C &i2c)
: m_i2cBus(i2c)
{
    //Perform a software reset upon power on.
    softwareReset();
}




//*********************************************************************
int8_t GridEye::gridEyeReadReg(GridEyeRegister reg_addr, int num_bytes, char * data_buf)
{    
    int8_t result;
    char read_pointer[1];
    read_pointer[0] = reg_addr;
            
    result = GridEye::m_i2cBus.write(I2C_W_ADRS_GRIDEYE, read_pointer, 1);
    if (result == I2C_WR_SUCCESS) 
    { 
        result = GridEye::m_i2cBus.read(I2C_R_ADRS_GRIDEYE, data_buf , num_bytes, false);
        if (result == I2C_WR_SUCCESS)
            return result;
    }
    return I2C_WR_ERROR;
}



//*********************************************************************
int8_t GridEye::gridEyeWriteReg(GridEyeRegister reg_addr, int num_bytes, char * data_buf)
{    
    int8_t result;
    int num_indices = num_bytes + 1;
    char write_buffer[num_indices];
    
    //construct write buffer with the address followed by the data to be written
    write_buffer[0] = reg_addr;
    for (int idx=0; idx<num_bytes; idx++)
        write_buffer[idx+1] = data_buf[idx];
    
    result = GridEye::m_i2cBus.write(I2C_W_ADRS_GRIDEYE, write_buffer, num_indices);
    if (result == I2C_WR_SUCCESS)
        return result;
        
    //else, return -1
    return I2C_WR_ERROR;
}


//*********************************************************************
int8_t GridEye::getThermistorTemperature(int16_t & therm_temp)
{
    int8_t result;
    char data[2];
    
    result = this->gridEyeReadReg(GridEye::THERMISTOR_LOW, 2, data);
    
    if(result == I2C_WR_SUCCESS) 
    {
        convSingleRawTempData2Int(data, therm_temp);
        return therm_temp;
    }
    return I2C_WR_ERROR;
}
 
 
//*********************************************************************    
int8_t GridEye::getPixelTemperature(uint8_t pixel_addr, int16_t & pixel_temp)
{
    int8_t result;
    char data[2];
    
    result = this->gridEyeReadReg((GridEyeRegister) pixel_addr, 2, data);
    if(result == I2C_WR_SUCCESS)
    {
        convSingleRawTempData2Int(data, pixel_temp);
        return pixel_temp;
    }
    return I2C_WR_ERROR;
}
 
 
//*********************************************************************
int8_t GridEye::getRaw8x8FrameData(char * raw_frame_data)
{
    int8_t result;
    result = gridEyeReadReg((GridEyeRegister) GridEye::PIXEL_BASE_ADRS, 128, raw_frame_data);
    return result;
}

//*********************************************************************
int8_t GridEye::softwareReset() 
{
    int8_t result;
    char initial_reset[1];
    //This value when programmed to the Grid Eye's RESET reg will perform initial reset on device
    initial_reset[0] = 0x3F;
    result = this->gridEyeWriteReg(GridEye::RESET, 1, initial_reset);
    if (result == I2C_WR_SUCCESS)
        return result;
    return I2C_WR_ERROR;
}

//*********************************************************************
int8_t GridEye::setOperatingMode(GridEye::OperatingMode mode) 
{
    int8_t result;
    char set_mode[1];
    set_mode[0] = (char)mode;
    result = this->gridEyeWriteReg(GridEye::OPERATING_MODE, 1, set_mode);
    if (result == I2C_WR_SUCCESS)
        return result;
    return I2C_WR_ERROR;
}
        
//*********************************************************************
int8_t GridEye::setFrameRate(GridEye::FrameRate rate) 
{
    int8_t result;
    char set_rate[1];
    set_rate[0] = (char)rate;
    result = this->gridEyeWriteReg(GridEye::FRAME_RATE, 1, set_rate);
    if (result == I2C_WR_SUCCESS)
        return result;
    return I2C_WR_ERROR;
}

//*********************************************************************
void convSingleRawTempData2Int(char * data, int16_t & pixel_temp) 
{
    int8_t upper_byte = data[1];
    int8_t lower_byte = data[0];
    int16_t upper_byte_mask = 0x0F00;
    //int16_t sign_bit = 0x0400;
    int16_t sign_bit = 0x0200;
    int16_t finish_neg_val = 0xFC00;
    int16_t pixel;
    
    //construct the pixel based  off the 12 bit signed  data
    pixel = (upper_byte << 8);
    pixel &= upper_byte_mask;
    pixel |= lower_byte;
    
    //shift it over to gain integer value of the pixel
    pixel = pixel >> 2;
    
    //if negative, properly convert to 16 bit int format to represent 2's compliment
    if (pixel&sign_bit)
        pixel |= finish_neg_val;
    
    //set the coresponding pixel to be in the passed in array
    pixel_temp = pixel;
}

//*********************************************************************
void convRaw8x8Data2Int(char * data, int16_t * frame_temp) {
    int idx = 0;
    int8_t upper_byte;
    int8_t lower_byte;
    int16_t upper_byte_mask = 0x0F00;
    //int16_t sign_bit = 0x0400;
    int16_t sign_bit = 0x0200;
    int16_t finish_neg_val = 0xFC00;
    int16_t pixel;
    
    for (idx=0; idx<64; idx++) 
    {
        //construct the pixel based  off the 12 bit signed  data
        upper_byte = data[idx*2+1];
        lower_byte = data[idx*2+0];
        pixel = (upper_byte << 8);
        pixel &= upper_byte_mask;
        pixel |= lower_byte;
        
        //shift it over to gain integer value of the pixel
        pixel = pixel >> 2;
        
        //if negative, properly convert to 16 bit int format to represent 2's compliment
        if (pixel&sign_bit)
            pixel |= finish_neg_val;
        
        //set the coresponding pixel to be in the passed in array
        frame_temp[idx] = pixel;
    }
}

//*********************************************************************
void convRaw8x8Data2Point25degC(char * data, int16_t * frame_temp) 
{
    int idx = 0;
    int8_t upper_byte;
    int8_t lower_byte;
    int16_t upper_byte_mask = 0x0F00;
    //int16_t sign_bit = 0x1000;
    int16_t sign_bit = 0x0800;
    int16_t finish_neg_val = 0xF000;
    int16_t pixel;
    
    for (idx=0; idx<64; idx++) 
    {
        //construct the pixel based  off the 12 bit signed  data
        upper_byte = data[idx*2+1];
        lower_byte = data[idx*2+0];
        pixel = (upper_byte << 8);
        //pixel &= upper_byte_mask;
        pixel |= lower_byte;
        //no shift needed since we would lose the two lsb that give 0.25*C precision
        
        //if negative, properly convert to 16 bit int format to represent 2's compliment
        if (pixel & sign_bit)
            pixel |= finish_neg_val;
        else 
            pixel &= 0x0FFF;
        
        //set the coresponding pixel to be in the passed in array
        frame_temp[idx] = pixel;
    }
}
