/**********************************************************************
* Copyright (C) 2016 Maxim Integrated Products, Inc., All Rights Reserved.
*
* Permission is hereby granted, free of charge, to any person obtaining a
* copy of this software and associated documentation files (the "Software"),
* to deal in the Software without restriction, including without limitation
* the rights to use, copy, modify, merge, publish, distribute, sublicense,
* and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included
* in all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
* OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
* IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
* OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
* ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
* OTHER DEALINGS IN THE SOFTWARE.
*
* Except as contained in this notice, the name of Maxim Integrated
* Products, Inc. shall not be used except as stated in the Maxim Integrated
* Products, Inc. Branding Policy.
*
* The mere transfer of this software does not imply any licenses
* of trade secrets, proprietary technology, copyrights, patents,
* trademarks, maskwork rights, or any other form of intellectual
* property whatsoever. Maxim Integrated Products, Inc. retains all
* ownership rights.
**********************************************************************/


#ifndef GRIDEYE_H
#define GRIDEYE_H


#include "mbed.h"

#define I2C_WR_SUCCESS      0
#define I2C_WR_ERROR        -1

/**
* @brief Object for interfacing to MAXREFDES131#
*
* @details MAXREFDES131# combines the DS28E17 1-wire to I2C bridge
* with the Panasonic AMG8833 GridEye sensor.  The reference design also
* includes a DS2413 2ch open drain switch for controlling the MAX4717 
* dual SPDT analog switch.  The DS28E17 and AMG8833 are connected to 
* the 1-wire bus via COM2 of the MAX4717 and COM1 is used for 
* daisy-chaining additional modules.  Disconnecting the DS28E17/AMG8833
* from the main 1-wire branch puts both devices to sleep and reduces current
* consumption from 10mA to a couple hundred uA.
*/
class GridEye
{
    private:
    I2C &m_i2cBus;
    static const uint8_t I2C_ADRS = 0x68;
    static const uint8_t I2C_W_ADRS_GRIDEYE = (uint8_t)((I2C_ADRS)<<1);
    static const uint8_t I2C_R_ADRS_GRIDEYE = (uint8_t)((I2C_ADRS)<<1)+1;   
    
    public:
    
    /**
    * @brief GridEye operating modes
    */
    enum OperatingMode
    {
        NormalMode = 0x00,
        SleepMode = 0x10,
        StandBy60sec = 0x20,
        StandBy10sec = 0x21
    };
    
    /**
    * @brief GridEye operating modes
    */
    enum FrameRate
    {
        TenFPS = 0x00,
        OneFPS = 0x01
    };
    
    /**
    * @brief AMG8833 register map
    */
    enum GridEyeRegister
    {
        OPERATING_MODE,
        RESET,
        FRAME_RATE,
        INT_CONTROL,
        STATUS,
        STATUS_CLEAR,
        AVERAGE = 7,
        INT_LEVEL_1,
        INT_LEVEL_2,
        INT_LEVEL_3,
        INT_LEVEL_4,
        INT_LEVEL_5,
        INT_LEVEL_6,
        THERMISTOR_LOW,
        THERMISTOR_HI,
        INT_1,
        INT_2,
        INT_3,
        INT_4,
        INT_5,
        INT_6,
        INT_7,
        INT_8,
        PIXEL_BASE_ADRS = 0x80
    };
    
/*    static const uint8_t DS2413_FAMILY_CODE = 0x3A;
    
    static const uint8_t DS28E17_FAMILY_CODE = 0x19;
*/    
    /**
    * @brief GridEye Constructor
    *
    * @details setI2CBridge() by passing in I2C parameter
    *
    * @param[in] selector - MultidropRomIterator object that 
    * encapsulates ROM fxs of 1-wire protocol
    */
    GridEye(I2C &i2c);
    
    

    
    /**
    * @brief gridEyeWriteReg
    *
    * @details Provides read/write access to the AMG8833
    *
    * On Entry:
    * @param[in] regAdrs - AMG8833 register to start reading/writting 
    * from/to
    * @param[in] numBytes - Number of bytes of data to write 
    * @param[in] dataBuf - Pointer to data buffer for storing data in 
    * on read, or data to be written on write
    *
    * On Exit:
    *
    * @return CmdResult - return 0 if the operation was sucessful, return -1 if unsucessful
    */
    int8_t gridEyeWriteReg(GridEyeRegister reg_addr, int num_bytes, char * data_buf);
    
    /**
    * @brief gridEyeReadReg
    *
    * @details Provides read/write access to the AMG8833
    *
    * On Entry:
    * @param[in] regAdrs - AMG8833 register to start reading/writting 
    * from/to
    * @param[in] numBytes - Number of bytes to read
    * @param[in] dataBuf - Pointer to data buffer for storing data in 
    * on read, or data to be written on write
    *
    * On Exit:
    * @param[out] dataBuf - Read data on read operation
    *
    * @return CmdResult - return 0 if the operation was sucessful, return -1 if unsucessful
    */
    int8_t gridEyeReadReg(GridEyeRegister reg_addr, int num_bytes, char * data_buf);
    
    
    /**
    * @brief getThermistorTemperature
    *
    * @details Gets internal thermistor temperature 
    *
    * On Entry:
    * @param[in] thermTemp - reference to int16_t var that will be overwritten
    * with thermistor data
    *
    * On Exit:
    * @param[out] thermTemp - thermistor data
    *
    * @return CmdResult - result of operation
    */
    int8_t getThermistorTemperature(int16_t & therm_temp);
    
    
    /**
    * @brief getPixelTemperature
    *
    * @details Gets individual pixel temperature 
    *
    * On Entry:
    * @param[in] pixelAdrs - address of pixel to read
    * @param[in] pixelTemp - reference to int16_t var that will be overwritten
    * with thermistor data
    *
    * On Exit:
    * @param[out] pixelTemp - pixel data
    *
    * @return CmdResult - result of operation
    */
    int8_t getPixelTemperature(uint8_t pixel_addr, int16_t & pixel_temp);
    
    /**
    * @brief getRaw8x8FrameData
    *
    * @details Gets pixel frame(64 pixels in 128 bytes) temperature 
    *
    * On Entry:
    * @param[in] raw_frame_data - buffer to hold 2 bytes data per pixel
    *
    * On Exit:
    * @param[out] raw_frame_data - pixel data not yet formatted
    *
    * @return int8_t - result of operation, 0 on success, -1 on failure
    */   
    int8_t getRaw8x8FrameData(char * raw_frame_data);
    
    
    /**
    * @brief softwareReset
    *
    * @details Makes the Grid Eye device perform a software restart with Initial conditions
    *
    * @return int8_t - result of operation, 0 on success, -1 on failure
    */   
    int8_t softwareReset();
    
    
    /**
    * @brief setOperatingMode
    *
    * @details Sets the mode of operation for the Grid Eye device 
    *
    * On Entry:
    * @param[in] mode - Choose power operating mode from struct
    *
    * @return int8_t - result of operation, 0 on success, -1 on failure
    */   
    int8_t setOperatingMode(GridEye::OperatingMode mode);
    
    
    /**
    * @brief setFrameRate
    *
    * @details Sets the frames per second that the grid eye device resolves
    *
    * On Entry:
    * @param[in] mode - Choose option from either 10 fps or 1 fps
    *
    * @return int8_t - result of operation, 0 on success, -1 on failure
    */   
    int8_t setFrameRate(GridEye::FrameRate rate);

};


void convRaw8x8Data2Int(char * data, int16_t * frame_temp);
    
void convRaw8x8Data2Point25degC(char * data, int16_t * frame_temp);
    
void convSingleRawTempData2Int(char * data, int16_t & pixel_temp);

#endif /*GridEye_H*/