#include <mbed.h>

#define PCF8583_addr_1      0xA0     // PCF8583 Write address,
#define PCF8583_addr_2      0xA2     // PCF8583 Write address,
#define ON                  1
#define OFF                 0

//Configuration Alarm Control Register
#define alarm_enable_bit    0x04
#define alarm_flag          0x02;
#define timer_hundsek       0x01
#define timer_seconds       0x02
#define timer_minutes       0x03
#define timer_hours         0x04
#define timer_days          0x05
#define timer_int_enable    0x08
#define daily_alarm         0x10
#define weekday_alarm       0x20
#define dated_alarm         0x30
#define timer_alarm_enable  0x40
#define alarm_int_enable    0x80
#define timer_alarm_enable  0x40
#define alarm_int_enable    0x80

// Use the first 2 NVRAM addresses for the century and year bytes.
#define CENTURY_REG         0x10
#define YEAR_REG            0x11
#define USER_REG            0x12

#define TIME                1
#define ALARM               9

/// Structures
struct Time_t {
  bool fmt_hours;
  bool am_pm_flag;
  char hours;
  char minutes;
  char seconds;
  char hundreds;
};

struct Date_t {
  char day;
  char month;
  char year;
  char century;
  char weekday;
};

struct DateTime_t {
  struct Date_t date;
  struct Time_t time;
};

class PCF8583rtc {
  I2C *_i2c;
  
public:

/**
 * Set these public variables according to your locale, default (in brackets) is Australian English.
 *  char *ShortDateFormat;  ("d,m,yy")
 *  char *LongDateFormat;   ("dddd dd mmmm yyyy")
 *  char *ShortTimeFormat;  ("d:m:yy")
 *  char *LongTimeFormat;   ("dd:nn:yyyy")
 *
 *  char DateSeparator;     ("\")
 *  char TimeSeparator;     (":")
 *
 *  char *ShortDayNames[7];    ({"Mon", "Tue", "Wed", "Thu", "Fri", "Sat", "Sun"})
 *  char *LongDayNames[7];     ({"Monday", "Tuesday", "Wednesday", "Thursday", "Friday", "Saturday", "Sunday"})
 *  char *ShortMonthNames[12]; ({"Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"})
 *  char *LongMonthNames[12];  ({"January", "February", "March", "April", "May", "June", "July", "August", "September", "October", "November", "December"})
 *
* Example:
* @code
* #include <mbed.h>
* #include "PCF8583_rtc.h"
*
* I2C i2c(P0_10, P0_11);       // sda, scl
* PCF8583rtc rtc(&i2c, PCF8583_addr_2);
* 
*    struct DateTime_t dtl;
*
*    //Set the time
*    dtl = rtc.read(TIME);
*    dtl.time.hours = rtc.bin2bcd(11);
*    dtl.time.minutes = rtc.bin2bcd(43);
*    rtc.write(TIME, dtl);
*
*    //Read and display the time on the nixie display
*    dtl = rtc.read(TIME);
*    i2c.write(ADDR_8574_1, &dtl.time.hours, 1);
*    i2c.write(ADDR_8574_2, &dtl.time.minutes, 1);
*
* @endcode

  PCF8583rtc(I2C *i2c, char I2cAddress);

  DateTime_t read(const char address);
  void   write(const char address, DateTime_t dti);
  void   FormatDateTime(char *dest, char *format);
  bool   WriteNVram(char address, char * value, char num);
  bool   ReadNVram(char address, char * dest, char num);
  void   SetDateTime(DateTime_t dti);
  struct DateTime_t GetDateTimeBCD(void);

*/

  char *ShortDateFormat;
  char *LongDateFormat;
  char *ShortTimeFormat;
  char *LongTimeFormat;

  char DateSeparator;
  char TimeSeparator;

  char *ShortDayNames[7];
  char *LongDayNames[7];
  char *ShortMonthNames[12];
  char *LongMonthNames[12];
    
  PCF8583rtc(I2C *i2c, char I2cAddress);

/** read the current Time or Alarm
*
* @param address "TIME" = read the time, "ALARM" = read the alarm
* @returns
*   a DateTime_t structure
*/
  DateTime_t read(const char address);

/** write the current Time or Alarm
*
* @param address "TIME" = read the time, "ALARM" = read the alarm 
* @param dti a DateTime_t structure containing the time or alarm data 
*/
  void write(const char address, DateTime_t dti);

  void FormatDateTime(char *dest, char *format);

  bool WriteNVram(char address, char * value, char num);

  bool ReadNVram(char address, char * dest, char num);

  void SetDateTime(DateTime_t dti);

  char bin2bcd(char value);
  
  void configureAlarmReg(char alarm);
  void configureControlReg(char control);

private:
  struct DateTime_t dt;
  char   _I2cAddress;

  char Bcd2Char(char *d, char val, char WantLeadZero);
  void enableCounting(void);
  void pauseCounting(void);
  char readByte(char address);
  void writeByte(char address, char d);
};

/** PCF8583_rtc class.
 *
 * Example:
 * @code
 * #include <mbed.h>
 * #include <PCF8583_rtc.h>
 *
 * I2C i2c(P0_10, P0_11);       // sda, scl
 * PCF8583rtc rtc(&i2c);
 *
 * int main() { 
 *  
 * rtc.read(TIME);
 * i2c.write(ADDR_8574_1, &rtc.HoursBCD, 1);   //write hours to display
 * i2c.write(ADDR_8574_2, &rtc.MinsBCD, 1);    //write minutes to display
 * }
 * @endcode
 */
 
/** Create a PCF8583rtc object using a pointer to the given I2C object.
 *
 * @param i2c pointer to an I2C object to which the PCF8583 device is connected.
 */
//PCF8583rtc(I2C *i2c);

/** the Read_DateTime function.
 *
 * @param address  Determines whether to read the date/time or alarm registers
 *                 Values are retrieved from the internal datetime structure
 */
//void read(const char address);

/** the Write_DateTime function.
 *
 * @param address  Determines whether to set the date/time or alarm registers
 *                 Values are first set into the internal datetime structure prior to calling
 */
//void write(const char address);
 
/** the FormatDateTime function.
 *
 * @param dest   a pointer to a char array where formatted values are to be placed
 * @param format a pointer to a string describing how to format the values 
 */
//void FormatDateTime(char *dest, char *format);

/** the WriteNVram function.
 *
 * @param address  address where value is to be placed
 * @param value    pointer to char array containing the value(s) to be stored
 * @param num      the number bytes to store
 */
//void WriteNVram(char address, char *value, char num);

//#endif