#ifndef MTK3339_H
#define MTK3339_H

/**
 * An interface to the MTK3339 GPS module.
 */
class MTK3339
{
public:


    enum NmeaSentence {
        NmeaInvalid = 0,
        NmeaGga = 0x01,
//        NmeaGsa = 0x02,
//        NmeaGsv = 0x04,
//        NmeaRmc = 0x08,
        NmeaVtg = 0x10
    };

    struct GgaType {
        /** UTC time - hours */
        int hours;
        /** UTC time - minutes */
        int minutes;
        /** UTC time - seconds */
        int seconds;
        /** UTC time - milliseconds */
        int milliseconds;

        /** The latitude in ddmm.mmmm format (d = degrees, m = minutes) */
        double latitude;
        /** The longitude in dddmm.mmmm format */
        double longitude;
        /** North / South indicator */
        char nsIndicator;
        /** East / West indicator */
        char ewIndicator;

        /**
         * Position indicator:
         * 0 = Fix not available
         * 1 = GPS fix
         * 2 = Differential GPS fix
         */
        int fix;

        /** Number of used satellites */
        int satellites;
        /** Horizontal Dilution of Precision */
        double hdop;
        /** antenna altitude above/below mean sea-level */
        double altitude;
        /** geoidal separation */
        double geoidal;
    };

    struct VtgType {
        /** heading in degrees */
        double course;
        /** speed in Knots */
        double speedKnots;
        /** Speed in kilometer  per hour */
        double speedKmHour;
        /**
         * Mode
         * A = Autonomous mode
         * D = Differential mode
         * E = Estimated mode
         */
        char mode;
    };

    /**
     * Create an interface to the MTK3339 GPS module
     *
     * @param tx UART TX line pin
     * @param rx UART RX line pin
     */
    MTK3339(PinName tx, PinName rx);

    /**
     * Start to read data from the GPS module.
     *
     * @param fptr A pointer to a void function that will be called when there
     * is data available.
     * @param mask specifies which sentence types (NmeaSentence) that are of
     * interest. The callback function will only be called for messages
     * specified in this mask.
     */
    void start(void (*fptr)(void), int mask);

    /**
     * Start to read data from the GPS module.
     *
     * @param tptr pointer to the object to call the member function on
     * @param mptr pointer to the member function to be called
     * @param mask specifies which sentence types (NmeaSentence) that are of
     * interest. The member function will only be called for messages
     * specified in this mask.
     */
    template<typename T>
    void start(T* tptr, void (T::*mptr)(void), int mask) {
        if((mptr != NULL) && (tptr != NULL) && mask) {
            _dataCallback.attach(tptr, mptr);
            _sentenceMask = mask;
            _serial.attach(this, &MTK3339::uartIrq, Serial::RxIrq);
        }
    }

    /**
     * Stop to read data from GPS module
     */
    void stop();

    /**
     * Get the type of the data reported in available data callback.
     * This method will only return a valid type when called within the
     * callback.
     */
    NmeaSentence getAvailableDataType();

    /**
     * Get latitude in degrees (decimal format)
     */
    double getLatitudeAsDegrees();

    /**
     * Get longitude in degrees (decimal format)
     */
    double getLongitudeAsDegrees();

    /**
     * Time, position and fix related data
     */
    GgaType gga;

    /**
     * Course and speed information relative to ground
     */
    VtgType vtg;


private:

    enum PrivConstants {
        MTK3339_BUF_SZ = 255
    };

    enum DataState {
        StateStart = 0,
        StateData
    };

    FunctionPointer _dataCallback;
    char _buf[MTK3339_BUF_SZ];
    int _bufPos;
    DataState _state;
    int _sentenceMask;
    NmeaSentence _availDataType;

    Serial _serial;


    void parseGGA(char* data, int dataLen);
    void parseVTG(char* data, int dataLen);
    void parseData(char* data, int len);
    void uartIrq();


};

#endif

