/**
* @author Damien Frost
*
* @section LICENSE
*
*   Copyright (c) 2016 Damien Frost
*
*   Permission is hereby granted, free of charge, to any person obtaining a copy
*   of this software and associated documentation files (the "Software"), to deal
*   in the Software without restriction, including without limitation the rights
*   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
*   copies of the Software, and to permit persons to whom the Software is
*   furnished to do so, subject to the following conditions:
*
*   The above copyright notice and this permission notice shall be included in
*   all copies or substantial portions of the Software.
*
*   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
*   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
*   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
*   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
*   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
*   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
*   THE SOFTWARE.
*
* @file "messages.h"
*
* @section DESCRIPTION
*   Definitions for various messaging levels. See the example, below for usage.
*
* @section EXAMPLE
*   1) Add this library to your project.
*   2) Include the file as a regular .h file in *.cpp files only. This will allow
*      you to have different messages coming from different files.
*
*   Ex: main.cpp:
*
*   #include "mbed.h"
*   
*   // Include the following #defines and #include to use messages.h in every *.cpp file you would like to use it in:    
*   //#define MEMMESSAGES       // Uncomment this to use the MEM(...) message. This message will output the current values of the stack pointer and program counter
*   //#define DEBUG             // This define is commented out, so DEBUG messages will NOT be displayed.
*   #define INFOMESSAGES        // The next three are defined, thus INFO, WARN, and ERR messages will BE displayed.
*   #define WARNMESSAGES
*   #define ERRMESSAGES
*   #define FUNCNAME "MAIN"     // Set this to an intelligible string. Usually set to something to do with the filename.
*   #include "messages.h"       // Finally include the messages.h header.

*   // Main Loop!
*   int main() {
*       INFO("CPU SystemCoreClock is %d Hz", SystemCoreClock);
*       DBG("Testing %d, %.3f, %d", 1, 2.0f, 3);
*   }
*
*   
*
*/

#include "mbed.h"

#ifndef MESSAGES_H
#define MESSAGES_H

#ifdef DEBUG
#define DBG(serPort, x, ...) serPort->printf("["FUNCNAME" : DBG] "x" <line %d>\r\n", ##__VA_ARGS__,__LINE__);
#else
#define DBG(x, ...)
#endif

#ifdef ERRMESSAGES
#define ERR(serPort, x, ...) serPort->printf("["FUNCNAME" : ERR] "x" <line %d>\r\n", ##__VA_ARGS__,__LINE__);
#else
#define ERR(x, ...)
#endif

#ifdef WARNMESSAGES
#define WARN(serPort, x, ...) serPort->printf("["FUNCNAME" : WARN] "x" <line %d>\r\n", ##__VA_ARGS__,__LINE__);
#else
#define WARN(x, ...)
#endif

#ifdef INFOMESSAGES
#define INFO(serPort, x, ...) serPort->printf("["FUNCNAME" : INFO] "x" <line %d>\r\n", ##__VA_ARGS__,__LINE__);
#else
#define INFO(x, ...)
#endif

#ifdef MEMMESSAGES
//#define MEM(serPort, x, ...) serPort->printf("["FUNCNAME" : MEM] "x"\r\n", ##__VA_ARGS__);
#define MEM(serPort) serPort->printf("["FUNCNAME" : MEM] SP: %d, PC: %d\r\n", __current_pc(), __current_sp());
#else
#define MEM(x, ...)
#endif

#endif /* MESSAGES_H */