/**
 *@section DESCRIPTION
 * mbed SolarNanogrid  Library
 *@section LICENSE
 * Copyright (c) 2016, Malcolm McCulloch
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 * @file "SolarNanoGrid.c"
 */
#include <mbed.h>
#include "SolarNanoGrid.h"
#define FUNCNAME "SNGC"
#include "defs.h"
// Constructor:
/**
 * Constructor.
 */
SolarNanoGrid::SolarNanoGrid(FILE *fp=NULL, Serial *pc=NULL):pc(pc),_fp(fp){
    // Save the sd card pointer:
    DBG("Solar init");
    // Leds
    ledRed = new DigitalOut(LED_RED,1);
    ledGreen= new DigitalOut(LED_RED,1); // The converter uses the saem DIO as the green led as an interrupt in.
    ledBlue = new DigitalOut(LED_BLUE,1);
    // Realtime clock
    time_t now= time(NULL);
    struct tm * timeInf = localtime(&now);
    INFO ("Time is now: %04d-%02d-%02d %02d:%02d:%02d \n\r",timeInf->tm_year+1900,timeInf->tm_mon+1,timeInf->tm_mday,timeInf->tm_hour, timeInf->tm_min,timeInf->tm_sec);

    // Read config.ini
    if (fscanf (fp,"%d %*c %*s",&sdVersion )!=1) ERR("Config: cannot read version");
    if (fscanf (fp,"%u %*c %*s",&communityID  )!=1) ERR("Config: cannot read community ID");
    if (fscanf (fp,"%x %*c %*s",&id)!=1) ERR("Config: cannot read ID");
    if (fscanf (fp,"%x %*c %*s",&chan )!=1) ERR("Locker config: cannot read channel");

    INFO("config.ini:  Version %u, Community %u ID %x Channel %x",sdVersion,communityID,id,chan);
    spiSD();
    mkdir("/sd/data", 777);
    // Initialize nrf
    ce = new DigitalOut(PTB20);
    nrf = new NRF2401P(PTD6, PTD7, PTD5, PTD4, PTB20);       // Rev E
    //nrf1 = new NRF2401P(PTD6,PTD7, PTD5,PTD4, PTC12); //mosi, miso, sclk, csn, ce) REV D // irq ptc18 on k64f


}

/**
 * asks the user for the time
 */

void SolarNanoGrid::userSetRTCpc()
{
    // get the current time from the terminal
    struct tm t;
    printf("Enter current date :\n\r");
    printf("YYYY MM DD [enter]\n\r");
    scanf("%d %d %d", &t.tm_year, &t.tm_mon, &t.tm_mday);
    printf("Enter current time:\n\r");
    printf("HH MM SS [enter]\n\r");
    scanf("%d %d %d", &t.tm_hour, &t.tm_min, &t.tm_sec);

    // adjust for tm structure required values
    t.tm_year = t.tm_year - 1900;
    t.tm_mon = t.tm_mon - 1;

    // set the time
    set_time(mktime(&t));

}
/**
 * Flushes the rx and tx buffers and resets the status.
 */
void SolarNanoGrid::nrfFlush() {
    spiNRF();
    if(nrf!=NULL){
        nrf->clearStatus();
        nrf->flushRx();
        nrf->flushTx();
        nrf->clearStatus();
    }
}

/**
 * Turns SPI on for SD card
 */
void spiSD(void)
{
    //sd.select();
    pin_function(PTE1 , 7); //Set SD_MISO as SPI, this is the same as the last number in those tables
    pin_function(PTD7, 1);  //pin function 1 is GPIO
    return;
}

/**
 * Turns SPI on for nrf
 */
void spiNRF(void)
{
    //sd.deselect();
    pin_function(PTE1, 1); //pin function 1 is GPIO
    pin_function(PTD7, 7); //Set SD_MISO as SPI, this is the same as the last number in those tables
    return;
}

/**
 * Set NRF as RX and flush
 */
void SolarNanoGrid::setAsRX(long long addr){

    spiNRF();
    nrf->quickRxSetup(chan, addr); // Pipe 0
    nrf->setRadio(0x01, 0x03); // 2MB/S  0dB
    nrfFlush();
}
/**
 * loop

void SolarNanoGrid::loop(){
    ;
    }

 * Sets NRF as TX and flushes the NRF
 */
void  SolarNanoGrid::setAsTX(long long addr){
    DBG("setAsTx");
    spiNRF();
    nrf->quickTxSetup(chan, addr);
    nrf->setRadio(0x01, 0x03); // 2MB/S  0dB
    nrf->setTxRetry(0x0F, 0x0F);
    nrfFlush();
    DBG("Nrf Details:");
#ifdef DEBUG
    nrf->printDetails();
#endif

}

SongSettings::SongSettings(void){
    // Just set everything to null:
    role = NULL;
    sdVersion = NULL;
    communityID = NULL;
    id = NULL;
    chan = NULL;
    return;
}

bool SongSettings::LoadSongSettings(void){
    
    char * name = "/sd/config.ini";
    bool result = true;
    INFO("Loading configuration from %s", name);
    FILE *fp;
    INFO("Loading config file into settings...");
    spiSD();
    fp = fopen(name, "r");
    if(fp == NULL) {
        ERR("Config file cannot be read, USE PC TO CREATE config.ini");
        return false;
    }else{
        DBG("Reading values...");
        // Read config.ini
        // Role:
        DBG("Role:");
        if (fscanf (fp,"%c %*c %*s",&role )!=1){
            ERR("Config: cannot read role");
            result = false;
        }else{
            DBG("Read Role: %c", role);
        }
        // SD Version:
        DBG("SD Version:");
        if (fscanf (fp,"%d %*c %*s",&sdVersion )!=1){
            ERR("Config: cannot read version");
            result = false;
        }else{
            DBG("Read sdVersion: %d", sdVersion);
        }
        // Community ID:
        DBG("Community ID:");
        if (fscanf (fp,"%u %*c %*s",&communityID  )!=1){
            ERR("Config: cannot read community ID");
            result = false;
        }else{
            DBG("Read community ID: %u", communityID);
        }
        // ID:
        DBG("ID:");
        if (fscanf (fp,"%x %*c %*s",&id)!=1){
            ERR("Config: cannot read ID");
            result = false;
        }else{
            DBG("Read read ID: %x", id);
        }
        // Channel:
        DBG("Channel:");
        if (fscanf (fp,"%x %*c %*s",&chan )!=1){
            ERR("Locker config: cannot read channel");
            result = false;
        }else{
            DBG("Read channel: %x", chan);
        }
        
        if(result == true){
            // create the log directory string, this does NOT create the directory!
            sprintf(LogDir, "/sd/data/%02X%04X", (communityID&0XFF),(id&0XFFFF));
            // Print everything back:
            INFO("config.ini:  Version %u, Community %u ID %x Channel %x, Log Directory: %s",sdVersion, communityID, id, chan, LogDir);
        }else{
            ERR("Some of the settings were not loaded correctly. Perhaps %s is corrupted.", name);
        }        
        return result;
    }
    
}


void doNewConfig(Serial *pc)
{
//  H : Role
//  300 : Version
//  2540001 : Community_ID
//  FF00 : ID
//  5a : Channel

    char * name = "/sd/config.ini";
    char buffer[32];
    char CommunityIDbuff[32];
    char IDbuff[32];
    char channelbuff[32];
    char Role;
    FILE *fp;
    INFO("");
    INFO("***********************************");
    INFO("*** Creating a new config file. ***");
    INFO("***********************************");
    INFO("");
    fp = fopen(name, "w");
    //Empty buffer
    while (pc->readable()) {
        pc->getc();
    }
    
    char resp='k';
    
    do {
        INFO("Enter Role (B-Battery, H-Hmi, L-Locker, U-Utility, X-Exit:");
        buffer[0] = 'k';
        while(1){
            // Get a user input:
            pc->scanf("%c",buffer);
            // Check the input:
            if ((buffer[0]=='X') ||(buffer[0]=='x')) {
                INFO("Exiting...");
                return;
            }
            if((buffer[0]=='B')||(buffer[0]=='b')||(buffer[0]=='H')||(buffer[0]=='h')||(buffer[0]=='L')||(buffer[0]=='l')||(buffer[0]=='U')||(buffer[0]=='u')){
                break;
            }
            // If we get here, the input sucks:
            INFO("Invalid selection, please enter the Role (B-Battery, H-Hmi, L-Locker, U-Utility, X-Exit:");
        }
        // Save the result:
        Role = buffer[0];
        INFO("Role = %c, selected.", Role);
        INFO("");
        
        // Get the Community ID:
        INFO("Enter community ID (Decimal int: e.g. 2540001):");
        pc->scanf("%s",CommunityIDbuff);
        INFO("Community ID= %s",CommunityIDbuff);
        INFO("");
        
        // Get the ID:
        INFO("Enter ID: (Hex: Locker (2 Bytes) Pod(1 Byte) Battery number (1Byte) e.g 01a1)");
        pc->scanf("%s",IDbuff);
        INFO("Locker ID= %s",IDbuff);
        INFO("");
        
        // Get the Channel:
        INFO("Enter Channel: (default: 5a)");
        pc->scanf("%s",channelbuff);
        INFO("Channel %s",channelbuff);
        INFO("");
        
        // Display the results:
        INFO("The new configuration file is:");
        INFO("");
        INFO("%c : Role",Role);
        INFO("300 : Version");
        INFO("%s : CommunityID",CommunityIDbuff);
        INFO("%s : ID",IDbuff);
        INFO("%s : channel",channelbuff);
        INFO("");
        //Save the result to a file:
        INFO("Saving the new file to: %s...", name);
        fp = fopen(name, "w");
        fprintf (fp,"%c : Role\r\n",Role);
        fprintf (fp,"300 : Version\r\n");
        fprintf (fp,"%s : CommunityID\r\n",CommunityIDbuff);
        fprintf (fp,"%s : ID\r\n",IDbuff);
        fprintf (fp,"%s : channel\r\n",channelbuff);
        fclose(fp);
        INFO("Config file updated.");
        
        // Ask if we want to continue:
        INFO("OK? (y/n)");
        pc->scanf("%s",&buffer);
        resp = buffer[0];
        INFO("%c", resp);
        
    } while(resp!='y' and resp!='Y');
    
    INFO("Exiting...");
    INFO("");
    return;
}
    