/**
 *@section DESCRIPTION
 * mbed SolarNanogrid  Library
 * Hmi extends SolarNanoGrid.
 * Hmi does the human interface and talks to the lockers.
 * The  ID must be in FF 00.
 *@section LICENSE
 * Copyright (c) 2016, Malcolm McCulloch
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 * @file "Hmi.h"
 */

#ifndef HMI_H
#define HMI_H
#include "mbed.h"
//#include "NRF2401P.h"
//#include "nRF24l01.h"
#include "MFRC522.h"
#include "SPI_TFT_ILI9341.h"
#include "Arial24x23.h"
//#include "SDFileSystem.h"
#include "HubUser.h"
#include "HubLogging.h"
#include "SolarNanoGrid.h"




/** \brief Hmi manages the human interface and talks to the lockers.
 *
 * ## Config file
 *  The config.ini must start with H
 *  poshoInit.txt
 *
 * ## Hardware
 *  ### Buttons
 * - Button one interrupt (PTC3)
 * - Button two interrupt (PTC2)
 * - Button three interrupt (PTA2)
 * - Button four interrupt (PTB23)
 *
 *  ### RFID (SPI_MOSI, SPI_MISO, SPI_SCLK, SPI_CS, MF_RESET)
 *  - Battery reader : (PTD2, PTD3, PTD1, PTB2, PTB3)
 *  - Posho reader : (PTD2, PTD3, PTD1, PTB10, PTB11);
 *  - Incubator : (PTD2, PTD3, PTD1, PTC11, PTC10)
 *
 *  ### Screen SPI_TFT_ILI9341
 *  - (PTD2, PTD3, PTD1, PTE25, PTB20, PTB18,"TFT")
 *
 */
class Hmi : public SolarNanoGrid
{
public:

    // *********************
    // * Public variables: *
    // *********************


    // *********************
    // * Public functions: *
    // *********************
    /**
     * Constructor
     * param: fp is the link to the config file.
     */
    Hmi(FILE* fp, Serial *pc);
    /**
     * Loops through slow routines
     */

    void loopHub();
    
    /**
     * calls loopHub()
     */
    virtual void loop();


protected:
    // *********************
    // * Protected hardware: *
    // *********************


    // RFID Readers MFRC522 RfChip (SPI_MOSI, SPI_MISO, SPI_SCLK, SPI_CS, MF_RESET);
    MFRC522 *BatteryRfChipH;                    /**< Battery Rfid */
    MFRC522 *PoshoRfChipH;                      /**< Posho Rfid */
    MFRC522 *IncubatorRfChipH;                  /**< Incubator Rfid */

    InterruptIn *buttonOneH;                    /**< Button one interrupt (PTC3) */
    InterruptIn *buttonTwoH;                    /**< Button two interrupt (PTC2)*/
    InterruptIn *buttonThreeH;                  /**< Button three interrupt (PTA2) */
    InterruptIn *buttonFourH;                   /**< Button four interrupt (PTB23)*/


    SPI_TFT_ILI9341 *TFT_H;                     /**< LCD Display */

    // Tickers
    Ticker *tick1sHub;                          /**< Every second time check */
    Ticker *tickBatteryRfidH;                   /**< Battery Rfid check */
    Ticker *tickIncubatorRfidH;                 /**< Incubator Rfid check */
    Ticker *tickPoshoRfidH;                     /**< Posho Rfid check */
    Ticker *tickTimeCheckH;                     /**< Time check for terminal display */
    Ticker *tickerUpdateUserTableH;             /**< Update user table to SD card ticker (every 30 min) */

    // *********************
    // * Protected flags: *
    // *********************

    char flagBatteryRfidH;      /**< Battery Rfid flag */
    char flagButtonOneH;                    /**< Button one flag */
    char flagButtonTwoH;                    /**< Button two flag */
    char flagButtonThreeH;                  /**< Button three flag */
    char flagButtonFourH;                   /**< Button four flag */

    char flagPoshoRfidH;        /**< Posho Rfid flag */
    char flagIncubatorRfidH;    /**< Incubator Rfid flag */
    char flagTimeCheckH;        /**< Time check flag */
    char flagUpdateUserTableH;  /**< Update user table to SD card ticker */
    unsigned char flag1sH;      /**< Every second flag */

    //State
    /*
     * The hub display screen can involve multiple stages that are enumerated here.
     * All user action waiting stages also listen for the user selecting to cancel/exit
     */
    enum HubScreenStageH {
        initialScanRfid,                    /**< Waiting for RFID with clearing and text display */
        waitForRfid,                        /**< Listening for all interrupts */
        batterySelectAction,                /**< Listening for either battery drop off or pick up */
        batterySelectNumberForPickup,       /**< Listening for number of batteries to drop off */
        batterySelectNumberForDropoff,      /**< Listening for number of batteries to pick up */
        poshoSelectKilograms                /**< Listening for number of kgs of material to posho */
    };


    enum HubScreenStageH currentScreenH ;   /**< Globally accessible screen selector */

    // *********************
    // * Protected variables: *
    // *********************

    float currentMaxH;              /**< Maximum current that each cube can consume. */

    HubUser *allUsersH;             /**< Array to be dynamically allocated storing all users */
    int userCountH;                 /**< Number of active users on this hub mbed */
    int currentUserH;               /**< Index of the current user */

    int poshoPricePerKgH;           /**< price per kg of material to be used with the posho */
    int timeForOneKgH;              /**< time for one kg of material to be used with the posho */
    int timeForTwoKgH;              /**< time for two kg of material to be used with the posho */
    int timeForThreeKgH;            /**< time for three kg of material to be used with the posho */

    Ticker tickPoshoInUseClearH;    /**< ticker to clear posho in use flag after a specific time */
    char flagPoshoAlreadyInUseH;

    long long baseAddr;             /**< Base address of community to tell for pickup or drop off*/
    long long lockerAddr;               /**< Address of locker to transmit to tell for pickup or drop off*/
    long long utilityAddr;              /**< Address of locker to transmit to tell for pickup or drop off*/

     // *********************
    // * Protected functions: *
    // *********************

    /**
     * Creates a new user by defining user attributes.
     * @param rfid - uint32_t corresponding to unique id of the RFID tag
     * @param accountCredit - int32 for user account balance (can be negative)
     * @param locker - int32_t for which locker (originaly 1-4) user is assigned
     * @param batterySubscription -  int32_t max batteries user can get (can be 0)
     * @param batteriesOut - int32_t number of batteries user has out (usually 0 initially)
     * @param pod - int32_t pod associated with the user (originally 0-15) *must be 0 indexed*
     */
    void createNewUserH(uint32_t rfid, int32_t accountCredit, int32_t locker,
            int32_t batterySubscription, int32_t batteriesOut, int32_t pod, char* name);
    /**
     *  Initialize Global User Table Variable with malloc
     *  @param users - int representing how many users you want the system to hold
     */
    void initializeGlobalUserTableH(int users);
    /**
     *  Initialize posho functionality
     */
    void initializePoshoFunctionalityH();
    /**
     * Initialize system with users from users.txt on SD card
     */
    void initializeUsersFromSD_H();


    /* ************************************************************************************************/
    /* Set a flag when an interrupt is detected */
    /* ************************************************************************************************/

    /**
     *  Time check interrupt
     */
    void interruptTimeCheckH();
    /**
     *  Update user table interrupt
     */
    void interruptUpdateUserTableH();
    /**
     *  Battery RFID reader interrupt
     */
    void interruptBatteryRfidH();
    /**
     *  Posho RFID reader interrupt
     */
    void interruptPoshoRfidH();
    /**
     *  Incubator RFID reader interrupt
     */
    void interruptIncubatorRfidH();
    /**
     *  button one interrupt
     */
    void interruptButtonOneH();
    /**
     *  button two interrupt
     */
    void interruptButtonTwoH();
    /**
     *  button three interrupt
     */
    void interruptButtonThreeH();
    /**
     *  button four interrupt
     */
    void interruptButtonFourH();
    /**
     * Fast interrupt routine for every 1 second
     */
    void int1sH();
    /**
     *  interrupt to clear the posho is in use flag after a designated time
     */
    void interruptPoshoInUseClearH();

    /*************************************************************************************************/
    /*
     *  Reset all user initiated flags. Useful after sequences of events where flags may have
     *  piled up from misc inputs and you don't want the system to act seemingly sporadically.
     */
    void clearAllUserInitiatedFlagsH();



    /*
     * User presses the button corresponding to an exit. Returns to home screen
     * after logging the action.
     * @param userIndex - int representing index of user in user table
     */
    void cancelPressedH(int userIndex);
    /**
     *  Do if time check flag is set
     *  reads the unix time, converts into human readable format, and displays on PC
     */
    void doTimeCheckH();
    /**
     *  Do if update user table flag is set
     */
    void doUpdateUserTableH();

    /* ************************************************************************************************/
    /* RFID support */
    /* ************************************************************************************************/
    /*
     * Read Rfid card and parse out the tag id into a single integer. Log if tag is either known
     * or unknown. Sets the global variable for the current user if a user is found.
     * @param rfidReader - MFRC522* of the RFID reader
     * @param source - HubLoggingRfidSource enum for which RFID reader (battery, posho, incubator)
     * @return int representing the userIndex of the HubUser found by the RFID tag. If no user is
            found, returns a -1.
     */
    int rfidReadHelper(MFRC522 *rfidReader, enum HubLoggingRfidSource source);
    /*************************************************************************************************/
    /* Battery Services */
    /*************************************************************************************************/
    /**
     *  Do if Battery RFID flag is set. Read RFID tag and check user status.
     *  If there's a valid battery subscription, display drop off and/or pick up.
     *  Otherwise, returns to initial screen.
     */
    void doBatteryRfidH();
    /**
     *  Do if user selects to pickup a battery. Determines how many batteries
     *  a user can pickup based off user info and displays corresponding text
     *  @param userIndex - int representing index of user in user table
     */
    void batteryPickUpScreenH(int userIndex);
    /**
     *  Do if user selects to dropoff a battery. Determines how many batteries
     *  a user can pickup based off user info and displays corresponding text
     *  @param userIndex - int representing index of user in user table
     */
    void batteryDropOffScreenH(int userIndex);

    /**
     *  Do after user selects number of batteries to drop off.
     *  Logs the action, changes user info, transmits instructions to other systems
     *  @param numBatteries - int for number of batteries selected to drop off
     *  @param userIndex - int representing index of user in user table
     */
    void batteryDropOffH(int numBatteries, int userIndex);
    /**
     *  Do after user selects number of batteries to pick up.
     *  Logs the action, changes user info, transmits instructions to other systems
     *  @param numBatteries - int for number of batteries selected to pick up
     *  @param userIndex - int representing index of user in user table
     */
    void batteryPickUpH(int numBatteries, int userIndex);

    /* ************************************************************************************************/
    /* Misc Services */
    /* ************************************************************************************************/
    /**
     * Do if Posho RFID flag is set. Reads rfid and checks user table for a positive balance
     * capable of paying for the posho. Displays screen info for selecting posho amount to process.
     */
    void doPoshoRfidH();

    /**
     *  Prepare to active the posho and husker for serial use. Change user balance as necessary
     *  and send corresponding signal. Also sets posho flag to in use and sets an interrupt
     *  to clear that flag after a specified time.
     *  @param numKilograms - int for number of kilograms to be processed
     *  @param userIndex - int representing index of user in user table
     */
    void poshoSerialUseH(int numKilograms, int userIndex);
    /**
     *  Do if Incubator RFID flag is set. Add incubator functionality later.
     */
    void doIncubatorRfidH();


    void logActionWithUserInfo(enum HubActionForLogging action, HubUser *user);
    void logErrorUnknownRfidScanned(uint32_t unknownRfid, enum HubLoggingRfidSource source);

    /**
     * Sends a signal to a locker that the current user is picking up numBatteries batteries
      * @param userid - the numer in the all users array.
     * @param numBatteries - int representing number of batteries user is picking up
     */
    void txBatteryPickUp(int usedid, int numBatteries);

    /**
     * Sends a signal to a locker that the current user is dropping off numBatteries batteries
     * @param userid - the numer in the all users array.
     * @param numBatteries - int representing number of batteries user is dropping off
     */
    void txBatteryDropOff(int usedid, int numBatteries);

    /**
     * Sends a signal to a services mbed to schedule the posho/husker for use for x kilograms
     * @param numKilograms - int representing number of kilograms user is processing
     */
    void txPoshoSerialUse(int numKilograms);


};


#endif
