/**
 * HubUser.h
 *
 * HubUser class. A HubUser is an object containing a uid number, an rfid 
 * number, a max number of batteries that can be checked out, a locker
 * number, and a pod within a locker.
 * 
 * Author: Daniel Yang
 */

#ifndef HUBUSER_H
#define HUBUSER_H

#include "mbed.h"

class HubUser {
    public:
        /**
         * Generic constructor. Initializies all values with 0.
         */
        HubUser();

        /**
         * Constructor that allows for specific initializiation of all values
         * @param uid - uint32_t corresponding to unique id of the RFID tag
         * @param rfid - uint32_t corresponding to unique id of the RFID tag
         * @param accountCredit - int32 for user account balance (can be negative)
         * @param locker - int32_t for which locker (originaly 1-4) user is assigned
         * @param pod - int32_t pod associated with the user (originally 0-15) *must be 0 indexed*
         * @param batteriesOut - int32_t number of batteries user has out
         * @param batterySubscription -  int32_t max batteries user can get (can be 0)
         */
        HubUser(uint32_t uid, uint32_t rfid, int32_t accountCredit, 
                int32_t locker, int32_t pod, int32_t batteriesOut,
                int32_t batterySubscription, char* name);

        /**
         * Get the uid
         */
        uint32_t getUid();

        /**
         * Get the rfid
         */
        uint32_t getRfid();

        /**
         * Get the account balance
         */
        int32_t getCredit();

        /**
         * Get the locker number
         */
        int32_t getLocker();

        /**
         * Get the pod (0 indexed)
         */
        int32_t getPod();

        /**
         * Get the number of batteries currently out
         */
        int32_t getBatteriesOut();

        /**
         * Get the max number of batteries, aka the subscription
         */
        int32_t getBatteriesMax();

        /**
         * Returns the hub user name
         */

        char* getName();
        /**
         * Set the uid
         * @param uid
         */
        void setUid(uint32_t uid);

        /**
         * Set the rfid
         * @param rfid
         */
        void setRfid(uint32_t rfid);

        /**
         * Set the account balance
         * @param credit
         */
        void setCredit(int32_t credit);

        /**
         * Set the locker number
         * @param locker
         */
        void setLocker(int32_t locker);

        /**
         * Set the pod (0 indexed)
         * @param pod
         */
        void setPod(int32_t pod);

        /**
         * Set the number of batteries currently out
         * @param batteries
         */
        void setBatteriesOut(int32_t batteries);

        /**
         * Set the max number of batteries, aka the subscription
         * @param batteries
         */
        void setBatteriesMax(int32_t batteries);

        /**
         * Drop off x number of batteries
         * @param batteries - number of batteries being dropped off
         */
        void dropOffBattery(int32_t batteries);

        /**
         * Pick up x number of batteries
         * @param batteries - number of batteries being picked up
         */
        void pickUpBattery(int32_t batteries);

        /**
         * Decrease account credit after a transaction
         * @param credit - amount to decrease balance by 
         */
        void decreaseCredit(int32_t credit);

    protected:
        uint32_t uid;                   /**< uid - uint32_t corresponding to unique id of the RFID tag. */
        uint32_t rfid;                  /**< rfid - uint32_t corresponding to unique id of the RFID tag. */
        int32_t accountCredit;          /**< accountCredit - int32 for user account balance (can be negative). */
        int32_t locker;                 /**< locker - int32_t for which locker (originaly 1-4) user is assigned. */
        int32_t pod;                    /**< pod - int32_t pod associated with the user (originally 0-15) *must be 0 indexed*. */
        int32_t batteriesOut;           /**< batteriesOut - int32_t number of batteries user has out. */
        int32_t batterySubscription;    /**< batterySubscription - int32_t max batteries user can get (can be 0). */
        char* name;                     /**< Name of the user */
};

#endif
