/**
 *@section DESCRIPTION
 * mbed SolarNanogrid Battery.
 *@section LICENSE
 * Copyright (c) 2016, Malcolm McCulloch
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 * @file "Battery.h"
 */

#ifndef BATTERY_H
#define BATTERY_H

//#define USEHUBCHARGERATE     // define this to use charge rate broadcast by the hub

#include <mbed.h>
#include "NRF2401P.h"
#include "nRF24l01.h"
#include "SDFileSystem.h"
#include "SolarNanoGrid.h"

/** \brief The battery logs data and interacts with the locker.
 *
 *  The config.ini must start with B
 *
 */
class Battery : public SolarNanoGrid
{
public:
    // *********************
    // * Public variables: *
    // *********************
    char* logDir;
    float maxChargeRate; // Charge rate in amps

    // *********************
    // * Public functions: *
    // *********************

    // Constructor:
    Battery(FILE* fp, Serial *pc);

    // Main loop function:
    virtual void loop(void);
    /**
     * Checks in the battery to the locker
     * Transmits "C ID"
     */
    void checkIn();

    /**
     * Returns the max current that can be drawn from the system, in amps
     */

    float getMaxCurrent();

    /**
     * returns a string of the log directory
     */
    char *getLogDirectory();

    /**
     * Returns true if we are authorized to start charging.
     */
    bool startCharging(void);

    /**
     * Returns true if we are checked in.
     */
    bool isCheckedIn(void);

    /**
     * sends a request for a charge update.
     */
    void requestMaxCharge();

    /**
     * Disables all of the interrupts, for a proper sleep state.
     */
    void disableInt(void);

protected:
    // **********************
    // * Protected variables: *
    // **********************
    /**
     * Name of log directory, include last /
     */

    char count;
    char* txFileName;

    char* fullTxFilePath;

    int sizeRead;
    unsigned int fileLength;
    unsigned int filePointer;
    unsigned int fileLeft;
    char *sdBuffer;
    unsigned int sdBuffPnt;
    /**
     * Open channel address of the locker
     */
    long long openAddr;

    /**
     * Private channel address of the locker
     */
    long long privateAddr;

    unsigned int countSD;

    // **********************
    // * Protected flags: *
    // **********************
    volatile char * flag;
    volatile char checkedIn;
    volatile char flagGetTime;
    volatile char flagNextFile;
    volatile  char flagHubNotReady;  /**< True if flagStartFileTx and MaxRT */
    volatile  char flagStartFileTx;  /**< Start of file transmission. 0 when all files done */
    volatile  char  flagDirFileName;
    volatile char flagSendFileName;
    volatile char flagSendFileSize;
    volatile char flagSendFile;
    volatile char flagSendFileDone;
    volatile char flagGetMaxCharge;
    volatile char sendingFile;
    volatile bool startCharge;
    volatile char flagCheckOut;
    volatile char sending;

    // **********************
    // * Protected interrupts: *
    // **********************
    InterruptIn* button;
    Ticker* txWatch;
    Timeout* delay;

    // **********************
    // * Protected functions: *
    // **********************
    /**
     * Checks in the battery to the locker
     * Acknowledge received.
     * Changes the TX pipe address to private
     *
     */
    void doCheckIn();
    /**
     * Checks out the battery from the locker
     * Acknowledge received.
     * Changes the TX pipe address to open
     *
     */
    void doCheckOut();
    /**
     * sends a request for time update.
     */
    void requestTime();
    /**
     * Sets the value pointed to by flag to 1.
     */

    void nextFileToTx();
    void sendDirName();
    void sendFileName();
    void sendFileSize();
    void sendFile();
    void sendFileDone();
    void setFlag();

    /**
    * Checks out. Sends O %d, ID to locker
    *
    */
    void checkOut();
    /**
     * Retransmits the last data.
     *
     */
    void retransmit();


    /**
     * called when the nrf the button is pressed
     *
     */
    void intButton();
    /**
     * called when the nrf creates an interrupt.
     *
     */
    void intNrf();
   void resetPackets();
    /**
     * Gathers a packet from the file defined by fullTxFilePath
     * @param readOnly if true then it does not move the internal pointers on.
     */
    char* getPacket(unsigned int *sizePackett,char * fullTxFilePath);


private:
    // **********************
    // * Private variables: *
    // **********************

    volatile bool maxRT;


};
#endif
