/**
* @author Damien Frost
*
* @section LICENSE
*
*   Copyright (c) 2016 Damien Frost
*
*   Permission is hereby granted, free of charge, to any person obtaining a copy
*   of this software and associated documentation files (the "Software"), to deal
*   in the Software without restriction, including without limitation the rights
*   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
*   copies of the Software, and to permit persons to whom the Software is
*   furnished to do so, subject to the following conditions:
*
*   The above copyright notice and this permission notice shall be included in
*   all copies or substantial portions of the Software.
*
*   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
*   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
*   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
*   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
*   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
*   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
*   THE SOFTWARE.
*
* @file "main.cpp"
*
* @section DESCRIPTION
*   Simple Internet of Things main program. The device sends data every 3
*   seconds, and can receive data from a server.
*
*/

#include "mbed.h"
#include "globals.h"
#include "WiflyInterface.h"
#include "Commands.h"
#include "Websocket.h"
#include "ADC.h"

//#define DEBUG
#define INFOMESSAGES
#define WARNMESSAGES
#define ERRMESSAGES
#define FUNCNAME "IoT"
#include "messages.h"


// Main Loop!
int main() {
    unsigned int wifi_cmd = NO_WIFI_CMD;
    float wifi_data = 0.0f;
    unsigned int ADCRaw;
    char msg[128];
    
    // Set the IoT ID:
    IoT_ID = 5;
    
    // Set the Auto reconnect flag:
    IotStatus.SetFlag(SF_AUTOCONNECT);
    
    // Send a startup message to serial port:
    INFO("");
    INFO("");
    INFO("Starting up...");
    INFO("CPU SystemCoreClock is %d Hz", SystemCoreClock); 
    
    // Configure the ADC to sample the internal temperature sensor. You cannot
    // use AnalogIn() unfortunately...
    ConfigureADC();
    
    // Connect to the wifi network. It will basically get stuck here until it
    // connects to the network.
    SetupNetwork(5000);
    
    // Configure the baud rate of the wifi shield:
    // This will make our wireless transmissions much faster.
    ws.setBaud(115200);
    wait(0.5f);
    
    // Check to see we are connected to the network:
    if(IotStatus.CheckFlag(SF_WIRELESSCONNECTED)){
        // Try to connect to the WebSocket server:
        sprintf(msg, "ws://%s:%d/ws", SERVER_IP, WS_PORT);
        ws.Initialize(msg);
        INFO("Connecting to Websocket Server on %s...", msg);
        if(ws.connect()){
            // Set a status flag:
            INFO("Connected.");
            IotStatus.SetFlag(SF_SERVERCONNECTED);
        }else{
            // We could not connect right now..
            IotStatus.ClearFlag(SF_SERVERCONNECTED);
            INFO("Could not connect to server, will try again later.");
            ReconnectAttempts++;
        }
    }
    
    // Start the display timer which will send data to the server every
    // 3 seconds.
    DisplayTimer.start();
    
    // Inifinite main loop:
    while(1) {
        
        // Process the wifi command:
        if(wifi_cmd > NO_WIFI_CMD){
            // Modify the desired variable:
            ModifyVariable(wifi_cmd, wifi_data);
            // Reset the command:
            wifi_cmd = NO_WIFI_CMD;
        }
        
        // Check for new wifi data:
        if((wifi_cmd == NO_WIFI_CMD)){
            ReceiveNetworkData(&wifi_cmd, &wifi_data);
        }
        
        // Send the network data every 3 seconds:
        if(DisplayTimer.read()>(3.0f)){
            // Sample the internal temperature sensor:
            STARTADCCONVERSION;
            // Wait for the conversion to complete:
            while(!ADCCONVERSIONCOMPLETE);
            // Save the raw value from the ADC:
            ADCRaw = ADC1->DR;
            // Calculate the temperature using information from the datasheet:
            TempSensor = ((((float)ADCRaw)/ADC_MAX)*IT_VMAX - IT_V25)/IT_AVG_SLOPE + 25.0f;
            // Output the result:
            DBG("TempSensor = %.5f", TempSensor);
            DBG("ADC1->DR = %d", ADCRaw);
            
            // Send data over network:
            SendNetworkData();               
            
            // Increment a counter:
            SendCounter++;
            
            // Reset the timer:
            DisplayTimer.reset();
            
        }
    } // while(1)
} // main()
