/**
* @author Damien Frost
*
* @section LICENSE
*
*   Copyright (c) 2016 Damien Frost
*
*   Permission is hereby granted, free of charge, to any person obtaining a copy
*   of this software and associated documentation files (the "Software"), to deal
*   in the Software without restriction, including without limitation the rights
*   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
*   copies of the Software, and to permit persons to whom the Software is
*   furnished to do so, subject to the following conditions:
*
*   The above copyright notice and this permission notice shall be included in
*   all copies or substantial portions of the Software.
*
*   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
*   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
*   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
*   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
*   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
*   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
*   THE SOFTWARE.
*
* @file "globals.cpp"
*
* @section DESCRIPTION
*   Global definitions for the IoT example program.
*
*/

#include "mbed.h"
#include "globals.h"
#include "pwm.h"

#define DEBUG
#define INFOMESSAGES
#define WARNMESSAGES
#define ERRMESSAGES
#define FUNCNAME "GBL"
#include "messages.h"

// Hardware declarations:
Serial          pc(USBTX, USBRX);
InterruptIn     UIBut1(USER_BUTTON);
Timer           DisplayTimer;
DigitalOut      Led(D5);

// Global variable declarations:
int ReconnectAttempts = 0;
int SendCounter = 0;
extern int IoT_ID = 0;
float TempSensor = 0.0f;
float VoltageMeasurement = 0.0f;
float CurrentMeasurement = 0.0f;
float PwmPeriod_us = 20.0f;
float Duty_us = 10.0f;
char* wifissid = "SC";
char* wifipassword = "smartcellshield";
DigitalOut      db(PC_10);
AnalogIn        VoltageSensor(VOLTAGE_SENSOR_PIN);
AnalogIn        CurrentSensor(CURRENT_SENSOR_PIN);


// Wifily interface declaration:
WiflyInterface eth(D8, D2, D6, LED1, wifissid, wifipassword, WPA2);
// WebSocket declaration:
Websocket ws;


int SetupNetwork(int Tries){
    // Initialize the interface.
    // If no param is passed to init() then DHCP will be used on connect()
    int s = eth.init();
    int attempts = 1;
       
        wait(1);
        if (s != NULL) {
            ERR("Could not initialise. Halting!");
            exit(0);
        }
    
        INFO("Connecting to: %s", wifissid);
        DBG("Getting IP address...");
        
        while (1) {
            // Connect to network:
            s = eth.connect();
            // If connection fails, retry for 5 attempts:
            if (s == false || s < 0) {
                INFO("Could not connect to network. Retrying!");
                attempts++;
                wait(1);
            } else {
                
                break;
            }
            if(attempts > Tries){
                ERR("Network connection failed after %d attempts", Tries);
                return 0;
            }
        }
        INFO("Connected to: %s", wifissid);
        INFO("Got IP address: %s", eth.getIPAddress());
        IotStatus.SetFlag(SF_WIRELESSCONNECTED);
        return 1;
    
}

void SendNetworkData(void){
    char msg_buffer[CHARMSGBUFF];
    int intresult;    
    
    if(IotStatus.CheckFlag(SF_SERVERCONNECTED)){
        sprintf(msg_buffer, "%d,%d,%.5f,%.5f,%.5f,%.5f", IoT_ID, SendCounter,TempSensor, VoltageMeasurement, CurrentMeasurement, PwmPeriod_us);
        INFO("Sending: %s", msg_buffer);    // When this line is commented out, the mbed never tries to reconnect to the server after one try. SUPER. Keeping this here also uses precious CPU time
        intresult = ws.send(msg_buffer);
    }else{
        intresult = -1;
    }
    DBG("intresult: %d", intresult);
        
    if(intresult < 0){
        // Clear a status flag:
        IotStatus.ClearFlag(SF_SERVERCONNECTED);
        // Check to see if the wireless is still connected:
        DBG("Checking network status...");
        if(eth.checkNetworkStatus() != 3){
            IotStatus.ClearFlag(SF_WIRELESSCONNECTED);
            // Connect to the wireless network:
            if(IotStatus.CheckFlag(SF_AUTOCONNECT)){
                INFO("Reconnecting to Network...");
                if(SetupNetwork(1)>0){
                    IotStatus.SetFlag(SF_WIRELESSCONNECTED);
                    INFO("Connected to Network.");
                }else{
                    WARN("Could not re-connect to the wireless network.");
                }
            }
        }else{
            DBG("Network connected.");
        }
        
        if(IotStatus.CheckFlag(SF_AUTOCONNECT) && IotStatus.CheckFlag(SF_WIRELESSCONNECTED)){
            // Server connection was closed, try to reconnect:
            INFO("Reconnecting to Websocket Server on ws://%s:%d/ws...", SERVER_IP, WS_PORT);
            if(!ws.connect()){
                WARN("Could not connect to the server again...");
                IotStatus.ClearFlag(SF_SERVERCONNECTED);
                ReconnectAttempts++;
                if(ReconnectAttempts > 4){
                    INFO("Failed after %d reconnect attempts. Resetting the Wifi Shield...", ReconnectAttempts);
                    SetupNetwork(1);
                    ReconnectAttempts = 0;
                }
            }else{
                INFO("Connected to ws://%s:%d/ws", SERVER_IP, WS_PORT);
                // Set a status flag:
                IotStatus.SetFlag(SF_SERVERCONNECTED);
            }
        }
    }

    return;
}

void ReceiveNetworkData(unsigned int * wifi_cmd, float * value){
    char msg_buffer[CHARMSGBUFF];
    char msg_buffer2[CHARMSGBUFF];
    int resp;
    if(IotStatus.CheckFlag(SF_SERVERCONNECTED)){
        // Check for data on the websocket:
        resp = ws.readmsg(msg_buffer);
        if(resp == 1){
            INFO("Received: %s", msg_buffer);
            sscanf(msg_buffer, "%d,%s", wifi_cmd, msg_buffer2);
            if(*wifi_cmd == CV_LED_WIFI_CMD){
                // Get one more value:
                sscanf(msg_buffer2, "%f", value);
            }
            if(*wifi_cmd == CV_PWM_PERIOD_US_CMD){
                // Get one more value:
                sscanf(msg_buffer2, "%f", value);
            }
        }else if(resp == -1){
            // Connection to the server is lost:
            IotStatus.ClearFlag(SF_SERVERCONNECTED);
        }else{
            //DBG("Did not receive anything :(\n\r");
            *wifi_cmd = NO_WIFI_CMD;
            *value = 0.0f;
        }
    }
    return;
}

void ModifyVariable(unsigned int wifi_var, float wifi_data){
    // modifies something in the SCS Controller:
    switch(wifi_var){
        case CV_LED_WIFI_CMD:
            if(wifi_data > 0){
                Led = 1;
            }else{
                Led = 0;
            }
            break;
        case CV_PWM_PERIOD_US_CMD:
            DBG("wifi_data: %.3f", wifi_data);
            PwmPeriod_us = wifi_data;
            SetPWMPeriodAndDuty_us(PwmPeriod_us);
            break;
            
        
        default:
            break;
    }
    return;
}
        

 