/* mbed DS1820 Library, for the Dallas (Maxim) 1-Wire Digital Thermometer
 * Copyright (c) 2010, Michael Hagberg Michael@RedBoxCode.com
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifndef MBED_DS1820_H
#define MBED_DS1820_H

#include "mbed.h"
#include "LinkedList.h"

#define FAMILY_CODE _ROM[0]
#define FAMILY_CODE_DS1820 0x10
#define FAMILY_CODE_DS18B20 0x28
#define FAMILY_CODE_DS1822  0x22

/** DS1820 Dallas 1-Wire Temperature Probe
 *
 * Example:
 * @code
 * #include "mbed.h"
 * #include "DS1820.h"
 *
 * DS1820 probe(DATA_PIN);
 *  
 * int main() {
 *     while(1) {
 *         probe.convertTemperature(true, DS1820::all_devices);         //Start temperature conversion, wait until ready
 *         printf("It is %3.1foC\r\n", probe.temperature());
 *         wait(1);
 *     }
 * }
 * @endcode
 */
class DS1820 {
public:
    enum devices{
        this_device,     // command applies to only this device
        all_devices };   // command applies to all devices
    
    enum {
        invalid_conversion = -1000
    };

    /** Create a probe object connected to the specified pins
     *
     * @param data_pin DigitalInOut pin for the data bus
     */
    DS1820(PinName data_pin);
    ~DS1820();

    /** This routine will initiate the temperature conversion within
      * one or all DS1820 probes. 
      *
      * @param wait if true or parisitic power is used, waits up to 750 ms for 
      * conversion otherwise returns immediatly.
      * @param device allows the function to apply to a specific device or
      * to all devices on the 1-Wire bus.
      * @returns milliseconds untill conversion will complete.
      */
    int convertTemperature(bool wait, devices device=all_devices);

    /** This function will return the probe temperature. Approximately 10ms per
      * probe to read its RAM, do CRC check and convert temperature on the LPC1768.
      *
      * @param scale, may be either 'c' or 'f'
      * @returns temperature for that scale, or DS1820::invalid_conversion (-1000) if CRC error detected.
      */
    float temperature(char scale='c');

    /** This function sets the temperature resolution for the DS18B20
      * in the configuration register.
      *
      * @param a number between 9 and 12 to specify resolution
      * @returns true if successful
      */ 
    bool setResolution(unsigned int resolution);       

private:
   
    static char CRC_byte(char _CRC, char byte );
    bool onewire_reset();
    void match_ROM();
    void skip_ROM();
    bool search_ROM_routine(char command, char *ROM_address);
    void onewire_bit_out (bool bit_data);
    void onewire_byte_out(char data);
    bool onewire_bit_in();
    char onewire_byte_in();
    static bool ROM_checksum_error(char *_ROM_address);
    bool RAM_checksum_error();
    void read_RAM();
    bool unassignedProbe(char *ROM_address);
    void write_scratchpad(int data);

    DigitalInOut _datapin;
    
    char _ROM[8];
    char RAM[9];
    
    static LinkedList<node> probes;
};


#endif