/*-----------------------------------------------------------
 * F469_BUTTON Library v1.0
 * Copyright (c) 2018 Wynand Steenberg
 * s216875730@mandela.ac.za
 *
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 *-----------------------------------------------------------
 */

#ifndef F469_BUTTON_HPP
#define F469_BUTTON_HPP

#include "mbed.h"
#include <string>
#include "TS_DISCO_F469NI.h"
#include "LCD_DISCO_F469NI.h"
#include "F469GUI.hpp"

/** A Class library for using Buttons on the DISCO-F469NI Development board.  The class
 * uses the existing BSP class created by Team ST.
 *
 * Example:
 * @code
 * #include "mbed.h"
 * #include "F469_BUTTON.hpp"
 * 
 * DigitalOut led_green(LED1);
 * DigitalOut led_orange(LED2);
 * DigitalOut led_red(LED3);
 * DigitalOut led_blue(LED4);
 * 
 * TS_DISCO_F469NI ts_;
 * LCD_DISCO_F469NI lcd_;
 * 
 * int main()
 * {
 *     led_green = 1;         // Switch off all LEDs
 *     led_orange = 1;
 *     led_red = 1;
 *     led_blue = 1; 
 * 
 *     lcd_.Clear(LCD_COLOR_WHITE);                                        // Set LCD Background colour
 *        
 *     Button btn1(lcd_, ts_, 20, 100, 120, 60,            
 *                  LCD_COLOR_BLUE, LCD_COLOR_YELLOW, 1, "BUTTON1", Font20);  // Define btn1 button            
 *     btn1.Render();                                                         // Draw btn1 button
 *     
 *     Button test(lcd_, ts_, 160, 100, 120, 60,
 *                  LCD_COLOR_DARKBLUE, LCD_COLOR_GREEN, 1, "TEST", Font20);  // Define test button                   
 *     test.Render();                                                         // Draw test button
 *     
 *     Button show(lcd_, ts_, 300, 100, 120, 60,            
 *                  LCD_COLOR_BROWN, LCD_COLOR_GRAY, 1,  "SHOW", Font20);     // Define hide button            
 *     show.Hide();  
 *     
 *     Button hide(lcd_, ts_, 300, 100, 120, 60,            
 *                  LCD_COLOR_BROWN, LCD_COLOR_GRAY, 1,  "HIDE", Font20);     // Define hide button             
 *     hide.Render();                                                         // Draw hide button
 *     
 *     Button button3D(lcd_, ts_, 580, 100, 160, 60,            
 *                  LCD_COLOR_BLUE, LCD_COLOR_YELLOW, 2, "3D BUTTON", Font20);    // Define button3D button            
 *     button3D.Render();                                                         // Draw 3Dbutton button
 * 
 * 
 * 
 *     lcd_.SetTextColor(LCD_COLOR_BLACK);                                                         // Set Text colour to Black
 *     lcd_.SetFont(&Font20);                                                                      // Font size 20
 *     lcd_.DisplayStringAt(5, 5, (uint8_t *)"Button / Slider driver for DISCO_F469", LEFT_MODE);  // Display main header text
 * 
 *     lcd_.SetFont(&Font16);                                                                      // Font size 16
 *     lcd_.DisplayStringAt(5, 45, (uint8_t *)"Button example", LEFT_MODE);                        // Display secondary header text
 * 
 * 
 *     while (true)                                                    // Main program loop
 *     {
 *         led_green = 1;                                              // Switch off all LEDs
 *         led_orange = 1;
 *         led_red = 1;
 *         led_blue = 1; 
 *     
 *     
 *         if (btn1.Press())                                         // Check if btn1 button was touched and run instructions if true
 *         {    
 *             lcd_.SetFont(&Font16);
 *             lcd_.SetTextColor(LCD_COLOR_BLACK);
 *             lcd_.DisplayStringAt(5, 200, (uint8_t *)"Button1 pressed - switch on orange LED", LEFT_MODE);
 *             led_orange = 0; 
 *             wait(0.5); 
 *             lcd_.DisplayStringAt(5, 200, (uint8_t *)"                                        ", LEFT_MODE);
 *         }                                                           // End btn1 button instructions
 *         
 *         if (test.Press())                                         // Check if test button was touched and run instructions if true
 *         {
 *             lcd_.SetFont(&Font16);
 *             lcd_.SetTextColor(LCD_COLOR_BLACK);
 *             lcd_.DisplayStringAt(5, 200, (uint8_t *)"TEST pressed - switch on red LED", LEFT_MODE);
 *             led_red = 0;
 *             wait(0.5);
 *             lcd_.DisplayStringAt(5, 200, (uint8_t *)"                                        ", LEFT_MODE);
 *             
 *         }                                                           // End test button instructions
 *         
 *         if (hide.Press())                                         // Check if hide button was touched and run instructions if true
 *         {    
 *             lcd_.SetFont(&Font16);
 *             lcd_.SetTextColor(LCD_COLOR_BLACK);
 *             lcd_.DisplayStringAt(5, 200, (uint8_t *)"Hide pressed - Hide other buttons", LEFT_MODE);
 *             led_green = 0;
 *             btn1.Hide();                                            // Hide btn1 button
 *             test.Hide();                                            // Hide test button
 *             hide.Hide();  
 *             show.Render();
 *             wait(0.5);
 *             lcd_.DisplayStringAt(5, 200, (uint8_t *)"                                        ", LEFT_MODE);
 *         }                                                           // End hide button instructions
 *         
 *          if (show.Press())                                        // Check if hide button was touched and run instructions if true
 *         {  
 *             lcd_.SetFont(&Font16);
 *             lcd_.SetTextColor(LCD_COLOR_BLACK);
 *             lcd_.DisplayStringAt(5, 200, (uint8_t *)"UNHIDE pressed - Restore other buttons", LEFT_MODE);
 *             led_blue = 0; 
 *             wait(0.5);
 *             lcd_.DisplayStringAt(5, 200, (uint8_t *)"                                        ", LEFT_MODE);
 *             show.Hide();                                            // Hide show button before drawing new button in same place
 *             hide.Render();                                            // Draw hide button after hiding button in same position
 *             btn1.Render();                                            // Draw btn1 button
 *             test.Render();                                            // Draw test button
 *             
 *         }         
 *         
 *         if (button3D.Press())                                     // Check if 3Dbutton button was touched and run instructions if true
 *         {    
 *             lcd_.SetFont(&Font16);
 *             lcd_.SetTextColor(LCD_COLOR_BLACK);
 *             lcd_.DisplayStringAt(5, 200, (uint8_t *)"3D Button pressed - switch on blue LED", LEFT_MODE);
 *             led_blue = 0;
 *             wait(0.5);
 *             lcd_.DisplayStringAt(5, 200, (uint8_t *)"                                           ", LEFT_MODE);
 *         }                                                           // End 3D button instructions
 *  
 *         wait(0.02f);   
 *     }                                                               // End Main program loop
 * }                                                                   // End Main program 
 * @endcode
 */

    class Button
    {
    public:
    
        //! Constructor
        Button(LCD_DISCO_F469NI &lcd, TS_DISCO_F469NI &ts,
               uint16_t x, uint16_t y, uint16_t width, uint16_t height,
               uint32_t buttonColour, uint32_t changeColour,
               int style, const string label = "", sFONT &fonts = Font12,
               uint32_t textColour = LCD_COLOR_WHITE)
               : lcd_(lcd), ts_(ts), ButtonX_(x), ButtonY_(y), ButtonW_(width), ButtonH_(height),
                 BUTTON_COLOUR_(buttonColour), CHANGE_COLOUR_(changeColour),
                 STYLE_(style), LABEL_(label), FONTS_(&fonts), FONT_WIDTH_(fonts.Width),
                 FONT_HEIGHT_(fonts.Height), active_(true)
                 { Render(); }

        /**
          * @brief  Draw Button.
          *         
          */
        void Render();
        
        /**
          * @brief  Draw 3D Button.
          *         
          */
        void Render3D();
        
        /**
          * @brief  Change colour of Button.
          *         
          */
        void Change();

        /**
          * @brief  Change colour of 3D Button.
          *         
          */
        void Change3D();
        
        /**
          * @brief  Hide Button.
          *         
          */
        void Hide();

        /**
          * @brief  Check touch detected on Button.
          *         
          */
        bool Press();
        
        /**
          * @brief  Check if touch is on Button.
          *         
          */      
        bool ButtonBoundaryCheck();
        
        /**
          * @brief  Check previous state of Button.
          *         
          */
        static TS_StateTypeDef GottenState()
        {   return state_; }

        /**
          * @brief  Set or reset multi-touch.
          *         
          */
        static void SetMultiTouch(bool tf) { multiTouch = tf; }


    private:

        LCD_DISCO_F469NI &lcd_;         // LCD
        TS_DISCO_F469NI &ts_;           // Touch
        
        const uint16_t ButtonX_, ButtonY_, ButtonW_, ButtonH_;
        const uint32_t BUTTON_COLOUR_;  // Original colour
        const uint32_t CHANGE_COLOUR_;  // Colour to change to color
        const int STYLE_;               // Button style
        const string LABEL_;            // Button label
        sFONT *const FONTS_;            // Button Style
        const uint16_t FONT_WIDTH_;     
        const uint16_t FONT_HEIGHT_;
        bool active_;                   // Button active boolean
        
        static TS_StateTypeDef state_;
        static bool multiTouch;

        /**
          * @brief  Disallow copy constructor and assignment operator.
          *         
          */ 
        Button(const Button&);
        Button& operator=(const Button&);
        

    };
#endif  // F469_BUTTON_HPP
