#include "mbed.h"
#include "EthernetNetIf.h"
#include "HTTPClient.h"
#include "ID12RFID.h"
#include "Camera_LS_Y201.h"
#include "SDFileSystem.h"

#define DEBMSG      printf
#define NEWLINE()   printf("\r\n")

#define FILENAME    "/local/%08d.jpg"
LocalFileSystem fs("local");
Camera_LS_Y201 cam1(p9, p10);

typedef struct work {
    FILE *fp;
} work_t;
work_t work;

ID12RFID rfid(p14); // uart rx
EthernetNetIf eth;
HTTPClient http;
DigitalOut myled(LED1);
DigitalOut myled2(p20);
DigitalOut door(p21);
int tagNum;

bool CheckTag(int tag){
    HTTPText txt;
    const char * txtBuffer;
    char* bufferCopy;
    char* line;
    
    // retrieve authorized tag numbers, CSV format
    HTTPResult r = http.get("http://www.prism.gatech.edu/~dburnham6/tags.txt", &txt);
    if(r==HTTP_OK){
        txtBuffer = txt.gets();
        bufferCopy = const_cast<char *> (txtBuffer); // allow parsing of the text
        line = strtok(bufferCopy, ","); // separate by commas
        while (line != NULL) {
            if(atoi(line) == tag) {
                return true;
            }
            line = strtok(NULL, ",");
        }
        
        printf("Tag Not Found.\r\n"); 
    }
    else
    {
        printf("HTTP Error %d\n", r);
    }
    return false;
}

/** [from the Camera_LS_Y201 test code]
 * Callback function for readJpegFileContent.
 *
 * @param buf A pointer to a buffer.
 * @param siz A size of the buffer.
 */
void callback_func(int done, int total, uint8_t *buf, size_t siz) {
    fwrite(buf, siz, 1, work.fp);

    static int n = 0;
    int tmp = done * 100 / total;
    if (n != tmp) {
        n = tmp;
        DEBMSG("Writing...: %3d%%", n);
        NEWLINE();
    }
}

/** [from the Camera_LS_Y201 test code]
 * Capture.
 *
 * @param cam A pointer to a camera object.
 * @param filename The file name.
 *
 * @return Return 0 if it succeed.
 */
int capture(Camera_LS_Y201 *cam, char *filename) {
    /*
     * Take a picture.
     */
    if (cam->takePicture() != 0) {
        return -1;
    }
    DEBMSG("Captured.");
    NEWLINE();

    /*
     * Open file.
     */
    work.fp = fopen(filename, "wb");
    if (work.fp == NULL) {
        return -2;
    }

    /*
     * Read the content.
     */
    DEBMSG("%s", filename);
    NEWLINE();
    if (cam->readJpegFileContent(callback_func) != 0) {
        fclose(work.fp);
        return -3;
    }
    fclose(work.fp);

    /*
     * Stop taking pictures.
     */
    cam->stopTakingPictures();

    return 0;
}

int main() {
    // init ethernet
    EthernetErr ethErr = eth.setup();
    if(ethErr)
    {
        printf("Error %d in setup.\n", ethErr);
    }
    printf("Setup OK\n");
    
    // init camera
    DEBMSG("Camera module");
    NEWLINE();
    DEBMSG("Resetting...");
    NEWLINE();
    wait(1);
    if (cam1.reset() == 0) {
        DEBMSG("Reset OK.");
        NEWLINE();
    } else {
        DEBMSG("Reset fail.");
        NEWLINE();
        error("Reset fail.");
    }
    wait(1);
    
    // wait for tag
    while(1) {
        if(rfid.readable()) {
            int tag = rfid.read();
            printf("RFID Tag number : %d\r\n", tag);
            if(CheckTag(tag)) {
                printf("Authorized user confirmed. Unlocking door...\r\n");
                door = 1;
                myled = 1;
                wait(10);
                door = 0;
                myled = 0;
            } else {
                printf("Unauthorized access tag. Picture for evidence... \r\n");
                myled2 = 1;
                char fname[64];
                snprintf(fname, sizeof(fname) - 1, FILENAME, tag);
                int r = capture(&cam1, fname); // take a picture
                if (r == 0) {
                    DEBMSG("[%04d]:OK.", tag);
                    NEWLINE();
                } else {
                    DEBMSG("[%04d]:NG. (code=%d)", tag, r);
                    NEWLINE();
                    error("Picture failed.");
                }
                myled2 = 0;
            }
        }
    }
}
