/* RFDuino Interface Library
 * Copyright (c) 2006-2013 Your Name Here
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef RFDUINO_H
#define RFDUINO_H

#include "mbed.h"

int const buff_size = 255;

typedef struct  {
    unsigned char buff[buff_size];
    int len;
} RFD_data;

/** How to use the RFDuino library
 *  Here is an example:
 *  @code
* #include "mbed.h"
* #include "RFDuino.h"
* 
* PwmOut red(LED_RED);
* PwmOut green(LED_GREEN);
* PwmOut blue(LED_BLUE);
* 
* RFDuino rfd(PTC4, PTC3); // defaults to 9600 baud
* 
* typedef union {
*     float fp;
*     unsigned int uint_32;
* } intFloat;
* 
* typedef struct  {
*     intFloat red;
*     intFloat green;
*     intFloat blue;
* } RGB;
* 
* void bblink(void)
* {
*     blue=0;
*     wait(0.5);
*     blue=1;
*     wait(0.5);
* }
* 
* void byteSwap(RGB* rgb)
* {
*     intFloat temp;
* 
*     temp.uint_32 = __REV(rgb->red.uint_32);
*     rgb->red.fp = temp.fp;
* 
*     temp.uint_32 = __REV(rgb->blue.uint_32);
*     rgb->blue.fp = temp.fp;
* 
*     temp.uint_32 = __REV(rgb->green.uint_32);
*     rgb->green.fp = temp.fp;
* 
* }
* 
* int main()
* {
*     RGB rgb = {1.0f, 1.0f, 1.0f};
*     
*     red = blue = green = 1.0f;
*     wait(2.0f); // make sure to wait at least 2 seconds before handshake
* 
*     // flash red and halt here if rfduino is not on the line
*     if(!rfd.handshake()) {
*         while(1) {
*             red = !red;
*             wait(0.2f);
*         }
*     }
* 
*     // wait for a connection
*     while(!rfd.isConnected()) {
*         bblink();
*     }
* 
*     while(1) {
*         if(rfd.readable()) {
*             // copy data into data struct
*             rfd.read((unsigned char*)&rgb.red, sizeof(rgb));
*             
*             // re-arrange bytes
*             byteSwap(&rgb);
* 
*             //set pwms
*             red   = rgb.red.fp;
*             blue  = rgb.blue.fp;
*             green = rgb.green.fp;
*         }
*     }
* }
 *
 *  @endcode
 */

class RFDuino
{

private:
    Serial      rfd;
    RFD_data    data;
    bool        dataFlag;
    
    void receive_isr();

public:

    /**
     *  Constructor for RFDuino
     *  @param tx - Serial TX pin from target MCU
     *  @param rx - Serial RX pin from target MCU
     */
    RFDuino(PinName tx, PinName rx);
    
    /**
     *  Do a handshake with the device
     *  @return 1 if the RFduino device is connected, 0 otherwise
     */
    bool handshake(void);
    
    /**
     *  Check to see if data is ready to read
     *  @return 1 if data is available to read, 0 otherwise
     */
    bool readable(void) const;
    
    /**
     *  Check to see if the device is connected
     *  @return 1 if Bluetooth Connection is established, 0 otherwise
     */
    bool isConnected(void);
    
    /**
     *  Send data to the device
     *  @param buff - A buffer of data to send
     *  @param len - the amount of data in the buff
     */
    bool transmit(const unsigned char *buff, const int len);
    
    /**
     *  Read data from the device
     *  @param buff - A buffer to read data into
     *  @param size - The max amount of data to read
     *  @return - The amount of data read from the device
     */
    int read(unsigned char *buff, const int size);
};

#endif
