///////////////////////////////////////////////////////////////////////////////
// SST25VF064C.cpp - Flash driver
//
// COPYRIGHT (c) 2012 by David Van Wagner
//
// dave@vanwagner.org
// http://techwithdave.blogspot.com
//
// License: Creative Commons Attribution-ShareAlike 3.0 Unported License
// http://creativecommons.org/licenses/by-sa/3.0/
///////////////////////////////////////////////////////////////////////////////

#include <mbed.h>
#ifdef SPIDEBUG
#include "SPIDebug.h"
extern bool debug;
#endif

class SST25VF064C
{
public:
    typedef char int8;
    typedef short int16;
    typedef long int32;
    typedef long long int64;
    
    typedef unsigned char uint8;
    typedef unsigned short uint16;
    typedef unsigned long uint32;
    typedef unsigned long long uint64;

    static const int SID_LEN = 32;
    static const int SECTOR_LEN = 4096;
    static const int PAGE_LEN = 256;
    static const int MAX_ADDR = 0x7FFFFF;

private:
#ifdef SPIDEBUG
    SPIDebug* spi;
    CSDebug* cs;
#else
    SPI* spi;
    DigitalOut* cs;
#endif    
    int frequency;
    
    static uint8 SID_buffer[SID_LEN];
    static uint8 sector_buffer[SECTOR_LEN];
    static int32 buffered_addr;
    static bool buffered_erased;

public:
    SST25VF064C(PinName mosi, PinName miso, PinName sclk, PinName cs, int frequency=10000000);
    ~SST25VF064C();
    
    uint16 Id();
    uint32 JEDECId();
    uint8* SID(); // Security ID (128 bits = 16 bytes)

    // Read Status Register    
    // bit 0 BUSY 1=Write in progress
    // bit 1 WEL  1=Write Enabled
    // bit 2 BP0  block write protection
    // bit 3 BP1  block write protection
    // bit 4 BP2  block write protection
    // bit 5 BP3  block write protection
    // bit 6 SEC  1=Security ID space locked
    // bit 7 BPL  1=BP0..BP3 are read-only, 0=r/w
    uint8 RDSR();
    void EWSR(); // Enable Write Status Register
    void WRSR(uint8 status); // Write Status Register
    
    void WREN(); // Write Enable
    void WRDI(); // Write Disable
    bool BUSY(); // Write in Progress
    bool WEL(); // Write Enabled
    uint8 BP(); // Block Protection
    bool SEC(); // Security ID Locked
    bool BPL(); // Block Protection Locked
    void wait_while_busy();

    uint8 read(int32 addr);
    bool read(int32 addr, uint8* dst, int32 len);
    void hsread(int32 addr, uint8* dst, int32 len, int frequency);

    void sector_erase_4k(int32 addr);
    void block_erase_32k(int32 addr);
    void block_erase_64k(int32 addr);
    void chip_erase();

    bool page_program(int32 addr, uint8* write_buffer, short len);
    bool write(int32 addr, uint8* write_buffer, int32 len);
    bool rewrite(int32 addr, uint8* write_buffer, int32 len);

    bool buffered_write(uint8 data);
    bool buffered_write(uint8* src, int len);
    uint8 buffered_read();
    bool buffered_read(uint8* dest, int len);
    void buffered_seek(int32 addr);
    void buffered_sync();
};
