// emuc64.cpp - Commodore 64 Emulator
//
////////////////////////////////////////////////////////////////////////////////
//
// C64-stm429_discovery
// C64/6502 Emulator targeting STM32F429 LCD/USBHOST
// [ported from c-simple-emu-cbm (C Portable Version - for console)]
//
// MIT License
//
// Copyright(c) 2020 by David R.Van Wagner
// davevw.com
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
// SOFTWARE.
//
////////////////////////////////////////////////////////////////////////////////
//
// This is a C64 Emulator, designed for targeting STM32F29 w/ LCD and USB kbd
//
// LIMITATIONS:
// Loading of files at startup is optional depending on availability of
//   a local file system store (e.g. Mbed MSD)
// Device I/O not implemented.  No tape/serial/printer/disk/joystick.
// Limited VIC II register support (see Memory Map).  No graphics.
// No CIA1/CIA2 except for keyboard scan.
// No sound.  Sorry no SID.
// No cartridges.
//
// MEMORY MAP
//   $00         (data direction missing)
//   $01         Banking implemented (tape sense/controls missing)
//   $0000-$9FFF RAM
//   $A000-$BFFF BASIC ROM
//   $A000-$BFFF Banked LORAM
//   $C000-$CFFF RAM
//   $D000-$D7FF (I/O missing except for select few, reads as zeros)
//    D018       Memory Control Register (bit 1 limited to selecting which ROM character set)
//    D020       Border Color
//    D021       Background Color
//    DC00       CIA#1 limited to writing value to Port A (select keyboard column)
//    DC01       CIA#1 limited to reading value from Port B (read keyboard row)
//   $D800-$DFFF VIC-II color RAM nybbles in I/O space (1K x 4bits)
//   $D000-$DFFF Banked RAM
//   $D000-$DFFF Banked Character ROM
//   $E000-$FFFF KERNAL ROM
//   $E000-$FFFF Banked HIRAM
//
// Requires user provided Commodore 64 BASIC/KERNAL ROMs (e.g. from VICE)
//   as they are not provided, others copyrights may still be in effect.
//
////////////////////////////////////////////////////////////////////////////////
// ROMs copyright Commodore or their assignees
////////////////////////////////////////////////////////////////////////////////

#include <mbed.h>
#include "LCD_DISCO_F429ZI.h"
#include "USBHostKeyboard.h"
//#include <LocalFileSystem.h>
#include "emu6502.h"

// globals
char* StartupPRG = 0;

// locals

LCD_DISCO_F429ZI lcd;

//static int startup_state = 0;
//LocalFileSystem local("local");

// array allows multiple keys/modifiers pressed at one time
static int scan_codes[9] = { 64, 64, 64, 64, 64, 64, 64, 64, 64 } ;

// USB key code to C64 keyboard scan code, plus shift modifiers
// +256 means to apply L.Shift
// -512 means to take away Shift
// +1024 means to apply RESTORE
static int usb_to_c64[2][100] = {
{ // normal/other modifier
	64, 64,	64,	64, 10, 28, 20, 18, 14, 21, // na, na, na, na, a, b, c, d, e, f
	26, 29, 33, 34, 37, 42, 36, 39, 38, 41, // g, h, i, j, k, l, m, n, o, p,
	62, 17, 13, 22, 30, 31, 9, 23, 25, 12, // q, r, s, t, u, v, w, x, y, z
	56, 59, 8, 11, 16, 19, 24, 27, 32, 35, // 1, 2, 3, 4, 5, 6, 7, 8, 9, 0
	1, 63, 0, 58, 60, 43, 53, 256+45, 256+50, 48, // RET, STOP, DEL, CTRL, SPC, -, =, [, ], £
	64, 50, 256+24, 64, 47, 44, 55, 64, 4, 256+4, // na, ;, ', na, ,, ., /, na, f1, f2
	5, 256+5, 6, 256+6, 3, 256+3, 64, 64, 64, 64, // f3, f4, f5, f6, f7, f8, na, na, na, na
	1024+64, 64, 63, 256+0, 51, 1024+64, 0, 64, 64, 2, // RESTORE, na, STOP, INS, HM, RESTORE, DEL, na, na, RT
	256+2, 7, 256+7, 64, 55, 49, 43, 40, 1, 56, // LT, DN, UP, na, /, *, -, +, ENTER, 1
	59, 8, 11, 16, 19, 24, 27, 32, 35, 44 // 2, 3, 4, 5, 6, 7, 8, 9, 0, . (keypad)
},
{ // shift modifier
	64, 64,	64,	64, 10, 28, 20, 18, 14, 21, // na, na, na, na, a, b, c, d, e, f
	26, 29, 33, 34, 37, 42, 36, 39, 38, 41, // g, h, i, j, k, l, m, n, o, p,
	62, 17, 13, 22, 30, 31, 9, 23, 25, 12, // q, r, s, t, u, v, w, x, y, z
	56, 512+46, 8, 11, 16, 512+54, 19, 512+49, 27, 32, // !, @, #, $, %, ^, &, *, (, )
	1, 63, 0, 58, 60, 512+57, 512+40, 256+45, 256+50, 48, // RET, STOP, DEL, CTRL, SPC, L.Arrow, +, [, ], £
	64, 512+45, 256+59, 64, 47, 44, 55, 64, 4, 256+4, // na, :, ", na, ,, ., /, na, f1, f2
	5, 256+5, 6, 256+6, 3, 256+3, 64, 64, 64, 64, // f3, f4, f5, f6, f7, f8, na, na, na, na
	1024+64, 64, 63, 256+0, 51, 1024+64, 0, 64, 64, 2, // RESTORE, na, STOP, INS, HM, RESTORE, DEL, na, na, RT
	256+2, 7, 256+7, 64, 55, 49, 43, 40, 1, 56, // LT, DN, UP, na, /, *, -, +, ENTER, 1
	59, 8, 11, 16, 19, 24, 27, 32, 35, 44 // 2, 3, 4, 5, 6, 7, 8, 9, 0, . (keypad)
}
};

// Resources
// See Keyboard/Keypad Page (0x07) of https://www.usb.org/sites/default/files/documents/hut1_12v2.pdf
// See Commodore 64 Keycodes in Appendix of https://archive.org/details/Compute_s_Mapping_the_64_and_64C/

// The initial goal of the keyboard tables and logic is to provide access to at 
// least the following keys from a standard USB keyboard, symbolic mapping more 
// like a PC (not positional)
// USB keyboard support is provided by USBHOST library and STM32F429 BSP
//
// STOP(ESC) F1 F2 F3 F4 F5 F6 F7 F8            Restore(PrtScr/SysRq) Run/Stop(Pause/Break)
//           1! 2@ 3# 4$ 5% 6^ 7& 8* 9( 0) -_ += Del/Ins(Back)    Ins Hme/Clr     / * -
// Ctrl(Tab) Q  W  E  R  T  Y  U  I  O  P  [  ]  £ (\)            Del           7 8 9 +
//           A  S  D  F  G  H  J  K  L  ;: '" Return(ENTER)                     4 5 6
// LShift    Z  X  C  V  B  N  M  ,< .> /?  RShift                     Up       1 2 3
// C=(Ctrl)           SPACEBAR              C=(Ctrl)              Lft Down Rt   0 .   Enter
//
// Note C64 Ctrl key is the PC Tab key
// Note C64 Commodore key is the PC Ctrl key
// Note keys and modifiers not shown do nothing, not support for C64
// Note PgUp is also Restore key
//
// Also, most Commodore key combinations should work for alphanumeric, some will 
// be different/missing for punctuation, full mapping of PETSCII will require 
// additional development

static void onKeyData(uint8_t len, uint8_t* data)
{
//	static DigitalOut led2(LED2);
//	led2 = !led2;
//	
//	printf("len=%d", len);
//	for (int i=0; i<len; ++i)
//	   printf(" %02X", data[i]);
//	printf("\n");
//	fflush(stdout);
	
	if (len == 8)
	{ 
		if ((data[0] & 0x11) != 0) // PC Ctrl => Commodore
			scan_codes[6] = 61; // usb hid buffer supports 6 simultaneous keys, put modifier in next available slot
		else
			scan_codes[6] = 64;
			
		if ((data[0] & 2) != 0) // LShift
			scan_codes[7] = 15; // usb hid buffer supports 6 simultaneous keys, put modifier in next available slot
		else
			scan_codes[7] = 64;
			
		if ((data[0] & 0x20) != 0) // RShift
			scan_codes[8] = 52; // usb hid buffer supports 6 simultaneous keys, put modifier in next available slot
		else
			scan_codes[8] = 64;

	    for (int i=0; i<6; ++i)
	    {
			if (data[i+2] < 100)
			{
				scan_codes[i] = usb_to_c64[((data[0] & 0x22) != 0) ? 1 : 0][data[i+2]]; // Normal vs. Shift
				if ((scan_codes[i] & 256) != 0)
					scan_codes[7] = 15; // LShift
				if (i==0 && (scan_codes[i] & 512) != 0) // remove shift flag works only if key is first non-modifier pressed
				{
					scan_codes[7] = 64; // No LShift
					scan_codes[8] = 64; // No RShift
				}
				if (scan_codes[i] != 64)
				{
//					printf(" %d", scan_codes[i]);
//					fflush(stdout);
				}
			}
			else
				scan_codes[i] = 64;
		}
	}
}

// CREDIT: keyboard_task() borrowed from USBHOST sample program main.cpp at https://os.mbed.com/teams/ST/code/Nucleo_usbhost/
void keyboard_task(void const *)
{
    USBHostKeyboard keyboard;
    
    while(1) {
        // try to connect a USB keyboard
        while(!keyboard.connect())
            Thread::wait(500);
    
        // when connected, attach handler called on keyboard event
        //keyboard.attach(onKey);
        //keyboard.attach(onKeyEvent);
        keyboard.attach(onKeyData);
        
        // wait until the keyboard is disconnected
        while(keyboard.connected())
            Thread::wait(500);
    }
}

//static void File_ReadAllBytes(byte* bytes, unsigned int size, const char* filename)
//{
//	int file;
//	file = open(filename, O_RDONLY);
//	if (file < 0)
//	{
//		pc.printf("file ""%s"", errno=%d\n", filename, errno);
//		exit(1);
//	}
//	read(file, bytes, size);
//	close(file);
//}

// returns true if BASIC
//static bool LoadPRG(const char* filename)
//{
//	bool result;
//	byte lo, hi;
//	int file;
//	ushort loadaddr;
//	
//	file = open(filename, O_RDONLY);
//	if (file < 0
//		|| read(file, &lo, 1) != 1
//		|| read(file, &hi, 1) != 1
//		)
//	{
//		pc.printf("file ""%s"", errno=%d\n", filename, errno);
//		exit(1);
//	}
//	if (lo == 1)
//	{
//		loadaddr = (ushort)(GetMemory(43) | (GetMemory(44) << 8));
//		result = true;
//	}
//	else
//	{
//		loadaddr = (ushort)(lo | (hi << 8));
//		result = false;
//	}
//	while (true)
//	{
//		byte value;
//		if (read(file, &value, 1) == 1)
//			SetMemory(loadaddr++, value);
//		else
//			break;
//	}
//	close(file);
//	return result;
//}

// remembering state because NMI is edge triggered false->true
static bool NMI = false;

bool ExecutePatch(void)
{
	int found_NMI = 0;
	for (int i=0; !found_NMI && i<9; ++i)
		if (scan_codes[i] == 1024+64)
			found_NMI = 1;
	
	if (NMI)
	{
		if (!found_NMI)
			NMI = false; // reset when not pressed
	}
	else if (found_NMI) // newly pressed, detected edge
	{
		NMI = true; // set so won't trigger again until cleared
		Push(HI(PC));
		Push(LO(PC));
		PHP();
		PC = (ushort)(GetMemory(0xFFFA) + (GetMemory(0xFFFB) << 8)); // JMP(NMI)
		return true; // overriden, and PC changed, so caller should reloop before execution to allow breakpoint/trace/ExecutePatch/etc.
	}
	
//	if (PC == 0xFFD2) // CHROUT
//	{
//		CBM_Console_WriteChar((char)A);
//		// fall through to draw character in screen memory too
//	}
//	else if (PC == 0xFFCF) // CHRIN
//	{
//		A = CBM_Console_ReadChar();
//
//		// SetA equivalent for flags
//		Z = (A == 0);
//		N = ((A & 0x80) != 0);
//		C = false;
//
//		// RTS equivalent
//		byte lo = Pop();
//		byte hi = Pop();
//		PC = (ushort)(((hi << 8) | lo) + 1);
//
//		return true; // overriden, and PC changed, so caller should reloop before execution to allow breakpoint/trace/ExecutePatch/etc.
//	}
//	else if (PC == 0xA474) // READY
//	{
//		if (StartupPRG != 0 && strlen(StartupPRG) > 0) // User requested program be loaded at startup
//		{
//			const char* filename = StartupPRG;
//			StartupPRG = 0;
//
//			if (LoadPRG(filename))
//			{
//				//UNNEW that I used in late 1980s, should work well for loang a program too, probably gleaned from BASIC ROM
//				//ldy #0
//				//lda #1
//				//sta(43),y
//				//iny
//				//sta(43),y
//				//jsr $a533 ; LINKPRG
//				//clc
//				//lda $22
//				//adc #2
//				//sta 45
//				//lda $23
//				//adc #0
//				//sta 46
//				//lda #0
//				//jsr $a65e ; CLEAR/CLR
//				//jmp $a474 ; READY
//
//				// This part shouldn't be necessary as we have loaded, not recovering from NEW, bytes should still be there
//				ushort addr = (ushort)(GetMemory(43) | (GetMemory(44) << 8));
//				SetMemory(addr, 1);
//				SetMemory((ushort)(addr + 1), 1);
//
//				// JSR equivalent
//				ushort retaddr = (ushort)(PC - 1);
//				Push(HI(retaddr));
//				Push(LO(retaddr));
//				PC = 0xA533; // LINKPRG
//
//				//startup_state = 1; // should be able to regain control when returns...
//
//				return true; // overriden, and PC changed, so caller should reloop before execution to allow breakpoint/trace/ExecutePatch/etc.
//			}
//		}
//		else if (startup_state == 1)
//		{
//			ushort addr = (ushort)(GetMemory(0x22) | (GetMemory(0x23) << 8) + 2);
//			SetMemory(45, (byte)addr);
//			SetMemory(46, (byte)(addr >> 8));
//
//			// JSR equivalent
//			ushort retaddr = (ushort)(PC - 1);
//			Push(HI(retaddr));
//			Push(LO(retaddr));
//			PC = 0xA65E; // CLEAR/CLR
//			A = 0;
//
//			startup_state = 2; // should be able to regain control when returns...
//
//			return true; // overriden, and PC changed, so caller should reloop before execution to allow breakpoint/trace/ExecutePatch/etc.
//		}
//		else if (startup_state == 2)
//		{
//			CBM_Console_Push("RUN\r");
//			PC = 0xA47B; // skip READY message, but still set direct mode, and continue to MAIN
//			C = false;
//			startup_state = 0;
//			return true; // overriden, and PC changed, so caller should reloop before execution to allow breakpoint/trace/ExecutePatch/etc.
//		}
//	}
	return false; // execute normally
}

static const byte basic_rom[8 * 1024] = {
'\x94','\xE3','\x7B','\xE3','\x43','\x42','\x4D','\x42','\x41','\x53','\x49','\x43','\x30','\xA8','\x41','\xA7','\x1D','\xAD','\xF7','\xA8','\xA4','\xAB','\xBE','\xAB','\x80','\xB0','\x05','\xAC','\xA4','\xA9','\x9F','\xA8',
'\x70','\xA8','\x27','\xA9','\x1C','\xA8','\x82','\xA8','\xD1','\xA8','\x3A','\xA9','\x2E','\xA8','\x4A','\xA9','\x2C','\xB8','\x67','\xE1','\x55','\xE1','\x64','\xE1','\xB2','\xB3','\x23','\xB8','\x7F','\xAA','\x9F','\xAA',
'\x56','\xA8','\x9B','\xA6','\x5D','\xA6','\x85','\xAA','\x29','\xE1','\xBD','\xE1','\xC6','\xE1','\x7A','\xAB','\x41','\xA6','\x39','\xBC','\xCC','\xBC','\x58','\xBC','\x10','\x03','\x7D','\xB3','\x9E','\xB3','\x71','\xBF',
'\x97','\xE0','\xEA','\xB9','\xED','\xBF','\x64','\xE2','\x6B','\xE2','\xB4','\xE2','\x0E','\xE3','\x0D','\xB8','\x7C','\xB7','\x65','\xB4','\xAD','\xB7','\x8B','\xB7','\xEC','\xB6','\x00','\xB7','\x2C','\xB7','\x37','\xB7',
'\x79','\x69','\xB8','\x79','\x52','\xB8','\x7B','\x2A','\xBA','\x7B','\x11','\xBB','\x7F','\x7A','\xBF','\x50','\xE8','\xAF','\x46','\xE5','\xAF','\x7D','\xB3','\xBF','\x5A','\xD3','\xAE','\x64','\x15','\xB0','\x45','\x4E',
'\xC4','\x46','\x4F','\xD2','\x4E','\x45','\x58','\xD4','\x44','\x41','\x54','\xC1','\x49','\x4E','\x50','\x55','\x54','\xA3','\x49','\x4E','\x50','\x55','\xD4','\x44','\x49','\xCD','\x52','\x45','\x41','\xC4','\x4C','\x45',
'\xD4','\x47','\x4F','\x54','\xCF','\x52','\x55','\xCE','\x49','\xC6','\x52','\x45','\x53','\x54','\x4F','\x52','\xC5','\x47','\x4F','\x53','\x55','\xC2','\x52','\x45','\x54','\x55','\x52','\xCE','\x52','\x45','\xCD','\x53',
'\x54','\x4F','\xD0','\x4F','\xCE','\x57','\x41','\x49','\xD4','\x4C','\x4F','\x41','\xC4','\x53','\x41','\x56','\xC5','\x56','\x45','\x52','\x49','\x46','\xD9','\x44','\x45','\xC6','\x50','\x4F','\x4B','\xC5','\x50','\x52',
'\x49','\x4E','\x54','\xA3','\x50','\x52','\x49','\x4E','\xD4','\x43','\x4F','\x4E','\xD4','\x4C','\x49','\x53','\xD4','\x43','\x4C','\xD2','\x43','\x4D','\xC4','\x53','\x59','\xD3','\x4F','\x50','\x45','\xCE','\x43','\x4C',
'\x4F','\x53','\xC5','\x47','\x45','\xD4','\x4E','\x45','\xD7','\x54','\x41','\x42','\xA8','\x54','\xCF','\x46','\xCE','\x53','\x50','\x43','\xA8','\x54','\x48','\x45','\xCE','\x4E','\x4F','\xD4','\x53','\x54','\x45','\xD0',
'\xAB','\xAD','\xAA','\xAF','\xDE','\x41','\x4E','\xC4','\x4F','\xD2','\xBE','\xBD','\xBC','\x53','\x47','\xCE','\x49','\x4E','\xD4','\x41','\x42','\xD3','\x55','\x53','\xD2','\x46','\x52','\xC5','\x50','\x4F','\xD3','\x53',
'\x51','\xD2','\x52','\x4E','\xC4','\x4C','\x4F','\xC7','\x45','\x58','\xD0','\x43','\x4F','\xD3','\x53','\x49','\xCE','\x54','\x41','\xCE','\x41','\x54','\xCE','\x50','\x45','\x45','\xCB','\x4C','\x45','\xCE','\x53','\x54',
'\x52','\xA4','\x56','\x41','\xCC','\x41','\x53','\xC3','\x43','\x48','\x52','\xA4','\x4C','\x45','\x46','\x54','\xA4','\x52','\x49','\x47','\x48','\x54','\xA4','\x4D','\x49','\x44','\xA4','\x47','\xCF','\x00','\x54','\x4F',
'\x4F','\x20','\x4D','\x41','\x4E','\x59','\x20','\x46','\x49','\x4C','\x45','\xD3','\x46','\x49','\x4C','\x45','\x20','\x4F','\x50','\x45','\xCE','\x46','\x49','\x4C','\x45','\x20','\x4E','\x4F','\x54','\x20','\x4F','\x50',
'\x45','\xCE','\x46','\x49','\x4C','\x45','\x20','\x4E','\x4F','\x54','\x20','\x46','\x4F','\x55','\x4E','\xC4','\x44','\x45','\x56','\x49','\x43','\x45','\x20','\x4E','\x4F','\x54','\x20','\x50','\x52','\x45','\x53','\x45',
'\x4E','\xD4','\x4E','\x4F','\x54','\x20','\x49','\x4E','\x50','\x55','\x54','\x20','\x46','\x49','\x4C','\xC5','\x4E','\x4F','\x54','\x20','\x4F','\x55','\x54','\x50','\x55','\x54','\x20','\x46','\x49','\x4C','\xC5','\x4D',
'\x49','\x53','\x53','\x49','\x4E','\x47','\x20','\x46','\x49','\x4C','\x45','\x20','\x4E','\x41','\x4D','\xC5','\x49','\x4C','\x4C','\x45','\x47','\x41','\x4C','\x20','\x44','\x45','\x56','\x49','\x43','\x45','\x20','\x4E',
'\x55','\x4D','\x42','\x45','\xD2','\x4E','\x45','\x58','\x54','\x20','\x57','\x49','\x54','\x48','\x4F','\x55','\x54','\x20','\x46','\x4F','\xD2','\x53','\x59','\x4E','\x54','\x41','\xD8','\x52','\x45','\x54','\x55','\x52',
'\x4E','\x20','\x57','\x49','\x54','\x48','\x4F','\x55','\x54','\x20','\x47','\x4F','\x53','\x55','\xC2','\x4F','\x55','\x54','\x20','\x4F','\x46','\x20','\x44','\x41','\x54','\xC1','\x49','\x4C','\x4C','\x45','\x47','\x41',
'\x4C','\x20','\x51','\x55','\x41','\x4E','\x54','\x49','\x54','\xD9','\x4F','\x56','\x45','\x52','\x46','\x4C','\x4F','\xD7','\x4F','\x55','\x54','\x20','\x4F','\x46','\x20','\x4D','\x45','\x4D','\x4F','\x52','\xD9','\x55',
'\x4E','\x44','\x45','\x46','\x27','\x44','\x20','\x53','\x54','\x41','\x54','\x45','\x4D','\x45','\x4E','\xD4','\x42','\x41','\x44','\x20','\x53','\x55','\x42','\x53','\x43','\x52','\x49','\x50','\xD4','\x52','\x45','\x44',
'\x49','\x4D','\x27','\x44','\x20','\x41','\x52','\x52','\x41','\xD9','\x44','\x49','\x56','\x49','\x53','\x49','\x4F','\x4E','\x20','\x42','\x59','\x20','\x5A','\x45','\x52','\xCF','\x49','\x4C','\x4C','\x45','\x47','\x41',
'\x4C','\x20','\x44','\x49','\x52','\x45','\x43','\xD4','\x54','\x59','\x50','\x45','\x20','\x4D','\x49','\x53','\x4D','\x41','\x54','\x43','\xC8','\x53','\x54','\x52','\x49','\x4E','\x47','\x20','\x54','\x4F','\x4F','\x20',
'\x4C','\x4F','\x4E','\xC7','\x46','\x49','\x4C','\x45','\x20','\x44','\x41','\x54','\xC1','\x46','\x4F','\x52','\x4D','\x55','\x4C','\x41','\x20','\x54','\x4F','\x4F','\x20','\x43','\x4F','\x4D','\x50','\x4C','\x45','\xD8',
'\x43','\x41','\x4E','\x27','\x54','\x20','\x43','\x4F','\x4E','\x54','\x49','\x4E','\x55','\xC5','\x55','\x4E','\x44','\x45','\x46','\x27','\x44','\x20','\x46','\x55','\x4E','\x43','\x54','\x49','\x4F','\xCE','\x56','\x45',
'\x52','\x49','\x46','\xD9','\x4C','\x4F','\x41','\xC4','\x9E','\xA1','\xAC','\xA1','\xB5','\xA1','\xC2','\xA1','\xD0','\xA1','\xE2','\xA1','\xF0','\xA1','\xFF','\xA1','\x10','\xA2','\x25','\xA2','\x35','\xA2','\x3B','\xA2',
'\x4F','\xA2','\x5A','\xA2','\x6A','\xA2','\x72','\xA2','\x7F','\xA2','\x90','\xA2','\x9D','\xA2','\xAA','\xA2','\xBA','\xA2','\xC8','\xA2','\xD5','\xA2','\xE4','\xA2','\xED','\xA2','\x00','\xA3','\x0E','\xA3','\x1E','\xA3',
'\x24','\xA3','\x83','\xA3','\x0D','\x4F','\x4B','\x0D','\x00','\x20','\x20','\x45','\x52','\x52','\x4F','\x52','\x00','\x20','\x49','\x4E','\x20','\x00','\x0D','\x0A','\x52','\x45','\x41','\x44','\x59','\x2E','\x0D','\x0A',
'\x00','\x0D','\x0A','\x42','\x52','\x45','\x41','\x4B','\x00','\xA0','\xBA','\xE8','\xE8','\xE8','\xE8','\xBD','\x01','\x01','\xC9','\x81','\xD0','\x21','\xA5','\x4A','\xD0','\x0A','\xBD','\x02','\x01','\x85','\x49','\xBD',
'\x03','\x01','\x85','\x4A','\xDD','\x03','\x01','\xD0','\x07','\xA5','\x49','\xDD','\x02','\x01','\xF0','\x07','\x8A','\x18','\x69','\x12','\xAA','\xD0','\xD8','\x60','\x20','\x08','\xA4','\x85','\x31','\x84','\x32','\x38',
'\xA5','\x5A','\xE5','\x5F','\x85','\x22','\xA8','\xA5','\x5B','\xE5','\x60','\xAA','\xE8','\x98','\xF0','\x23','\xA5','\x5A','\x38','\xE5','\x22','\x85','\x5A','\xB0','\x03','\xC6','\x5B','\x38','\xA5','\x58','\xE5','\x22',
'\x85','\x58','\xB0','\x08','\xC6','\x59','\x90','\x04','\xB1','\x5A','\x91','\x58','\x88','\xD0','\xF9','\xB1','\x5A','\x91','\x58','\xC6','\x5B','\xC6','\x59','\xCA','\xD0','\xF2','\x60','\x0A','\x69','\x3E','\xB0','\x35',
'\x85','\x22','\xBA','\xE4','\x22','\x90','\x2E','\x60','\xC4','\x34','\x90','\x28','\xD0','\x04','\xC5','\x33','\x90','\x22','\x48','\xA2','\x09','\x98','\x48','\xB5','\x57','\xCA','\x10','\xFA','\x20','\x26','\xB5','\xA2',
'\xF7','\x68','\x95','\x61','\xE8','\x30','\xFA','\x68','\xA8','\x68','\xC4','\x34','\x90','\x06','\xD0','\x05','\xC5','\x33','\xB0','\x01','\x60','\xA2','\x10','\x6C','\x00','\x03','\x8A','\x0A','\xAA','\xBD','\x26','\xA3',
'\x85','\x22','\xBD','\x27','\xA3','\x85','\x23','\x20','\xCC','\xFF','\xA9','\x00','\x85','\x13','\x20','\xD7','\xAA','\x20','\x45','\xAB','\xA0','\x00','\xB1','\x22','\x48','\x29','\x7F','\x20','\x47','\xAB','\xC8','\x68',
'\x10','\xF4','\x20','\x7A','\xA6','\xA9','\x69','\xA0','\xA3','\x20','\x1E','\xAB','\xA4','\x3A','\xC8','\xF0','\x03','\x20','\xC2','\xBD','\xA9','\x76','\xA0','\xA3','\x20','\x1E','\xAB','\xA9','\x80','\x20','\x90','\xFF',
'\x6C','\x02','\x03','\x20','\x60','\xA5','\x86','\x7A','\x84','\x7B','\x20','\x73','\x00','\xAA','\xF0','\xF0','\xA2','\xFF','\x86','\x3A','\x90','\x06','\x20','\x79','\xA5','\x4C','\xE1','\xA7','\x20','\x6B','\xA9','\x20',
'\x79','\xA5','\x84','\x0B','\x20','\x13','\xA6','\x90','\x44','\xA0','\x01','\xB1','\x5F','\x85','\x23','\xA5','\x2D','\x85','\x22','\xA5','\x60','\x85','\x25','\xA5','\x5F','\x88','\xF1','\x5F','\x18','\x65','\x2D','\x85',
'\x2D','\x85','\x24','\xA5','\x2E','\x69','\xFF','\x85','\x2E','\xE5','\x60','\xAA','\x38','\xA5','\x5F','\xE5','\x2D','\xA8','\xB0','\x03','\xE8','\xC6','\x25','\x18','\x65','\x22','\x90','\x03','\xC6','\x23','\x18','\xB1',
'\x22','\x91','\x24','\xC8','\xD0','\xF9','\xE6','\x23','\xE6','\x25','\xCA','\xD0','\xF2','\x20','\x59','\xA6','\x20','\x33','\xA5','\xAD','\x00','\x02','\xF0','\x88','\x18','\xA5','\x2D','\x85','\x5A','\x65','\x0B','\x85',
'\x58','\xA4','\x2E','\x84','\x5B','\x90','\x01','\xC8','\x84','\x59','\x20','\xB8','\xA3','\xA5','\x14','\xA4','\x15','\x8D','\xFE','\x01','\x8C','\xFF','\x01','\xA5','\x31','\xA4','\x32','\x85','\x2D','\x84','\x2E','\xA4',
'\x0B','\x88','\xB9','\xFC','\x01','\x91','\x5F','\x88','\x10','\xF8','\x20','\x59','\xA6','\x20','\x33','\xA5','\x4C','\x80','\xA4','\xA5','\x2B','\xA4','\x2C','\x85','\x22','\x84','\x23','\x18','\xA0','\x01','\xB1','\x22',
'\xF0','\x1D','\xA0','\x04','\xC8','\xB1','\x22','\xD0','\xFB','\xC8','\x98','\x65','\x22','\xAA','\xA0','\x00','\x91','\x22','\xA5','\x23','\x69','\x00','\xC8','\x91','\x22','\x86','\x22','\x85','\x23','\x90','\xDD','\x60',
'\xA2','\x00','\x20','\x12','\xE1','\xC9','\x0D','\xF0','\x0D','\x9D','\x00','\x02','\xE8','\xE0','\x59','\x90','\xF1','\xA2','\x17','\x4C','\x37','\xA4','\x4C','\xCA','\xAA','\x6C','\x04','\x03','\xA6','\x7A','\xA0','\x04',
'\x84','\x0F','\xBD','\x00','\x02','\x10','\x07','\xC9','\xFF','\xF0','\x3E','\xE8','\xD0','\xF4','\xC9','\x20','\xF0','\x37','\x85','\x08','\xC9','\x22','\xF0','\x56','\x24','\x0F','\x70','\x2D','\xC9','\x3F','\xD0','\x04',
'\xA9','\x99','\xD0','\x25','\xC9','\x30','\x90','\x04','\xC9','\x3C','\x90','\x1D','\x84','\x71','\xA0','\x00','\x84','\x0B','\x88','\x86','\x7A','\xCA','\xC8','\xE8','\xBD','\x00','\x02','\x38','\xF9','\x9E','\xA0','\xF0',
'\xF5','\xC9','\x80','\xD0','\x30','\x05','\x0B','\xA4','\x71','\xE8','\xC8','\x99','\xFB','\x01','\xB9','\xFB','\x01','\xF0','\x36','\x38','\xE9','\x3A','\xF0','\x04','\xC9','\x49','\xD0','\x02','\x85','\x0F','\x38','\xE9',
'\x55','\xD0','\x9F','\x85','\x08','\xBD','\x00','\x02','\xF0','\xDF','\xC5','\x08','\xF0','\xDB','\xC8','\x99','\xFB','\x01','\xE8','\xD0','\xF0','\xA6','\x7A','\xE6','\x0B','\xC8','\xB9','\x9D','\xA0','\x10','\xFA','\xB9',
'\x9E','\xA0','\xD0','\xB4','\xBD','\x00','\x02','\x10','\xBE','\x99','\xFD','\x01','\xC6','\x7B','\xA9','\xFF','\x85','\x7A','\x60','\xA5','\x2B','\xA6','\x2C','\xA0','\x01','\x85','\x5F','\x86','\x60','\xB1','\x5F','\xF0',
'\x1F','\xC8','\xC8','\xA5','\x15','\xD1','\x5F','\x90','\x18','\xF0','\x03','\x88','\xD0','\x09','\xA5','\x14','\x88','\xD1','\x5F','\x90','\x0C','\xF0','\x0A','\x88','\xB1','\x5F','\xAA','\x88','\xB1','\x5F','\xB0','\xD7',
'\x18','\x60','\xD0','\xFD','\xA9','\x00','\xA8','\x91','\x2B','\xC8','\x91','\x2B','\xA5','\x2B','\x18','\x69','\x02','\x85','\x2D','\xA5','\x2C','\x69','\x00','\x85','\x2E','\x20','\x8E','\xA6','\xA9','\x00','\xD0','\x2D',
'\x20','\xE7','\xFF','\xA5','\x37','\xA4','\x38','\x85','\x33','\x84','\x34','\xA5','\x2D','\xA4','\x2E','\x85','\x2F','\x84','\x30','\x85','\x31','\x84','\x32','\x20','\x1D','\xA8','\xA2','\x19','\x86','\x16','\x68','\xA8',
'\x68','\xA2','\xFA','\x9A','\x48','\x98','\x48','\xA9','\x00','\x85','\x3E','\x85','\x10','\x60','\x18','\xA5','\x2B','\x69','\xFF','\x85','\x7A','\xA5','\x2C','\x69','\xFF','\x85','\x7B','\x60','\x90','\x06','\xF0','\x04',
'\xC9','\xAB','\xD0','\xE9','\x20','\x6B','\xA9','\x20','\x13','\xA6','\x20','\x79','\x00','\xF0','\x0C','\xC9','\xAB','\xD0','\x8E','\x20','\x73','\x00','\x20','\x6B','\xA9','\xD0','\x86','\x68','\x68','\xA5','\x14','\x05',
'\x15','\xD0','\x06','\xA9','\xFF','\x85','\x14','\x85','\x15','\xA0','\x01','\x84','\x0F','\xB1','\x5F','\xF0','\x43','\x20','\x2C','\xA8','\x20','\xD7','\xAA','\xC8','\xB1','\x5F','\xAA','\xC8','\xB1','\x5F','\xC5','\x15',
'\xD0','\x04','\xE4','\x14','\xF0','\x02','\xB0','\x2C','\x84','\x49','\x20','\xCD','\xBD','\xA9','\x20','\xA4','\x49','\x29','\x7F','\x20','\x47','\xAB','\xC9','\x22','\xD0','\x06','\xA5','\x0F','\x49','\xFF','\x85','\x0F',
'\xC8','\xF0','\x11','\xB1','\x5F','\xD0','\x10','\xA8','\xB1','\x5F','\xAA','\xC8','\xB1','\x5F','\x86','\x5F','\x85','\x60','\xD0','\xB5','\x4C','\x86','\xE3','\x6C','\x06','\x03','\x10','\xD7','\xC9','\xFF','\xF0','\xD3',
'\x24','\x0F','\x30','\xCF','\x38','\xE9','\x7F','\xAA','\x84','\x49','\xA0','\xFF','\xCA','\xF0','\x08','\xC8','\xB9','\x9E','\xA0','\x10','\xFA','\x30','\xF5','\xC8','\xB9','\x9E','\xA0','\x30','\xB2','\x20','\x47','\xAB',
'\xD0','\xF5','\xA9','\x80','\x85','\x10','\x20','\xA5','\xA9','\x20','\x8A','\xA3','\xD0','\x05','\x8A','\x69','\x0F','\xAA','\x9A','\x68','\x68','\xA9','\x09','\x20','\xFB','\xA3','\x20','\x06','\xA9','\x18','\x98','\x65',
'\x7A','\x48','\xA5','\x7B','\x69','\x00','\x48','\xA5','\x3A','\x48','\xA5','\x39','\x48','\xA9','\xA4','\x20','\xFF','\xAE','\x20','\x8D','\xAD','\x20','\x8A','\xAD','\xA5','\x66','\x09','\x7F','\x25','\x62','\x85','\x62',
'\xA9','\x8B','\xA0','\xA7','\x85','\x22','\x84','\x23','\x4C','\x43','\xAE','\xA9','\xBC','\xA0','\xB9','\x20','\xA2','\xBB','\x20','\x79','\x00','\xC9','\xA9','\xD0','\x06','\x20','\x73','\x00','\x20','\x8A','\xAD','\x20',
'\x2B','\xBC','\x20','\x38','\xAE','\xA5','\x4A','\x48','\xA5','\x49','\x48','\xA9','\x81','\x48','\x20','\x2C','\xA8','\xA5','\x7A','\xA4','\x7B','\xC0','\x02','\xEA','\xF0','\x04','\x85','\x3D','\x84','\x3E','\xA0','\x00',
'\xB1','\x7A','\xD0','\x43','\xA0','\x02','\xB1','\x7A','\x18','\xD0','\x03','\x4C','\x4B','\xA8','\xC8','\xB1','\x7A','\x85','\x39','\xC8','\xB1','\x7A','\x85','\x3A','\x98','\x65','\x7A','\x85','\x7A','\x90','\x02','\xE6',
'\x7B','\x6C','\x08','\x03','\x20','\x73','\x00','\x20','\xED','\xA7','\x4C','\xAE','\xA7','\xF0','\x3C','\xE9','\x80','\x90','\x11','\xC9','\x23','\xB0','\x17','\x0A','\xA8','\xB9','\x0D','\xA0','\x48','\xB9','\x0C','\xA0',
'\x48','\x4C','\x73','\x00','\x4C','\xA5','\xA9','\xC9','\x3A','\xF0','\xD6','\x4C','\x08','\xAF','\xC9','\x4B','\xD0','\xF9','\x20','\x73','\x00','\xA9','\xA4','\x20','\xFF','\xAE','\x4C','\xA0','\xA8','\x38','\xA5','\x2B',
'\xE9','\x01','\xA4','\x2C','\xB0','\x01','\x88','\x85','\x41','\x84','\x42','\x60','\x20','\xE1','\xFF','\xB0','\x01','\x18','\xD0','\x3C','\xA5','\x7A','\xA4','\x7B','\xA6','\x3A','\xE8','\xF0','\x0C','\x85','\x3D','\x84',
'\x3E','\xA5','\x39','\xA4','\x3A','\x85','\x3B','\x84','\x3C','\x68','\x68','\xA9','\x81','\xA0','\xA3','\x90','\x03','\x4C','\x69','\xA4','\x4C','\x86','\xE3','\xD0','\x17','\xA2','\x1A','\xA4','\x3E','\xD0','\x03','\x4C',
'\x37','\xA4','\xA5','\x3D','\x85','\x7A','\x84','\x7B','\xA5','\x3B','\xA4','\x3C','\x85','\x39','\x84','\x3A','\x60','\x08','\xA9','\x00','\x20','\x90','\xFF','\x28','\xD0','\x03','\x4C','\x59','\xA6','\x20','\x60','\xA6',
'\x4C','\x97','\xA8','\xA9','\x03','\x20','\xFB','\xA3','\xA5','\x7B','\x48','\xA5','\x7A','\x48','\xA5','\x3A','\x48','\xA5','\x39','\x48','\xA9','\x8D','\x48','\x20','\x79','\x00','\x20','\xA0','\xA8','\x4C','\xAE','\xA7',
'\x20','\x6B','\xA9','\x20','\x09','\xA9','\x38','\xA5','\x39','\xE5','\x14','\xA5','\x3A','\xE5','\x15','\xB0','\x0B','\x98','\x38','\x65','\x7A','\xA6','\x7B','\x90','\x07','\xE8','\xB0','\x04','\xA5','\x2B','\xA6','\x2C',
'\x20','\x17','\xA6','\x90','\x1E','\xA5','\x5F','\xE9','\x01','\x85','\x7A','\xA5','\x60','\xE9','\x00','\x85','\x7B','\x60','\xD0','\xFD','\xA9','\xFF','\x85','\x4A','\x20','\x8A','\xA3','\x9A','\xC9','\x8D','\xF0','\x0B',
'\xA2','\x0C','\x2C','\xA2','\x11','\x4C','\x37','\xA4','\x4C','\x08','\xAF','\x68','\x68','\x85','\x39','\x68','\x85','\x3A','\x68','\x85','\x7A','\x68','\x85','\x7B','\x20','\x06','\xA9','\x98','\x18','\x65','\x7A','\x85',
'\x7A','\x90','\x02','\xE6','\x7B','\x60','\xA2','\x3A','\x2C','\xA2','\x00','\x86','\x07','\xA0','\x00','\x84','\x08','\xA5','\x08','\xA6','\x07','\x85','\x07','\x86','\x08','\xB1','\x7A','\xF0','\xE8','\xC5','\x08','\xF0',
'\xE4','\xC8','\xC9','\x22','\xD0','\xF3','\xF0','\xE9','\x20','\x9E','\xAD','\x20','\x79','\x00','\xC9','\x89','\xF0','\x05','\xA9','\xA7','\x20','\xFF','\xAE','\xA5','\x61','\xD0','\x05','\x20','\x09','\xA9','\xF0','\xBB',
'\x20','\x79','\x00','\xB0','\x03','\x4C','\xA0','\xA8','\x4C','\xED','\xA7','\x20','\x9E','\xB7','\x48','\xC9','\x8D','\xF0','\x04','\xC9','\x89','\xD0','\x91','\xC6','\x65','\xD0','\x04','\x68','\x4C','\xEF','\xA7','\x20',
'\x73','\x00','\x20','\x6B','\xA9','\xC9','\x2C','\xF0','\xEE','\x68','\x60','\xA2','\x00','\x86','\x14','\x86','\x15','\xB0','\xF7','\xE9','\x2F','\x85','\x07','\xA5','\x15','\x85','\x22','\xC9','\x19','\xB0','\xD4','\xA5',
'\x14','\x0A','\x26','\x22','\x0A','\x26','\x22','\x65','\x14','\x85','\x14','\xA5','\x22','\x65','\x15','\x85','\x15','\x06','\x14','\x26','\x15','\xA5','\x14','\x65','\x07','\x85','\x14','\x90','\x02','\xE6','\x15','\x20',
'\x73','\x00','\x4C','\x71','\xA9','\x20','\x8B','\xB0','\x85','\x49','\x84','\x4A','\xA9','\xB2','\x20','\xFF','\xAE','\xA5','\x0E','\x48','\xA5','\x0D','\x48','\x20','\x9E','\xAD','\x68','\x2A','\x20','\x90','\xAD','\xD0',
'\x18','\x68','\x10','\x12','\x20','\x1B','\xBC','\x20','\xBF','\xB1','\xA0','\x00','\xA5','\x64','\x91','\x49','\xC8','\xA5','\x65','\x91','\x49','\x60','\x4C','\xD0','\xBB','\x68','\xA4','\x4A','\xC0','\xBF','\xD0','\x4C',
'\x20','\xA6','\xB6','\xC9','\x06','\xD0','\x3D','\xA0','\x00','\x84','\x61','\x84','\x66','\x84','\x71','\x20','\x1D','\xAA','\x20','\xE2','\xBA','\xE6','\x71','\xA4','\x71','\x20','\x1D','\xAA','\x20','\x0C','\xBC','\xAA',
'\xF0','\x05','\xE8','\x8A','\x20','\xED','\xBA','\xA4','\x71','\xC8','\xC0','\x06','\xD0','\xDF','\x20','\xE2','\xBA','\x20','\x9B','\xBC','\xA6','\x64','\xA4','\x63','\xA5','\x65','\x4C','\xDB','\xFF','\xB1','\x22','\x20',
'\x80','\x00','\x90','\x03','\x4C','\x48','\xB2','\xE9','\x2F','\x4C','\x7E','\xBD','\xA0','\x02','\xB1','\x64','\xC5','\x34','\x90','\x17','\xD0','\x07','\x88','\xB1','\x64','\xC5','\x33','\x90','\x0E','\xA4','\x65','\xC4',
'\x2E','\x90','\x08','\xD0','\x0D','\xA5','\x64','\xC5','\x2D','\xB0','\x07','\xA5','\x64','\xA4','\x65','\x4C','\x68','\xAA','\xA0','\x00','\xB1','\x64','\x20','\x75','\xB4','\xA5','\x50','\xA4','\x51','\x85','\x6F','\x84',
'\x70','\x20','\x7A','\xB6','\xA9','\x61','\xA0','\x00','\x85','\x50','\x84','\x51','\x20','\xDB','\xB6','\xA0','\x00','\xB1','\x50','\x91','\x49','\xC8','\xB1','\x50','\x91','\x49','\xC8','\xB1','\x50','\x91','\x49','\x60',
'\x20','\x86','\xAA','\x4C','\xB5','\xAB','\x20','\x9E','\xB7','\xF0','\x05','\xA9','\x2C','\x20','\xFF','\xAE','\x08','\x86','\x13','\x20','\x18','\xE1','\x28','\x4C','\xA0','\xAA','\x20','\x21','\xAB','\x20','\x79','\x00',
'\xF0','\x35','\xF0','\x43','\xC9','\xA3','\xF0','\x50','\xC9','\xA6','\x18','\xF0','\x4B','\xC9','\x2C','\xF0','\x37','\xC9','\x3B','\xF0','\x5E','\x20','\x9E','\xAD','\x24','\x0D','\x30','\xDE','\x20','\xDD','\xBD','\x20',
'\x87','\xB4','\x20','\x21','\xAB','\x20','\x3B','\xAB','\xD0','\xD3','\xA9','\x00','\x9D','\x00','\x02','\xA2','\xFF','\xA0','\x01','\xA5','\x13','\xD0','\x10','\xA9','\x0D','\x20','\x47','\xAB','\x24','\x13','\x10','\x05',
'\xA9','\x0A','\x20','\x47','\xAB','\x49','\xFF','\x60','\x38','\x20','\xF0','\xFF','\x98','\x38','\xE9','\x0A','\xB0','\xFC','\x49','\xFF','\x69','\x01','\xD0','\x16','\x08','\x38','\x20','\xF0','\xFF','\x84','\x09','\x20',
'\x9B','\xB7','\xC9','\x29','\xD0','\x59','\x28','\x90','\x06','\x8A','\xE5','\x09','\x90','\x05','\xAA','\xE8','\xCA','\xD0','\x06','\x20','\x73','\x00','\x4C','\xA2','\xAA','\x20','\x3B','\xAB','\xD0','\xF2','\x20','\x87',
'\xB4','\x20','\xA6','\xB6','\xAA','\xA0','\x00','\xE8','\xCA','\xF0','\xBC','\xB1','\x22','\x20','\x47','\xAB','\xC8','\xC9','\x0D','\xD0','\xF3','\x20','\xE5','\xAA','\x4C','\x28','\xAB','\xA5','\x13','\xF0','\x03','\xA9',
'\x20','\x2C','\xA9','\x1D','\x2C','\xA9','\x3F','\x20','\x0C','\xE1','\x29','\xFF','\x60','\xA5','\x11','\xF0','\x11','\x30','\x04','\xA0','\xFF','\xD0','\x04','\xA5','\x3F','\xA4','\x40','\x85','\x39','\x84','\x3A','\x4C',
'\x08','\xAF','\xA5','\x13','\xF0','\x05','\xA2','\x18','\x4C','\x37','\xA4','\xA9','\x0C','\xA0','\xAD','\x20','\x1E','\xAB','\xA5','\x3D','\xA4','\x3E','\x85','\x7A','\x84','\x7B','\x60','\x20','\xA6','\xB3','\xC9','\x23',
'\xD0','\x10','\x20','\x73','\x00','\x20','\x9E','\xB7','\xA9','\x2C','\x20','\xFF','\xAE','\x86','\x13','\x20','\x1E','\xE1','\xA2','\x01','\xA0','\x02','\xA9','\x00','\x8D','\x01','\x02','\xA9','\x40','\x20','\x0F','\xAC',
'\xA6','\x13','\xD0','\x13','\x60','\x20','\x9E','\xB7','\xA9','\x2C','\x20','\xFF','\xAE','\x86','\x13','\x20','\x1E','\xE1','\x20','\xCE','\xAB','\xA5','\x13','\x20','\xCC','\xFF','\xA2','\x00','\x86','\x13','\x60','\xC9',
'\x22','\xD0','\x0B','\x20','\xBD','\xAE','\xA9','\x3B','\x20','\xFF','\xAE','\x20','\x21','\xAB','\x20','\xA6','\xB3','\xA9','\x2C','\x8D','\xFF','\x01','\x20','\xF9','\xAB','\xA5','\x13','\xF0','\x0D','\x20','\xB7','\xFF',
'\x29','\x02','\xF0','\x06','\x20','\xB5','\xAB','\x4C','\xF8','\xA8','\xAD','\x00','\x02','\xD0','\x1E','\xA5','\x13','\xD0','\xE3','\x20','\x06','\xA9','\x4C','\xFB','\xA8','\xA5','\x13','\xD0','\x06','\x20','\x45','\xAB',
'\x20','\x3B','\xAB','\x4C','\x60','\xA5','\xA6','\x41','\xA4','\x42','\xA9','\x98','\x2C','\xA9','\x00','\x85','\x11','\x86','\x43','\x84','\x44','\x20','\x8B','\xB0','\x85','\x49','\x84','\x4A','\xA5','\x7A','\xA4','\x7B',
'\x85','\x4B','\x84','\x4C','\xA6','\x43','\xA4','\x44','\x86','\x7A','\x84','\x7B','\x20','\x79','\x00','\xD0','\x20','\x24','\x11','\x50','\x0C','\x20','\x24','\xE1','\x8D','\x00','\x02','\xA2','\xFF','\xA0','\x01','\xD0',
'\x0C','\x30','\x75','\xA5','\x13','\xD0','\x03','\x20','\x45','\xAB','\x20','\xF9','\xAB','\x86','\x7A','\x84','\x7B','\x20','\x73','\x00','\x24','\x0D','\x10','\x31','\x24','\x11','\x50','\x09','\xE8','\x86','\x7A','\xA9',
'\x00','\x85','\x07','\xF0','\x0C','\x85','\x07','\xC9','\x22','\xF0','\x07','\xA9','\x3A','\x85','\x07','\xA9','\x2C','\x18','\x85','\x08','\xA5','\x7A','\xA4','\x7B','\x69','\x00','\x90','\x01','\xC8','\x20','\x8D','\xB4',
'\x20','\xE2','\xB7','\x20','\xDA','\xA9','\x4C','\x91','\xAC','\x20','\xF3','\xBC','\xA5','\x0E','\x20','\xC2','\xA9','\x20','\x79','\x00','\xF0','\x07','\xC9','\x2C','\xF0','\x03','\x4C','\x4D','\xAB','\xA5','\x7A','\xA4',
'\x7B','\x85','\x43','\x84','\x44','\xA5','\x4B','\xA4','\x4C','\x85','\x7A','\x84','\x7B','\x20','\x79','\x00','\xF0','\x2D','\x20','\xFD','\xAE','\x4C','\x15','\xAC','\x20','\x06','\xA9','\xC8','\xAA','\xD0','\x12','\xA2',
'\x0D','\xC8','\xB1','\x7A','\xF0','\x6C','\xC8','\xB1','\x7A','\x85','\x3F','\xC8','\xB1','\x7A','\xC8','\x85','\x40','\x20','\xFB','\xA8','\x20','\x79','\x00','\xAA','\xE0','\x83','\xD0','\xDC','\x4C','\x51','\xAC','\xA5',
'\x43','\xA4','\x44','\xA6','\x11','\x10','\x03','\x4C','\x27','\xA8','\xA0','\x00','\xB1','\x43','\xF0','\x0B','\xA5','\x13','\xD0','\x07','\xA9','\xFC','\xA0','\xAC','\x4C','\x1E','\xAB','\x60','\x3F','\x45','\x58','\x54',
'\x52','\x41','\x20','\x49','\x47','\x4E','\x4F','\x52','\x45','\x44','\x0D','\x00','\x3F','\x52','\x45','\x44','\x4F','\x20','\x46','\x52','\x4F','\x4D','\x20','\x53','\x54','\x41','\x52','\x54','\x0D','\x00','\xD0','\x04',
'\xA0','\x00','\xF0','\x03','\x20','\x8B','\xB0','\x85','\x49','\x84','\x4A','\x20','\x8A','\xA3','\xF0','\x05','\xA2','\x0A','\x4C','\x37','\xA4','\x9A','\x8A','\x18','\x69','\x04','\x48','\x69','\x06','\x85','\x24','\x68',
'\xA0','\x01','\x20','\xA2','\xBB','\xBA','\xBD','\x09','\x01','\x85','\x66','\xA5','\x49','\xA4','\x4A','\x20','\x67','\xB8','\x20','\xD0','\xBB','\xA0','\x01','\x20','\x5D','\xBC','\xBA','\x38','\xFD','\x09','\x01','\xF0',
'\x17','\xBD','\x0F','\x01','\x85','\x39','\xBD','\x10','\x01','\x85','\x3A','\xBD','\x12','\x01','\x85','\x7A','\xBD','\x11','\x01','\x85','\x7B','\x4C','\xAE','\xA7','\x8A','\x69','\x11','\xAA','\x9A','\x20','\x79','\x00',
'\xC9','\x2C','\xD0','\xF1','\x20','\x73','\x00','\x20','\x24','\xAD','\x20','\x9E','\xAD','\x18','\x24','\x38','\x24','\x0D','\x30','\x03','\xB0','\x03','\x60','\xB0','\xFD','\xA2','\x16','\x4C','\x37','\xA4','\xA6','\x7A',
'\xD0','\x02','\xC6','\x7B','\xC6','\x7A','\xA2','\x00','\x24','\x48','\x8A','\x48','\xA9','\x01','\x20','\xFB','\xA3','\x20','\x83','\xAE','\xA9','\x00','\x85','\x4D','\x20','\x79','\x00','\x38','\xE9','\xB1','\x90','\x17',
'\xC9','\x03','\xB0','\x13','\xC9','\x01','\x2A','\x49','\x01','\x45','\x4D','\xC5','\x4D','\x90','\x61','\x85','\x4D','\x20','\x73','\x00','\x4C','\xBB','\xAD','\xA6','\x4D','\xD0','\x2C','\xB0','\x7B','\x69','\x07','\x90',
'\x77','\x65','\x0D','\xD0','\x03','\x4C','\x3D','\xB6','\x69','\xFF','\x85','\x22','\x0A','\x65','\x22','\xA8','\x68','\xD9','\x80','\xA0','\xB0','\x67','\x20','\x8D','\xAD','\x48','\x20','\x20','\xAE','\x68','\xA4','\x4B',
'\x10','\x17','\xAA','\xF0','\x56','\xD0','\x5F','\x46','\x0D','\x8A','\x2A','\xA6','\x7A','\xD0','\x02','\xC6','\x7B','\xC6','\x7A','\xA0','\x1B','\x85','\x4D','\xD0','\xD7','\xD9','\x80','\xA0','\xB0','\x48','\x90','\xD9',
'\xB9','\x82','\xA0','\x48','\xB9','\x81','\xA0','\x48','\x20','\x33','\xAE','\xA5','\x4D','\x4C','\xA9','\xAD','\x4C','\x08','\xAF','\xA5','\x66','\xBE','\x80','\xA0','\xA8','\x68','\x85','\x22','\xE6','\x22','\x68','\x85',
'\x23','\x98','\x48','\x20','\x1B','\xBC','\xA5','\x65','\x48','\xA5','\x64','\x48','\xA5','\x63','\x48','\xA5','\x62','\x48','\xA5','\x61','\x48','\x6C','\x22','\x00','\xA0','\xFF','\x68','\xF0','\x23','\xC9','\x64','\xF0',
'\x03','\x20','\x8D','\xAD','\x84','\x4B','\x68','\x4A','\x85','\x12','\x68','\x85','\x69','\x68','\x85','\x6A','\x68','\x85','\x6B','\x68','\x85','\x6C','\x68','\x85','\x6D','\x68','\x85','\x6E','\x45','\x66','\x85','\x6F',
'\xA5','\x61','\x60','\x6C','\x0A','\x03','\xA9','\x00','\x85','\x0D','\x20','\x73','\x00','\xB0','\x03','\x4C','\xF3','\xBC','\x20','\x13','\xB1','\x90','\x03','\x4C','\x28','\xAF','\xC9','\xFF','\xD0','\x0F','\xA9','\xA8',
'\xA0','\xAE','\x20','\xA2','\xBB','\x4C','\x73','\x00','\x82','\x49','\x0F','\xDA','\xA1','\xC9','\x2E','\xF0','\xDE','\xC9','\xAB','\xF0','\x58','\xC9','\xAA','\xF0','\xD1','\xC9','\x22','\xD0','\x0F','\xA5','\x7A','\xA4',
'\x7B','\x69','\x00','\x90','\x01','\xC8','\x20','\x87','\xB4','\x4C','\xE2','\xB7','\xC9','\xA8','\xD0','\x13','\xA0','\x18','\xD0','\x3B','\x20','\xBF','\xB1','\xA5','\x65','\x49','\xFF','\xA8','\xA5','\x64','\x49','\xFF',
'\x4C','\x91','\xB3','\xC9','\xA5','\xD0','\x03','\x4C','\xF4','\xB3','\xC9','\xB4','\x90','\x03','\x4C','\xA7','\xAF','\x20','\xFA','\xAE','\x20','\x9E','\xAD','\xA9','\x29','\x2C','\xA9','\x28','\x2C','\xA9','\x2C','\xA0',
'\x00','\xD1','\x7A','\xD0','\x03','\x4C','\x73','\x00','\xA2','\x0B','\x4C','\x37','\xA4','\xA0','\x15','\x68','\x68','\x4C','\xFA','\xAD','\x38','\xA5','\x64','\xE9','\x00','\xA5','\x65','\xE9','\xA0','\x90','\x08','\xA9',
'\xA2','\xE5','\x64','\xA9','\xE3','\xE5','\x65','\x60','\x20','\x8B','\xB0','\x85','\x64','\x84','\x65','\xA6','\x45','\xA4','\x46','\xA5','\x0D','\xF0','\x26','\xA9','\x00','\x85','\x70','\x20','\x14','\xAF','\x90','\x1C',
'\xE0','\x54','\xD0','\x18','\xC0','\xC9','\xD0','\x14','\x20','\x84','\xAF','\x84','\x5E','\x88','\x84','\x71','\xA0','\x06','\x84','\x5D','\xA0','\x24','\x20','\x68','\xBE','\x4C','\x6F','\xB4','\x60','\x24','\x0E','\x10',
'\x0D','\xA0','\x00','\xB1','\x64','\xAA','\xC8','\xB1','\x64','\xA8','\x8A','\x4C','\x91','\xB3','\x20','\x14','\xAF','\x90','\x2D','\xE0','\x54','\xD0','\x1B','\xC0','\x49','\xD0','\x25','\x20','\x84','\xAF','\x98','\xA2',
'\xA0','\x4C','\x4F','\xBC','\x20','\xDE','\xFF','\x86','\x64','\x84','\x63','\x85','\x65','\xA0','\x00','\x84','\x62','\x60','\xE0','\x53','\xD0','\x0A','\xC0','\x54','\xD0','\x06','\x20','\xB7','\xFF','\x4C','\x3C','\xBC',
'\xA5','\x64','\xA4','\x65','\x4C','\xA2','\xBB','\x0A','\x48','\xAA','\x20','\x73','\x00','\xE0','\x8F','\x90','\x20','\x20','\xFA','\xAE','\x20','\x9E','\xAD','\x20','\xFD','\xAE','\x20','\x8F','\xAD','\x68','\xAA','\xA5',
'\x65','\x48','\xA5','\x64','\x48','\x8A','\x48','\x20','\x9E','\xB7','\x68','\xA8','\x8A','\x48','\x4C','\xD6','\xAF','\x20','\xF1','\xAE','\x68','\xA8','\xB9','\xEA','\x9F','\x85','\x55','\xB9','\xEB','\x9F','\x85','\x56',
'\x20','\x54','\x00','\x4C','\x8D','\xAD','\xA0','\xFF','\x2C','\xA0','\x00','\x84','\x0B','\x20','\xBF','\xB1','\xA5','\x64','\x45','\x0B','\x85','\x07','\xA5','\x65','\x45','\x0B','\x85','\x08','\x20','\xFC','\xBB','\x20',
'\xBF','\xB1','\xA5','\x65','\x45','\x0B','\x25','\x08','\x45','\x0B','\xA8','\xA5','\x64','\x45','\x0B','\x25','\x07','\x45','\x0B','\x4C','\x91','\xB3','\x20','\x90','\xAD','\xB0','\x13','\xA5','\x6E','\x09','\x7F','\x25',
'\x6A','\x85','\x6A','\xA9','\x69','\xA0','\x00','\x20','\x5B','\xBC','\xAA','\x4C','\x61','\xB0','\xA9','\x00','\x85','\x0D','\xC6','\x4D','\x20','\xA6','\xB6','\x85','\x61','\x86','\x62','\x84','\x63','\xA5','\x6C','\xA4',
'\x6D','\x20','\xAA','\xB6','\x86','\x6C','\x84','\x6D','\xAA','\x38','\xE5','\x61','\xF0','\x08','\xA9','\x01','\x90','\x04','\xA6','\x61','\xA9','\xFF','\x85','\x66','\xA0','\xFF','\xE8','\xC8','\xCA','\xD0','\x07','\xA6',
'\x66','\x30','\x0F','\x18','\x90','\x0C','\xB1','\x6C','\xD1','\x62','\xF0','\xEF','\xA2','\xFF','\xB0','\x02','\xA2','\x01','\xE8','\x8A','\x2A','\x25','\x12','\xF0','\x02','\xA9','\xFF','\x4C','\x3C','\xBC','\x20','\xFD',
'\xAE','\xAA','\x20','\x90','\xB0','\x20','\x79','\x00','\xD0','\xF4','\x60','\xA2','\x00','\x20','\x79','\x00','\x86','\x0C','\x85','\x45','\x20','\x79','\x00','\x20','\x13','\xB1','\xB0','\x03','\x4C','\x08','\xAF','\xA2',
'\x00','\x86','\x0D','\x86','\x0E','\x20','\x73','\x00','\x90','\x05','\x20','\x13','\xB1','\x90','\x0B','\xAA','\x20','\x73','\x00','\x90','\xFB','\x20','\x13','\xB1','\xB0','\xF6','\xC9','\x24','\xD0','\x06','\xA9','\xFF',
'\x85','\x0D','\xD0','\x10','\xC9','\x25','\xD0','\x13','\xA5','\x10','\xD0','\xD0','\xA9','\x80','\x85','\x0E','\x05','\x45','\x85','\x45','\x8A','\x09','\x80','\xAA','\x20','\x73','\x00','\x86','\x46','\x38','\x05','\x10',
'\xE9','\x28','\xD0','\x03','\x4C','\xD1','\xB1','\xA0','\x00','\x84','\x10','\xA5','\x2D','\xA6','\x2E','\x86','\x60','\x85','\x5F','\xE4','\x30','\xD0','\x04','\xC5','\x2F','\xF0','\x22','\xA5','\x45','\xD1','\x5F','\xD0',
'\x08','\xA5','\x46','\xC8','\xD1','\x5F','\xF0','\x7D','\x88','\x18','\xA5','\x5F','\x69','\x07','\x90','\xE1','\xE8','\xD0','\xDC','\xC9','\x41','\x90','\x05','\xE9','\x5B','\x38','\xE9','\xA5','\x60','\x68','\x48','\xC9',
'\x2A','\xD0','\x05','\xA9','\x13','\xA0','\xBF','\x60','\xA5','\x45','\xA4','\x46','\xC9','\x54','\xD0','\x0B','\xC0','\xC9','\xF0','\xEF','\xC0','\x49','\xD0','\x03','\x4C','\x08','\xAF','\xC9','\x53','\xD0','\x04','\xC0',
'\x54','\xF0','\xF5','\xA5','\x2F','\xA4','\x30','\x85','\x5F','\x84','\x60','\xA5','\x31','\xA4','\x32','\x85','\x5A','\x84','\x5B','\x18','\x69','\x07','\x90','\x01','\xC8','\x85','\x58','\x84','\x59','\x20','\xB8','\xA3',
'\xA5','\x58','\xA4','\x59','\xC8','\x85','\x2F','\x84','\x30','\xA0','\x00','\xA5','\x45','\x91','\x5F','\xC8','\xA5','\x46','\x91','\x5F','\xA9','\x00','\xC8','\x91','\x5F','\xC8','\x91','\x5F','\xC8','\x91','\x5F','\xC8',
'\x91','\x5F','\xC8','\x91','\x5F','\xA5','\x5F','\x18','\x69','\x02','\xA4','\x60','\x90','\x01','\xC8','\x85','\x47','\x84','\x48','\x60','\xA5','\x0B','\x0A','\x69','\x05','\x65','\x5F','\xA4','\x60','\x90','\x01','\xC8',
'\x85','\x58','\x84','\x59','\x60','\x90','\x80','\x00','\x00','\x00','\x20','\xBF','\xB1','\xA5','\x64','\xA4','\x65','\x60','\x20','\x73','\x00','\x20','\x9E','\xAD','\x20','\x8D','\xAD','\xA5','\x66','\x30','\x0D','\xA5',
'\x61','\xC9','\x90','\x90','\x09','\xA9','\xA5','\xA0','\xB1','\x20','\x5B','\xBC','\xD0','\x7A','\x4C','\x9B','\xBC','\xA5','\x0C','\x05','\x0E','\x48','\xA5','\x0D','\x48','\xA0','\x00','\x98','\x48','\xA5','\x46','\x48',
'\xA5','\x45','\x48','\x20','\xB2','\xB1','\x68','\x85','\x45','\x68','\x85','\x46','\x68','\xA8','\xBA','\xBD','\x02','\x01','\x48','\xBD','\x01','\x01','\x48','\xA5','\x64','\x9D','\x02','\x01','\xA5','\x65','\x9D','\x01',
'\x01','\xC8','\x20','\x79','\x00','\xC9','\x2C','\xF0','\xD2','\x84','\x0B','\x20','\xF7','\xAE','\x68','\x85','\x0D','\x68','\x85','\x0E','\x29','\x7F','\x85','\x0C','\xA6','\x2F','\xA5','\x30','\x86','\x5F','\x85','\x60',
'\xC5','\x32','\xD0','\x04','\xE4','\x31','\xF0','\x39','\xA0','\x00','\xB1','\x5F','\xC8','\xC5','\x45','\xD0','\x06','\xA5','\x46','\xD1','\x5F','\xF0','\x16','\xC8','\xB1','\x5F','\x18','\x65','\x5F','\xAA','\xC8','\xB1',
'\x5F','\x65','\x60','\x90','\xD7','\xA2','\x12','\x2C','\xA2','\x0E','\x4C','\x37','\xA4','\xA2','\x13','\xA5','\x0C','\xD0','\xF7','\x20','\x94','\xB1','\xA5','\x0B','\xA0','\x04','\xD1','\x5F','\xD0','\xE7','\x4C','\xEA',
'\xB2','\x20','\x94','\xB1','\x20','\x08','\xA4','\xA0','\x00','\x84','\x72','\xA2','\x05','\xA5','\x45','\x91','\x5F','\x10','\x01','\xCA','\xC8','\xA5','\x46','\x91','\x5F','\x10','\x02','\xCA','\xCA','\x86','\x71','\xA5',
'\x0B','\xC8','\xC8','\xC8','\x91','\x5F','\xA2','\x0B','\xA9','\x00','\x24','\x0C','\x50','\x08','\x68','\x18','\x69','\x01','\xAA','\x68','\x69','\x00','\xC8','\x91','\x5F','\xC8','\x8A','\x91','\x5F','\x20','\x4C','\xB3',
'\x86','\x71','\x85','\x72','\xA4','\x22','\xC6','\x0B','\xD0','\xDC','\x65','\x59','\xB0','\x5D','\x85','\x59','\xA8','\x8A','\x65','\x58','\x90','\x03','\xC8','\xF0','\x52','\x20','\x08','\xA4','\x85','\x31','\x84','\x32',
'\xA9','\x00','\xE6','\x72','\xA4','\x71','\xF0','\x05','\x88','\x91','\x58','\xD0','\xFB','\xC6','\x59','\xC6','\x72','\xD0','\xF5','\xE6','\x59','\x38','\xA5','\x31','\xE5','\x5F','\xA0','\x02','\x91','\x5F','\xA5','\x32',
'\xC8','\xE5','\x60','\x91','\x5F','\xA5','\x0C','\xD0','\x62','\xC8','\xB1','\x5F','\x85','\x0B','\xA9','\x00','\x85','\x71','\x85','\x72','\xC8','\x68','\xAA','\x85','\x64','\x68','\x85','\x65','\xD1','\x5F','\x90','\x0E',
'\xD0','\x06','\xC8','\x8A','\xD1','\x5F','\x90','\x07','\x4C','\x45','\xB2','\x4C','\x35','\xA4','\xC8','\xA5','\x72','\x05','\x71','\x18','\xF0','\x0A','\x20','\x4C','\xB3','\x8A','\x65','\x64','\xAA','\x98','\xA4','\x22',
'\x65','\x65','\x86','\x71','\xC6','\x0B','\xD0','\xCA','\x85','\x72','\xA2','\x05','\xA5','\x45','\x10','\x01','\xCA','\xA5','\x46','\x10','\x02','\xCA','\xCA','\x86','\x28','\xA9','\x00','\x20','\x55','\xB3','\x8A','\x65',
'\x58','\x85','\x47','\x98','\x65','\x59','\x85','\x48','\xA8','\xA5','\x47','\x60','\x84','\x22','\xB1','\x5F','\x85','\x28','\x88','\xB1','\x5F','\x85','\x29','\xA9','\x10','\x85','\x5D','\xA2','\x00','\xA0','\x00','\x8A',
'\x0A','\xAA','\x98','\x2A','\xA8','\xB0','\xA4','\x06','\x71','\x26','\x72','\x90','\x0B','\x18','\x8A','\x65','\x28','\xAA','\x98','\x65','\x29','\xA8','\xB0','\x93','\xC6','\x5D','\xD0','\xE3','\x60','\xA5','\x0D','\xF0',
'\x03','\x20','\xA6','\xB6','\x20','\x26','\xB5','\x38','\xA5','\x33','\xE5','\x31','\xA8','\xA5','\x34','\xE5','\x32','\xA2','\x00','\x86','\x0D','\x85','\x62','\x84','\x63','\xA2','\x90','\x4C','\x44','\xBC','\x38','\x20',
'\xF0','\xFF','\xA9','\x00','\xF0','\xEB','\xA6','\x3A','\xE8','\xD0','\xA0','\xA2','\x15','\x2C','\xA2','\x1B','\x4C','\x37','\xA4','\x20','\xE1','\xB3','\x20','\xA6','\xB3','\x20','\xFA','\xAE','\xA9','\x80','\x85','\x10',
'\x20','\x8B','\xB0','\x20','\x8D','\xAD','\x20','\xF7','\xAE','\xA9','\xB2','\x20','\xFF','\xAE','\x48','\xA5','\x48','\x48','\xA5','\x47','\x48','\xA5','\x7B','\x48','\xA5','\x7A','\x48','\x20','\xF8','\xA8','\x4C','\x4F',
'\xB4','\xA9','\xA5','\x20','\xFF','\xAE','\x09','\x80','\x85','\x10','\x20','\x92','\xB0','\x85','\x4E','\x84','\x4F','\x4C','\x8D','\xAD','\x20','\xE1','\xB3','\xA5','\x4F','\x48','\xA5','\x4E','\x48','\x20','\xF1','\xAE',
'\x20','\x8D','\xAD','\x68','\x85','\x4E','\x68','\x85','\x4F','\xA0','\x02','\xB1','\x4E','\x85','\x47','\xAA','\xC8','\xB1','\x4E','\xF0','\x99','\x85','\x48','\xC8','\xB1','\x47','\x48','\x88','\x10','\xFA','\xA4','\x48',
'\x20','\xD4','\xBB','\xA5','\x7B','\x48','\xA5','\x7A','\x48','\xB1','\x4E','\x85','\x7A','\xC8','\xB1','\x4E','\x85','\x7B','\xA5','\x48','\x48','\xA5','\x47','\x48','\x20','\x8A','\xAD','\x68','\x85','\x4E','\x68','\x85',
'\x4F','\x20','\x79','\x00','\xF0','\x03','\x4C','\x08','\xAF','\x68','\x85','\x7A','\x68','\x85','\x7B','\xA0','\x00','\x68','\x91','\x4E','\x68','\xC8','\x91','\x4E','\x68','\xC8','\x91','\x4E','\x68','\xC8','\x91','\x4E',
'\x68','\xC8','\x91','\x4E','\x60','\x20','\x8D','\xAD','\xA0','\x00','\x20','\xDF','\xBD','\x68','\x68','\xA9','\xFF','\xA0','\x00','\xF0','\x12','\xA6','\x64','\xA4','\x65','\x86','\x50','\x84','\x51','\x20','\xF4','\xB4',
'\x86','\x62','\x84','\x63','\x85','\x61','\x60','\xA2','\x22','\x86','\x07','\x86','\x08','\x85','\x6F','\x84','\x70','\x85','\x62','\x84','\x63','\xA0','\xFF','\xC8','\xB1','\x6F','\xF0','\x0C','\xC5','\x07','\xF0','\x04',
'\xC5','\x08','\xD0','\xF3','\xC9','\x22','\xF0','\x01','\x18','\x84','\x61','\x98','\x65','\x6F','\x85','\x71','\xA6','\x70','\x90','\x01','\xE8','\x86','\x72','\xA5','\x70','\xF0','\x04','\xC9','\x02','\xD0','\x0B','\x98',
'\x20','\x75','\xB4','\xA6','\x6F','\xA4','\x70','\x20','\x88','\xB6','\xA6','\x16','\xE0','\x22','\xD0','\x05','\xA2','\x19','\x4C','\x37','\xA4','\xA5','\x61','\x95','\x00','\xA5','\x62','\x95','\x01','\xA5','\x63','\x95',
'\x02','\xA0','\x00','\x86','\x64','\x84','\x65','\x84','\x70','\x88','\x84','\x0D','\x86','\x17','\xE8','\xE8','\xE8','\x86','\x16','\x60','\x46','\x0F','\x48','\x49','\xFF','\x38','\x65','\x33','\xA4','\x34','\xB0','\x01',
'\x88','\xC4','\x32','\x90','\x11','\xD0','\x04','\xC5','\x31','\x90','\x0B','\x85','\x33','\x84','\x34','\x85','\x35','\x84','\x36','\xAA','\x68','\x60','\xA2','\x10','\xA5','\x0F','\x30','\xB6','\x20','\x26','\xB5','\xA9',
'\x80','\x85','\x0F','\x68','\xD0','\xD0','\xA6','\x37','\xA5','\x38','\x86','\x33','\x85','\x34','\xA0','\x00','\x84','\x4F','\x84','\x4E','\xA5','\x31','\xA6','\x32','\x85','\x5F','\x86','\x60','\xA9','\x19','\xA2','\x00',
'\x85','\x22','\x86','\x23','\xC5','\x16','\xF0','\x05','\x20','\xC7','\xB5','\xF0','\xF7','\xA9','\x07','\x85','\x53','\xA5','\x2D','\xA6','\x2E','\x85','\x22','\x86','\x23','\xE4','\x30','\xD0','\x04','\xC5','\x2F','\xF0',
'\x05','\x20','\xBD','\xB5','\xF0','\xF3','\x85','\x58','\x86','\x59','\xA9','\x03','\x85','\x53','\xA5','\x58','\xA6','\x59','\xE4','\x32','\xD0','\x07','\xC5','\x31','\xD0','\x03','\x4C','\x06','\xB6','\x85','\x22','\x86',
'\x23','\xA0','\x00','\xB1','\x22','\xAA','\xC8','\xB1','\x22','\x08','\xC8','\xB1','\x22','\x65','\x58','\x85','\x58','\xC8','\xB1','\x22','\x65','\x59','\x85','\x59','\x28','\x10','\xD3','\x8A','\x30','\xD0','\xC8','\xB1',
'\x22','\xA0','\x00','\x0A','\x69','\x05','\x65','\x22','\x85','\x22','\x90','\x02','\xE6','\x23','\xA6','\x23','\xE4','\x59','\xD0','\x04','\xC5','\x58','\xF0','\xBA','\x20','\xC7','\xB5','\xF0','\xF3','\xB1','\x22','\x30',
'\x35','\xC8','\xB1','\x22','\x10','\x30','\xC8','\xB1','\x22','\xF0','\x2B','\xC8','\xB1','\x22','\xAA','\xC8','\xB1','\x22','\xC5','\x34','\x90','\x06','\xD0','\x1E','\xE4','\x33','\xB0','\x1A','\xC5','\x60','\x90','\x16',
'\xD0','\x04','\xE4','\x5F','\x90','\x10','\x86','\x5F','\x85','\x60','\xA5','\x22','\xA6','\x23','\x85','\x4E','\x86','\x4F','\xA5','\x53','\x85','\x55','\xA5','\x53','\x18','\x65','\x22','\x85','\x22','\x90','\x02','\xE6',
'\x23','\xA6','\x23','\xA0','\x00','\x60','\xA5','\x4F','\x05','\x4E','\xF0','\xF5','\xA5','\x55','\x29','\x04','\x4A','\xA8','\x85','\x55','\xB1','\x4E','\x65','\x5F','\x85','\x5A','\xA5','\x60','\x69','\x00','\x85','\x5B',
'\xA5','\x33','\xA6','\x34','\x85','\x58','\x86','\x59','\x20','\xBF','\xA3','\xA4','\x55','\xC8','\xA5','\x58','\x91','\x4E','\xAA','\xE6','\x59','\xA5','\x59','\xC8','\x91','\x4E','\x4C','\x2A','\xB5','\xA5','\x65','\x48',
'\xA5','\x64','\x48','\x20','\x83','\xAE','\x20','\x8F','\xAD','\x68','\x85','\x6F','\x68','\x85','\x70','\xA0','\x00','\xB1','\x6F','\x18','\x71','\x64','\x90','\x05','\xA2','\x17','\x4C','\x37','\xA4','\x20','\x75','\xB4',
'\x20','\x7A','\xB6','\xA5','\x50','\xA4','\x51','\x20','\xAA','\xB6','\x20','\x8C','\xB6','\xA5','\x6F','\xA4','\x70','\x20','\xAA','\xB6','\x20','\xCA','\xB4','\x4C','\xB8','\xAD','\xA0','\x00','\xB1','\x6F','\x48','\xC8',
'\xB1','\x6F','\xAA','\xC8','\xB1','\x6F','\xA8','\x68','\x86','\x22','\x84','\x23','\xA8','\xF0','\x0A','\x48','\x88','\xB1','\x22','\x91','\x35','\x98','\xD0','\xF8','\x68','\x18','\x65','\x35','\x85','\x35','\x90','\x02',
'\xE6','\x36','\x60','\x20','\x8F','\xAD','\xA5','\x64','\xA4','\x65','\x85','\x22','\x84','\x23','\x20','\xDB','\xB6','\x08','\xA0','\x00','\xB1','\x22','\x48','\xC8','\xB1','\x22','\xAA','\xC8','\xB1','\x22','\xA8','\x68',
'\x28','\xD0','\x13','\xC4','\x34','\xD0','\x0F','\xE4','\x33','\xD0','\x0B','\x48','\x18','\x65','\x33','\x85','\x33','\x90','\x02','\xE6','\x34','\x68','\x86','\x22','\x84','\x23','\x60','\xC4','\x18','\xD0','\x0C','\xC5',
'\x17','\xD0','\x08','\x85','\x16','\xE9','\x03','\x85','\x17','\xA0','\x00','\x60','\x20','\xA1','\xB7','\x8A','\x48','\xA9','\x01','\x20','\x7D','\xB4','\x68','\xA0','\x00','\x91','\x62','\x68','\x68','\x4C','\xCA','\xB4',
'\x20','\x61','\xB7','\xD1','\x50','\x98','\x90','\x04','\xB1','\x50','\xAA','\x98','\x48','\x8A','\x48','\x20','\x7D','\xB4','\xA5','\x50','\xA4','\x51','\x20','\xAA','\xB6','\x68','\xA8','\x68','\x18','\x65','\x22','\x85',
'\x22','\x90','\x02','\xE6','\x23','\x98','\x20','\x8C','\xB6','\x4C','\xCA','\xB4','\x20','\x61','\xB7','\x18','\xF1','\x50','\x49','\xFF','\x4C','\x06','\xB7','\xA9','\xFF','\x85','\x65','\x20','\x79','\x00','\xC9','\x29',
'\xF0','\x06','\x20','\xFD','\xAE','\x20','\x9E','\xB7','\x20','\x61','\xB7','\xF0','\x4B','\xCA','\x8A','\x48','\x18','\xA2','\x00','\xF1','\x50','\xB0','\xB6','\x49','\xFF','\xC5','\x65','\x90','\xB1','\xA5','\x65','\xB0',
'\xAD','\x20','\xF7','\xAE','\x68','\xA8','\x68','\x85','\x55','\x68','\x68','\x68','\xAA','\x68','\x85','\x50','\x68','\x85','\x51','\xA5','\x55','\x48','\x98','\x48','\xA0','\x00','\x8A','\x60','\x20','\x82','\xB7','\x4C',
'\xA2','\xB3','\x20','\xA3','\xB6','\xA2','\x00','\x86','\x0D','\xA8','\x60','\x20','\x82','\xB7','\xF0','\x08','\xA0','\x00','\xB1','\x22','\xA8','\x4C','\xA2','\xB3','\x4C','\x48','\xB2','\x20','\x73','\x00','\x20','\x8A',
'\xAD','\x20','\xB8','\xB1','\xA6','\x64','\xD0','\xF0','\xA6','\x65','\x4C','\x79','\x00','\x20','\x82','\xB7','\xD0','\x03','\x4C','\xF7','\xB8','\xA6','\x7A','\xA4','\x7B','\x86','\x71','\x84','\x72','\xA6','\x22','\x86',
'\x7A','\x18','\x65','\x22','\x85','\x24','\xA6','\x23','\x86','\x7B','\x90','\x01','\xE8','\x86','\x25','\xA0','\x00','\xB1','\x24','\x48','\x98','\x91','\x24','\x20','\x79','\x00','\x20','\xF3','\xBC','\x68','\xA0','\x00',
'\x91','\x24','\xA6','\x71','\xA4','\x72','\x86','\x7A','\x84','\x7B','\x60','\x20','\x8A','\xAD','\x20','\xF7','\xB7','\x20','\xFD','\xAE','\x4C','\x9E','\xB7','\xA5','\x66','\x30','\x9D','\xA5','\x61','\xC9','\x91','\xB0',
'\x97','\x20','\x9B','\xBC','\xA5','\x64','\xA4','\x65','\x84','\x14','\x85','\x15','\x60','\xA5','\x15','\x48','\xA5','\x14','\x48','\x20','\xF7','\xB7','\xA0','\x00','\xB1','\x14','\xA8','\x68','\x85','\x14','\x68','\x85',
'\x15','\x4C','\xA2','\xB3','\x20','\xEB','\xB7','\x8A','\xA0','\x00','\x91','\x14','\x60','\x20','\xEB','\xB7','\x86','\x49','\xA2','\x00','\x20','\x79','\x00','\xF0','\x03','\x20','\xF1','\xB7','\x86','\x4A','\xA0','\x00',
'\xB1','\x14','\x45','\x4A','\x25','\x49','\xF0','\xF8','\x60','\xA9','\x11','\xA0','\xBF','\x4C','\x67','\xB8','\x20','\x8C','\xBA','\xA5','\x66','\x49','\xFF','\x85','\x66','\x45','\x6E','\x85','\x6F','\xA5','\x61','\x4C',
'\x6A','\xB8','\x20','\x99','\xB9','\x90','\x3C','\x20','\x8C','\xBA','\xD0','\x03','\x4C','\xFC','\xBB','\xA6','\x70','\x86','\x56','\xA2','\x69','\xA5','\x69','\xA8','\xF0','\xCE','\x38','\xE5','\x61','\xF0','\x24','\x90',
'\x12','\x84','\x61','\xA4','\x6E','\x84','\x66','\x49','\xFF','\x69','\x00','\xA0','\x00','\x84','\x56','\xA2','\x61','\xD0','\x04','\xA0','\x00','\x84','\x70','\xC9','\xF9','\x30','\xC7','\xA8','\xA5','\x70','\x56','\x01',
'\x20','\xB0','\xB9','\x24','\x6F','\x10','\x57','\xA0','\x61','\xE0','\x69','\xF0','\x02','\xA0','\x69','\x38','\x49','\xFF','\x65','\x56','\x85','\x70','\xB9','\x04','\x00','\xF5','\x04','\x85','\x65','\xB9','\x03','\x00',
'\xF5','\x03','\x85','\x64','\xB9','\x02','\x00','\xF5','\x02','\x85','\x63','\xB9','\x01','\x00','\xF5','\x01','\x85','\x62','\xB0','\x03','\x20','\x47','\xB9','\xA0','\x00','\x98','\x18','\xA6','\x62','\xD0','\x4A','\xA6',
'\x63','\x86','\x62','\xA6','\x64','\x86','\x63','\xA6','\x65','\x86','\x64','\xA6','\x70','\x86','\x65','\x84','\x70','\x69','\x08','\xC9','\x20','\xD0','\xE4','\xA9','\x00','\x85','\x61','\x85','\x66','\x60','\x65','\x56',
'\x85','\x70','\xA5','\x65','\x65','\x6D','\x85','\x65','\xA5','\x64','\x65','\x6C','\x85','\x64','\xA5','\x63','\x65','\x6B','\x85','\x63','\xA5','\x62','\x65','\x6A','\x85','\x62','\x4C','\x36','\xB9','\x69','\x01','\x06',
'\x70','\x26','\x65','\x26','\x64','\x26','\x63','\x26','\x62','\x10','\xF2','\x38','\xE5','\x61','\xB0','\xC7','\x49','\xFF','\x69','\x01','\x85','\x61','\x90','\x0E','\xE6','\x61','\xF0','\x42','\x66','\x62','\x66','\x63',
'\x66','\x64','\x66','\x65','\x66','\x70','\x60','\xA5','\x66','\x49','\xFF','\x85','\x66','\xA5','\x62','\x49','\xFF','\x85','\x62','\xA5','\x63','\x49','\xFF','\x85','\x63','\xA5','\x64','\x49','\xFF','\x85','\x64','\xA5',
'\x65','\x49','\xFF','\x85','\x65','\xA5','\x70','\x49','\xFF','\x85','\x70','\xE6','\x70','\xD0','\x0E','\xE6','\x65','\xD0','\x0A','\xE6','\x64','\xD0','\x06','\xE6','\x63','\xD0','\x02','\xE6','\x62','\x60','\xA2','\x0F',
'\x4C','\x37','\xA4','\xA2','\x25','\xB4','\x04','\x84','\x70','\xB4','\x03','\x94','\x04','\xB4','\x02','\x94','\x03','\xB4','\x01','\x94','\x02','\xA4','\x68','\x94','\x01','\x69','\x08','\x30','\xE8','\xF0','\xE6','\xE9',
'\x08','\xA8','\xA5','\x70','\xB0','\x14','\x16','\x01','\x90','\x02','\xF6','\x01','\x76','\x01','\x76','\x01','\x76','\x02','\x76','\x03','\x76','\x04','\x6A','\xC8','\xD0','\xEC','\x18','\x60','\x81','\x00','\x00','\x00',
'\x00','\x03','\x7F','\x5E','\x56','\xCB','\x79','\x80','\x13','\x9B','\x0B','\x64','\x80','\x76','\x38','\x93','\x16','\x82','\x38','\xAA','\x3B','\x20','\x80','\x35','\x04','\xF3','\x34','\x81','\x35','\x04','\xF3','\x34',
'\x80','\x80','\x00','\x00','\x00','\x80','\x31','\x72','\x17','\xF8','\x20','\x2B','\xBC','\xF0','\x02','\x10','\x03','\x4C','\x48','\xB2','\xA5','\x61','\xE9','\x7F','\x48','\xA9','\x80','\x85','\x61','\xA9','\xD6','\xA0',
'\xB9','\x20','\x67','\xB8','\xA9','\xDB','\xA0','\xB9','\x20','\x0F','\xBB','\xA9','\xBC','\xA0','\xB9','\x20','\x50','\xB8','\xA9','\xC1','\xA0','\xB9','\x20','\x43','\xE0','\xA9','\xE0','\xA0','\xB9','\x20','\x67','\xB8',
'\x68','\x20','\x7E','\xBD','\xA9','\xE5','\xA0','\xB9','\x20','\x8C','\xBA','\xD0','\x03','\x4C','\x8B','\xBA','\x20','\xB7','\xBA','\xA9','\x00','\x85','\x26','\x85','\x27','\x85','\x28','\x85','\x29','\xA5','\x70','\x20',
'\x59','\xBA','\xA5','\x65','\x20','\x59','\xBA','\xA5','\x64','\x20','\x59','\xBA','\xA5','\x63','\x20','\x59','\xBA','\xA5','\x62','\x20','\x5E','\xBA','\x4C','\x8F','\xBB','\xD0','\x03','\x4C','\x83','\xB9','\x4A','\x09',
'\x80','\xA8','\x90','\x19','\x18','\xA5','\x29','\x65','\x6D','\x85','\x29','\xA5','\x28','\x65','\x6C','\x85','\x28','\xA5','\x27','\x65','\x6B','\x85','\x27','\xA5','\x26','\x65','\x6A','\x85','\x26','\x66','\x26','\x66',
'\x27','\x66','\x28','\x66','\x29','\x66','\x70','\x98','\x4A','\xD0','\xD6','\x60','\x85','\x22','\x84','\x23','\xA0','\x04','\xB1','\x22','\x85','\x6D','\x88','\xB1','\x22','\x85','\x6C','\x88','\xB1','\x22','\x85','\x6B',
'\x88','\xB1','\x22','\x85','\x6E','\x45','\x66','\x85','\x6F','\xA5','\x6E','\x09','\x80','\x85','\x6A','\x88','\xB1','\x22','\x85','\x69','\xA5','\x61','\x60','\xA5','\x69','\xF0','\x1F','\x18','\x65','\x61','\x90','\x04',
'\x30','\x1D','\x18','\x2C','\x10','\x14','\x69','\x80','\x85','\x61','\xD0','\x03','\x4C','\xFB','\xB8','\xA5','\x6F','\x85','\x66','\x60','\xA5','\x66','\x49','\xFF','\x30','\x05','\x68','\x68','\x4C','\xF7','\xB8','\x4C',
'\x7E','\xB9','\x20','\x0C','\xBC','\xAA','\xF0','\x10','\x18','\x69','\x02','\xB0','\xF2','\xA2','\x00','\x86','\x6F','\x20','\x77','\xB8','\xE6','\x61','\xF0','\xE7','\x60','\x84','\x20','\x00','\x00','\x00','\x20','\x0C',
'\xBC','\xA9','\xF9','\xA0','\xBA','\xA2','\x00','\x86','\x6F','\x20','\xA2','\xBB','\x4C','\x12','\xBB','\x20','\x8C','\xBA','\xF0','\x76','\x20','\x1B','\xBC','\xA9','\x00','\x38','\xE5','\x61','\x85','\x61','\x20','\xB7',
'\xBA','\xE6','\x61','\xF0','\xBA','\xA2','\xFC','\xA9','\x01','\xA4','\x6A','\xC4','\x62','\xD0','\x10','\xA4','\x6B','\xC4','\x63','\xD0','\x0A','\xA4','\x6C','\xC4','\x64','\xD0','\x04','\xA4','\x6D','\xC4','\x65','\x08',
'\x2A','\x90','\x09','\xE8','\x95','\x29','\xF0','\x32','\x10','\x34','\xA9','\x01','\x28','\xB0','\x0E','\x06','\x6D','\x26','\x6C','\x26','\x6B','\x26','\x6A','\xB0','\xE6','\x30','\xCE','\x10','\xE2','\xA8','\xA5','\x6D',
'\xE5','\x65','\x85','\x6D','\xA5','\x6C','\xE5','\x64','\x85','\x6C','\xA5','\x6B','\xE5','\x63','\x85','\x6B','\xA5','\x6A','\xE5','\x62','\x85','\x6A','\x98','\x4C','\x4F','\xBB','\xA9','\x40','\xD0','\xCE','\x0A','\x0A',
'\x0A','\x0A','\x0A','\x0A','\x85','\x70','\x28','\x4C','\x8F','\xBB','\xA2','\x14','\x4C','\x37','\xA4','\xA5','\x26','\x85','\x62','\xA5','\x27','\x85','\x63','\xA5','\x28','\x85','\x64','\xA5','\x29','\x85','\x65','\x4C',
'\xD7','\xB8','\x85','\x22','\x84','\x23','\xA0','\x04','\xB1','\x22','\x85','\x65','\x88','\xB1','\x22','\x85','\x64','\x88','\xB1','\x22','\x85','\x63','\x88','\xB1','\x22','\x85','\x66','\x09','\x80','\x85','\x62','\x88',
'\xB1','\x22','\x85','\x61','\x84','\x70','\x60','\xA2','\x5C','\x2C','\xA2','\x57','\xA0','\x00','\xF0','\x04','\xA6','\x49','\xA4','\x4A','\x20','\x1B','\xBC','\x86','\x22','\x84','\x23','\xA0','\x04','\xA5','\x65','\x91',
'\x22','\x88','\xA5','\x64','\x91','\x22','\x88','\xA5','\x63','\x91','\x22','\x88','\xA5','\x66','\x09','\x7F','\x25','\x62','\x91','\x22','\x88','\xA5','\x61','\x91','\x22','\x84','\x70','\x60','\xA5','\x6E','\x85','\x66',
'\xA2','\x05','\xB5','\x68','\x95','\x60','\xCA','\xD0','\xF9','\x86','\x70','\x60','\x20','\x1B','\xBC','\xA2','\x06','\xB5','\x60','\x95','\x68','\xCA','\xD0','\xF9','\x86','\x70','\x60','\xA5','\x61','\xF0','\xFB','\x06',
'\x70','\x90','\xF7','\x20','\x6F','\xB9','\xD0','\xF2','\x4C','\x38','\xB9','\xA5','\x61','\xF0','\x09','\xA5','\x66','\x2A','\xA9','\xFF','\xB0','\x02','\xA9','\x01','\x60','\x20','\x2B','\xBC','\x85','\x62','\xA9','\x00',
'\x85','\x63','\xA2','\x88','\xA5','\x62','\x49','\xFF','\x2A','\xA9','\x00','\x85','\x65','\x85','\x64','\x86','\x61','\x85','\x70','\x85','\x66','\x4C','\xD2','\xB8','\x46','\x66','\x60','\x85','\x24','\x84','\x25','\xA0',
'\x00','\xB1','\x24','\xC8','\xAA','\xF0','\xC4','\xB1','\x24','\x45','\x66','\x30','\xC2','\xE4','\x61','\xD0','\x21','\xB1','\x24','\x09','\x80','\xC5','\x62','\xD0','\x19','\xC8','\xB1','\x24','\xC5','\x63','\xD0','\x12',
'\xC8','\xB1','\x24','\xC5','\x64','\xD0','\x0B','\xC8','\xA9','\x7F','\xC5','\x70','\xB1','\x24','\xE5','\x65','\xF0','\x28','\xA5','\x66','\x90','\x02','\x49','\xFF','\x4C','\x31','\xBC','\xA5','\x61','\xF0','\x4A','\x38',
'\xE9','\xA0','\x24','\x66','\x10','\x09','\xAA','\xA9','\xFF','\x85','\x68','\x20','\x4D','\xB9','\x8A','\xA2','\x61','\xC9','\xF9','\x10','\x06','\x20','\x99','\xB9','\x84','\x68','\x60','\xA8','\xA5','\x66','\x29','\x80',
'\x46','\x62','\x05','\x62','\x85','\x62','\x20','\xB0','\xB9','\x84','\x68','\x60','\xA5','\x61','\xC9','\xA0','\xB0','\x20','\x20','\x9B','\xBC','\x84','\x70','\xA5','\x66','\x84','\x66','\x49','\x80','\x2A','\xA9','\xA0',
'\x85','\x61','\xA5','\x65','\x85','\x07','\x4C','\xD2','\xB8','\x85','\x62','\x85','\x63','\x85','\x64','\x85','\x65','\xA8','\x60','\xA0','\x00','\xA2','\x0A','\x94','\x5D','\xCA','\x10','\xFB','\x90','\x0F','\xC9','\x2D',
'\xD0','\x04','\x86','\x67','\xF0','\x04','\xC9','\x2B','\xD0','\x05','\x20','\x73','\x00','\x90','\x5B','\xC9','\x2E','\xF0','\x2E','\xC9','\x45','\xD0','\x30','\x20','\x73','\x00','\x90','\x17','\xC9','\xAB','\xF0','\x0E',
'\xC9','\x2D','\xF0','\x0A','\xC9','\xAA','\xF0','\x08','\xC9','\x2B','\xF0','\x04','\xD0','\x07','\x66','\x60','\x20','\x73','\x00','\x90','\x5C','\x24','\x60','\x10','\x0E','\xA9','\x00','\x38','\xE5','\x5E','\x4C','\x49',
'\xBD','\x66','\x5F','\x24','\x5F','\x50','\xC3','\xA5','\x5E','\x38','\xE5','\x5D','\x85','\x5E','\xF0','\x12','\x10','\x09','\x20','\xFE','\xBA','\xE6','\x5E','\xD0','\xF9','\xF0','\x07','\x20','\xE2','\xBA','\xC6','\x5E',
'\xD0','\xF9','\xA5','\x67','\x30','\x01','\x60','\x4C','\xB4','\xBF','\x48','\x24','\x5F','\x10','\x02','\xE6','\x5D','\x20','\xE2','\xBA','\x68','\x38','\xE9','\x30','\x20','\x7E','\xBD','\x4C','\x0A','\xBD','\x48','\x20',
'\x0C','\xBC','\x68','\x20','\x3C','\xBC','\xA5','\x6E','\x45','\x66','\x85','\x6F','\xA6','\x61','\x4C','\x6A','\xB8','\xA5','\x5E','\xC9','\x0A','\x90','\x09','\xA9','\x64','\x24','\x60','\x30','\x11','\x4C','\x7E','\xB9',
'\x0A','\x0A','\x18','\x65','\x5E','\x0A','\x18','\xA0','\x00','\x71','\x7A','\x38','\xE9','\x30','\x85','\x5E','\x4C','\x30','\xBD','\x9B','\x3E','\xBC','\x1F','\xFD','\x9E','\x6E','\x6B','\x27','\xFD','\x9E','\x6E','\x6B',
'\x28','\x00','\xA9','\x71','\xA0','\xA3','\x20','\xDA','\xBD','\xA5','\x3A','\xA6','\x39','\x85','\x62','\x86','\x63','\xA2','\x90','\x38','\x20','\x49','\xBC','\x20','\xDF','\xBD','\x4C','\x1E','\xAB','\xA0','\x01','\xA9',
'\x20','\x24','\x66','\x10','\x02','\xA9','\x2D','\x99','\xFF','\x00','\x85','\x66','\x84','\x71','\xC8','\xA9','\x30','\xA6','\x61','\xD0','\x03','\x4C','\x04','\xBF','\xA9','\x00','\xE0','\x80','\xF0','\x02','\xB0','\x09',
'\xA9','\xBD','\xA0','\xBD','\x20','\x28','\xBA','\xA9','\xF7','\x85','\x5D','\xA9','\xB8','\xA0','\xBD','\x20','\x5B','\xBC','\xF0','\x1E','\x10','\x12','\xA9','\xB3','\xA0','\xBD','\x20','\x5B','\xBC','\xF0','\x02','\x10',
'\x0E','\x20','\xE2','\xBA','\xC6','\x5D','\xD0','\xEE','\x20','\xFE','\xBA','\xE6','\x5D','\xD0','\xDC','\x20','\x49','\xB8','\x20','\x9B','\xBC','\xA2','\x01','\xA5','\x5D','\x18','\x69','\x0A','\x30','\x09','\xC9','\x0B',
'\xB0','\x06','\x69','\xFF','\xAA','\xA9','\x02','\x38','\xE9','\x02','\x85','\x5E','\x86','\x5D','\x8A','\xF0','\x02','\x10','\x13','\xA4','\x71','\xA9','\x2E','\xC8','\x99','\xFF','\x00','\x8A','\xF0','\x06','\xA9','\x30',
'\xC8','\x99','\xFF','\x00','\x84','\x71','\xA0','\x00','\xA2','\x80','\xA5','\x65','\x18','\x79','\x19','\xBF','\x85','\x65','\xA5','\x64','\x79','\x18','\xBF','\x85','\x64','\xA5','\x63','\x79','\x17','\xBF','\x85','\x63',
'\xA5','\x62','\x79','\x16','\xBF','\x85','\x62','\xE8','\xB0','\x04','\x10','\xDE','\x30','\x02','\x30','\xDA','\x8A','\x90','\x04','\x49','\xFF','\x69','\x0A','\x69','\x2F','\xC8','\xC8','\xC8','\xC8','\x84','\x47','\xA4',
'\x71','\xC8','\xAA','\x29','\x7F','\x99','\xFF','\x00','\xC6','\x5D','\xD0','\x06','\xA9','\x2E','\xC8','\x99','\xFF','\x00','\x84','\x71','\xA4','\x47','\x8A','\x49','\xFF','\x29','\x80','\xAA','\xC0','\x24','\xF0','\x04',
'\xC0','\x3C','\xD0','\xA6','\xA4','\x71','\xB9','\xFF','\x00','\x88','\xC9','\x30','\xF0','\xF8','\xC9','\x2E','\xF0','\x01','\xC8','\xA9','\x2B','\xA6','\x5E','\xF0','\x2E','\x10','\x08','\xA9','\x00','\x38','\xE5','\x5E',
'\xAA','\xA9','\x2D','\x99','\x01','\x01','\xA9','\x45','\x99','\x00','\x01','\x8A','\xA2','\x2F','\x38','\xE8','\xE9','\x0A','\xB0','\xFB','\x69','\x3A','\x99','\x03','\x01','\x8A','\x99','\x02','\x01','\xA9','\x00','\x99',
'\x04','\x01','\xF0','\x08','\x99','\xFF','\x00','\xA9','\x00','\x99','\x00','\x01','\xA9','\x00','\xA0','\x01','\x60','\x80','\x00','\x00','\x00','\x00','\xFA','\x0A','\x1F','\x00','\x00','\x98','\x96','\x80','\xFF','\xF0',
'\xBD','\xC0','\x00','\x01','\x86','\xA0','\xFF','\xFF','\xD8','\xF0','\x00','\x00','\x03','\xE8','\xFF','\xFF','\xFF','\x9C','\x00','\x00','\x00','\x0A','\xFF','\xFF','\xFF','\xFF','\xFF','\xDF','\x0A','\x80','\x00','\x03',
'\x4B','\xC0','\xFF','\xFF','\x73','\x60','\x00','\x00','\x0E','\x10','\xFF','\xFF','\xFD','\xA8','\x00','\x00','\x00','\x3C','\xEC','\xAA','\xAA','\xAA','\xAA','\xAA','\xAA','\xAA','\xAA','\xAA','\xAA','\xAA','\xAA','\xAA',
'\xAA','\xAA','\xAA','\xAA','\xAA','\xAA','\xAA','\xAA','\xAA','\xAA','\xAA','\xAA','\xAA','\xAA','\xAA','\xAA','\xAA','\x20','\x0C','\xBC','\xA9','\x11','\xA0','\xBF','\x20','\xA2','\xBB','\xF0','\x70','\xA5','\x69','\xD0',
'\x03','\x4C','\xF9','\xB8','\xA2','\x4E','\xA0','\x00','\x20','\xD4','\xBB','\xA5','\x6E','\x10','\x0F','\x20','\xCC','\xBC','\xA9','\x4E','\xA0','\x00','\x20','\x5B','\xBC','\xD0','\x03','\x98','\xA4','\x07','\x20','\xFE',
'\xBB','\x98','\x48','\x20','\xEA','\xB9','\xA9','\x4E','\xA0','\x00','\x20','\x28','\xBA','\x20','\xED','\xBF','\x68','\x4A','\x90','\x0A','\xA5','\x61','\xF0','\x06','\xA5','\x66','\x49','\xFF','\x85','\x66','\x60','\x81',
'\x38','\xAA','\x3B','\x29','\x07','\x71','\x34','\x58','\x3E','\x56','\x74','\x16','\x7E','\xB3','\x1B','\x77','\x2F','\xEE','\xE3','\x85','\x7A','\x1D','\x84','\x1C','\x2A','\x7C','\x63','\x59','\x58','\x0A','\x7E','\x75',
'\xFD','\xE7','\xC6','\x80','\x31','\x72','\x18','\x10','\x81','\x00','\x00','\x00','\x00','\xA9','\xBF','\xA0','\xBF','\x20','\x28','\xBA','\xA5','\x70','\x69','\x50','\x90','\x03','\x20','\x23','\xBC','\x4C','\x00','\xE0'
};

static const byte kernal_rom[8 * 1024] = {
'\x85','\x56','\x20','\x0F','\xBC','\xA5','\x61','\xC9','\x88','\x90','\x03','\x20','\xD4','\xBA','\x20','\xCC','\xBC','\xA5','\x07','\x18','\x69','\x81','\xF0','\xF3','\x38','\xE9','\x01','\x48','\xA2','\x05','\xB5','\x69',
'\xB4','\x61','\x95','\x61','\x94','\x69','\xCA','\x10','\xF5','\xA5','\x56','\x85','\x70','\x20','\x53','\xB8','\x20','\xB4','\xBF','\xA9','\xC4','\xA0','\xBF','\x20','\x59','\xE0','\xA9','\x00','\x85','\x6F','\x68','\x20',
'\xB9','\xBA','\x60','\x85','\x71','\x84','\x72','\x20','\xCA','\xBB','\xA9','\x57','\x20','\x28','\xBA','\x20','\x5D','\xE0','\xA9','\x57','\xA0','\x00','\x4C','\x28','\xBA','\x85','\x71','\x84','\x72','\x20','\xC7','\xBB',
'\xB1','\x71','\x85','\x67','\xA4','\x71','\xC8','\x98','\xD0','\x02','\xE6','\x72','\x85','\x71','\xA4','\x72','\x20','\x28','\xBA','\xA5','\x71','\xA4','\x72','\x18','\x69','\x05','\x90','\x01','\xC8','\x85','\x71','\x84',
'\x72','\x20','\x67','\xB8','\xA9','\x5C','\xA0','\x00','\xC6','\x67','\xD0','\xE4','\x60','\x98','\x35','\x44','\x7A','\x00','\x68','\x28','\xB1','\x46','\x00','\x20','\x2B','\xBC','\x30','\x37','\xD0','\x20','\x20','\xF3',
'\xFF','\x86','\x22','\x84','\x23','\xA0','\x04','\xB1','\x22','\x85','\x62','\xC8','\xB1','\x22','\x85','\x64','\xA0','\x08','\xB1','\x22','\x85','\x63','\xC8','\xB1','\x22','\x85','\x65','\x4C','\xE3','\xE0','\xA9','\x8B',
'\xA0','\x00','\x20','\xA2','\xBB','\xA9','\x8D','\xA0','\xE0','\x20','\x28','\xBA','\xA9','\x92','\xA0','\xE0','\x20','\x67','\xB8','\xA6','\x65','\xA5','\x62','\x85','\x65','\x86','\x62','\xA6','\x63','\xA5','\x64','\x85',
'\x63','\x86','\x64','\xA9','\x00','\x85','\x66','\xA5','\x61','\x85','\x70','\xA9','\x80','\x85','\x61','\x20','\xD7','\xB8','\xA2','\x8B','\xA0','\x00','\x4C','\xD4','\xBB','\xC9','\xF0','\xD0','\x07','\x84','\x38','\x86',
'\x37','\x4C','\x63','\xA6','\xAA','\xD0','\x02','\xA2','\x1E','\x4C','\x37','\xA4','\x20','\xD2','\xFF','\xB0','\xE8','\x60','\x20','\xCF','\xFF','\xB0','\xE2','\x60','\x20','\xAD','\xE4','\xB0','\xDC','\x60','\x20','\xC6',
'\xFF','\xB0','\xD6','\x60','\x20','\xE4','\xFF','\xB0','\xD0','\x60','\x20','\x8A','\xAD','\x20','\xF7','\xB7','\xA9','\xE1','\x48','\xA9','\x46','\x48','\xAD','\x0F','\x03','\x48','\xAD','\x0C','\x03','\xAE','\x0D','\x03',
'\xAC','\x0E','\x03','\x28','\x6C','\x14','\x00','\x08','\x8D','\x0C','\x03','\x8E','\x0D','\x03','\x8C','\x0E','\x03','\x68','\x8D','\x0F','\x03','\x60','\x20','\xD4','\xE1','\xA6','\x2D','\xA4','\x2E','\xA9','\x2B','\x20',
'\xD8','\xFF','\xB0','\x95','\x60','\xA9','\x01','\x2C','\xA9','\x00','\x85','\x0A','\x20','\xD4','\xE1','\xA5','\x0A','\xA6','\x2B','\xA4','\x2C','\x20','\xD5','\xFF','\xB0','\x57','\xA5','\x0A','\xF0','\x17','\xA2','\x1C',
'\x20','\xB7','\xFF','\x29','\x10','\xD0','\x17','\xA5','\x7A','\xC9','\x02','\xF0','\x07','\xA9','\x64','\xA0','\xA3','\x4C','\x1E','\xAB','\x60','\x20','\xB7','\xFF','\x29','\xBF','\xF0','\x05','\xA2','\x1D','\x4C','\x37',
'\xA4','\xA5','\x7B','\xC9','\x02','\xD0','\x0E','\x86','\x2D','\x84','\x2E','\xA9','\x76','\xA0','\xA3','\x20','\x1E','\xAB','\x4C','\x2A','\xA5','\x20','\x8E','\xA6','\x20','\x33','\xA5','\x4C','\x77','\xA6','\x20','\x19',
'\xE2','\x20','\xC0','\xFF','\xB0','\x0B','\x60','\x20','\x19','\xE2','\xA5','\x49','\x20','\xC3','\xFF','\x90','\xC3','\x4C','\xF9','\xE0','\xA9','\x00','\x20','\xBD','\xFF','\xA2','\x01','\xA0','\x00','\x20','\xBA','\xFF',
'\x20','\x06','\xE2','\x20','\x57','\xE2','\x20','\x06','\xE2','\x20','\x00','\xE2','\xA0','\x00','\x86','\x49','\x20','\xBA','\xFF','\x20','\x06','\xE2','\x20','\x00','\xE2','\x8A','\xA8','\xA6','\x49','\x4C','\xBA','\xFF',
'\x20','\x0E','\xE2','\x4C','\x9E','\xB7','\x20','\x79','\x00','\xD0','\x02','\x68','\x68','\x60','\x20','\xFD','\xAE','\x20','\x79','\x00','\xD0','\xF7','\x4C','\x08','\xAF','\xA9','\x00','\x20','\xBD','\xFF','\x20','\x11',
'\xE2','\x20','\x9E','\xB7','\x86','\x49','\x8A','\xA2','\x01','\xA0','\x00','\x20','\xBA','\xFF','\x20','\x06','\xE2','\x20','\x00','\xE2','\x86','\x4A','\xA0','\x00','\xA5','\x49','\xE0','\x03','\x90','\x01','\x88','\x20',
'\xBA','\xFF','\x20','\x06','\xE2','\x20','\x00','\xE2','\x8A','\xA8','\xA6','\x4A','\xA5','\x49','\x20','\xBA','\xFF','\x20','\x06','\xE2','\x20','\x0E','\xE2','\x20','\x9E','\xAD','\x20','\xA3','\xB6','\xA6','\x22','\xA4',
'\x23','\x4C','\xBD','\xFF','\xA9','\xE0','\xA0','\xE2','\x20','\x67','\xB8','\x20','\x0C','\xBC','\xA9','\xE5','\xA0','\xE2','\xA6','\x6E','\x20','\x07','\xBB','\x20','\x0C','\xBC','\x20','\xCC','\xBC','\xA9','\x00','\x85',
'\x6F','\x20','\x53','\xB8','\xA9','\xEA','\xA0','\xE2','\x20','\x50','\xB8','\xA5','\x66','\x48','\x10','\x0D','\x20','\x49','\xB8','\xA5','\x66','\x30','\x09','\xA5','\x12','\x49','\xFF','\x85','\x12','\x20','\xB4','\xBF',
'\xA9','\xEA','\xA0','\xE2','\x20','\x67','\xB8','\x68','\x10','\x03','\x20','\xB4','\xBF','\xA9','\xEF','\xA0','\xE2','\x4C','\x43','\xE0','\x20','\xCA','\xBB','\xA9','\x00','\x85','\x12','\x20','\x6B','\xE2','\xA2','\x4E',
'\xA0','\x00','\x20','\xF6','\xE0','\xA9','\x57','\xA0','\x00','\x20','\xA2','\xBB','\xA9','\x00','\x85','\x66','\xA5','\x12','\x20','\xDC','\xE2','\xA9','\x4E','\xA0','\x00','\x4C','\x0F','\xBB','\x48','\x4C','\x9D','\xE2',
'\x81','\x49','\x0F','\xDA','\xA2','\x83','\x49','\x0F','\xDA','\xA2','\x7F','\x00','\x00','\x00','\x00','\x05','\x84','\xE6','\x1A','\x2D','\x1B','\x86','\x28','\x07','\xFB','\xF8','\x87','\x99','\x68','\x89','\x01','\x87',
'\x23','\x35','\xDF','\xE1','\x86','\xA5','\x5D','\xE7','\x28','\x83','\x49','\x0F','\xDA','\xA2','\xA5','\x66','\x48','\x10','\x03','\x20','\xB4','\xBF','\xA5','\x61','\x48','\xC9','\x81','\x90','\x07','\xA9','\xBC','\xA0',
'\xB9','\x20','\x0F','\xBB','\xA9','\x3E','\xA0','\xE3','\x20','\x43','\xE0','\x68','\xC9','\x81','\x90','\x07','\xA9','\xE0','\xA0','\xE2','\x20','\x50','\xB8','\x68','\x10','\x03','\x4C','\xB4','\xBF','\x60','\x0B','\x76',
'\xB3','\x83','\xBD','\xD3','\x79','\x1E','\xF4','\xA6','\xF5','\x7B','\x83','\xFC','\xB0','\x10','\x7C','\x0C','\x1F','\x67','\xCA','\x7C','\xDE','\x53','\xCB','\xC1','\x7D','\x14','\x64','\x70','\x4C','\x7D','\xB7','\xEA',
'\x51','\x7A','\x7D','\x63','\x30','\x88','\x7E','\x7E','\x92','\x44','\x99','\x3A','\x7E','\x4C','\xCC','\x91','\xC7','\x7F','\xAA','\xAA','\xAA','\x13','\x81','\x00','\x00','\x00','\x00','\x20','\xCC','\xFF','\xA9','\x00',
'\x85','\x13','\x20','\x7A','\xA6','\x58','\xA2','\x80','\x6C','\x00','\x03','\x8A','\x30','\x03','\x4C','\x3A','\xA4','\x4C','\x74','\xA4','\x20','\x53','\xE4','\x20','\xBF','\xE3','\x20','\x22','\xE4','\xA2','\xFB','\x9A',
'\xD0','\xE4','\xE6','\x7A','\xD0','\x02','\xE6','\x7B','\xAD','\x60','\xEA','\xC9','\x3A','\xB0','\x0A','\xC9','\x20','\xF0','\xEF','\x38','\xE9','\x30','\x38','\xE9','\xD0','\x60','\x80','\x4F','\xC7','\x52','\x58','\xA9',
'\x4C','\x85','\x54','\x8D','\x10','\x03','\xA9','\x48','\xA0','\xB2','\x8D','\x11','\x03','\x8C','\x12','\x03','\xA9','\x91','\xA0','\xB3','\x85','\x05','\x84','\x06','\xA9','\xAA','\xA0','\xB1','\x85','\x03','\x84','\x04',
'\xA2','\x1C','\xBD','\xA2','\xE3','\x95','\x73','\xCA','\x10','\xF8','\xA9','\x03','\x85','\x53','\xA9','\x00','\x85','\x68','\x85','\x13','\x85','\x18','\xA2','\x01','\x8E','\xFD','\x01','\x8E','\xFC','\x01','\xA2','\x19',
'\x86','\x16','\x38','\x20','\x9C','\xFF','\x86','\x2B','\x84','\x2C','\x38','\x20','\x99','\xFF','\x86','\x37','\x84','\x38','\x86','\x33','\x84','\x34','\xA0','\x00','\x98','\x91','\x2B','\xE6','\x2B','\xD0','\x02','\xE6',
'\x2C','\x60','\xA5','\x2B','\xA4','\x2C','\x20','\x08','\xA4','\xA9','\x73','\xA0','\xE4','\x20','\x1E','\xAB','\xA5','\x37','\x38','\xE5','\x2B','\xAA','\xA5','\x38','\xE5','\x2C','\x20','\xCD','\xBD','\xA9','\x60','\xA0',
'\xE4','\x20','\x1E','\xAB','\x4C','\x44','\xA6','\x8B','\xE3','\x83','\xA4','\x7C','\xA5','\x1A','\xA7','\xE4','\xA7','\x86','\xAE','\xA2','\x0B','\xBD','\x47','\xE4','\x9D','\x00','\x03','\xCA','\x10','\xF7','\x60','\x00',
'\x20','\x42','\x41','\x53','\x49','\x43','\x20','\x42','\x59','\x54','\x45','\x53','\x20','\x46','\x52','\x45','\x45','\x0D','\x00','\x93','\x0D','\x20','\x20','\x20','\x20','\x2A','\x2A','\x2A','\x2A','\x20','\x43','\x4F',
'\x4D','\x4D','\x4F','\x44','\x4F','\x52','\x45','\x20','\x36','\x34','\x20','\x42','\x41','\x53','\x49','\x43','\x20','\x56','\x32','\x20','\x2A','\x2A','\x2A','\x2A','\x0D','\x0D','\x20','\x36','\x34','\x4B','\x20','\x52',
'\x41','\x4D','\x20','\x53','\x59','\x53','\x54','\x45','\x4D','\x20','\x20','\x00','\x81','\x48','\x20','\xC9','\xFF','\xAA','\x68','\x90','\x01','\x8A','\x60','\xAA','\xAA','\xAA','\xAA','\xAA','\xAA','\xAA','\xAA','\xAA',
'\xAA','\xAA','\xAA','\xAA','\xAA','\xAA','\xAA','\xAA','\xAA','\xAA','\xAA','\xAA','\xAA','\xAA','\xAA','\xAA','\xAA','\xAA','\xAA','\x85','\xA9','\xA9','\x01','\x85','\xAB','\x60','\xAD','\x86','\x02','\x91','\xF3','\x60',
'\x69','\x02','\xA4','\x91','\xC8','\xD0','\x04','\xC5','\xA1','\xD0','\xF7','\x60','\x19','\x26','\x44','\x19','\x1A','\x11','\xE8','\x0D','\x70','\x0C','\x06','\x06','\xD1','\x02','\x37','\x01','\xAE','\x00','\x69','\x00',
'\xA2','\x00','\xA0','\xDC','\x60','\xA2','\x28','\xA0','\x19','\x60','\xB0','\x07','\x86','\xD6','\x84','\xD3','\x20','\x6C','\xE5','\xA6','\xD6','\xA4','\xD3','\x60','\x20','\xA0','\xE5','\xA9','\x00','\x8D','\x91','\x02',
'\x85','\xCF','\xA9','\x48','\x8D','\x8F','\x02','\xA9','\xEB','\x8D','\x90','\x02','\xA9','\x0A','\x8D','\x89','\x02','\x8D','\x8C','\x02','\xA9','\x0E','\x8D','\x86','\x02','\xA9','\x04','\x8D','\x8B','\x02','\xA9','\x0C',
'\x85','\xCD','\x85','\xCC','\xAD','\x88','\x02','\x09','\x80','\xA8','\xA9','\x00','\xAA','\x94','\xD9','\x18','\x69','\x28','\x90','\x01','\xC8','\xE8','\xE0','\x1A','\xD0','\xF3','\xA9','\xFF','\x95','\xD9','\xA2','\x18',
'\x20','\xFF','\xE9','\xCA','\x10','\xFA','\xA0','\x00','\x84','\xD3','\x84','\xD6','\xA6','\xD6','\xA5','\xD3','\xB4','\xD9','\x30','\x08','\x18','\x69','\x28','\x85','\xD3','\xCA','\x10','\xF4','\x20','\xF0','\xE9','\xA9',
'\x27','\xE8','\xB4','\xD9','\x30','\x06','\x18','\x69','\x28','\xE8','\x10','\xF6','\x85','\xD5','\x4C','\x24','\xEA','\xE4','\xC9','\xF0','\x03','\x4C','\xED','\xE6','\x60','\xEA','\x20','\xA0','\xE5','\x4C','\x66','\xE5',
'\xA9','\x03','\x85','\x9A','\xA9','\x00','\x85','\x99','\xA2','\x2F','\xBD','\xB8','\xEC','\x9D','\xFF','\xCF','\xCA','\xD0','\xF7','\x60','\xAC','\x77','\x02','\xA2','\x00','\xBD','\x78','\x02','\x9D','\x77','\x02','\xE8',
'\xE4','\xC6','\xD0','\xF5','\xC6','\xC6','\x98','\x58','\x18','\x60','\x20','\x16','\xE7','\xA5','\xC6','\x85','\xCC','\x8D','\x92','\x02','\xF0','\xF7','\x78','\xA5','\xCF','\xF0','\x0C','\xA5','\xCE','\xAE','\x87','\x02',
'\xA0','\x00','\x84','\xCF','\x20','\x13','\xEA','\x20','\xB4','\xE5','\xC9','\x83','\xD0','\x10','\xA2','\x09','\x78','\x86','\xC6','\xBD','\xE6','\xEC','\x9D','\x76','\x02','\xCA','\xD0','\xF7','\xF0','\xCF','\xC9','\x0D',
'\xD0','\xC8','\xA4','\xD5','\x84','\xD0','\xB1','\xD1','\xC9','\x20','\xD0','\x03','\x88','\xD0','\xF7','\xC8','\x84','\xC8','\xA0','\x00','\x8C','\x92','\x02','\x84','\xD3','\x84','\xD4','\xA5','\xC9','\x30','\x1B','\xA6',
'\xD6','\x20','\x91','\xE5','\xE4','\xC9','\xD0','\x12','\xA5','\xCA','\x85','\xD3','\xC5','\xC8','\x90','\x0A','\xB0','\x2B','\x98','\x48','\x8A','\x48','\xA5','\xD0','\xF0','\x93','\xA4','\xD3','\xB1','\xD1','\x85','\xD7',
'\x29','\x3F','\x06','\xD7','\x24','\xD7','\x10','\x02','\x09','\x80','\x90','\x04','\xA6','\xD4','\xD0','\x04','\x70','\x02','\x09','\x40','\xE6','\xD3','\x20','\x84','\xE6','\xC4','\xC8','\xD0','\x17','\xA9','\x00','\x85',
'\xD0','\xA9','\x0D','\xA6','\x99','\xE0','\x03','\xF0','\x06','\xA6','\x9A','\xE0','\x03','\xF0','\x03','\x20','\x16','\xE7','\xA9','\x0D','\x85','\xD7','\x68','\xAA','\x68','\xA8','\xA5','\xD7','\xC9','\xDE','\xD0','\x02',
'\xA9','\xFF','\x18','\x60','\xC9','\x22','\xD0','\x08','\xA5','\xD4','\x49','\x01','\x85','\xD4','\xA9','\x22','\x60','\x09','\x40','\xA6','\xC7','\xF0','\x02','\x09','\x80','\xA6','\xD8','\xF0','\x02','\xC6','\xD8','\xAE',
'\x86','\x02','\x20','\x13','\xEA','\x20','\xB6','\xE6','\x68','\xA8','\xA5','\xD8','\xF0','\x02','\x46','\xD4','\x68','\xAA','\x68','\x18','\x58','\x60','\x20','\xB3','\xE8','\xE6','\xD3','\xA5','\xD5','\xC5','\xD3','\xB0',
'\x3F','\xC9','\x4F','\xF0','\x32','\xAD','\x92','\x02','\xF0','\x03','\x4C','\x67','\xE9','\xA6','\xD6','\xE0','\x19','\x90','\x07','\x20','\xEA','\xE8','\xC6','\xD6','\xA6','\xD6','\x16','\xD9','\x56','\xD9','\xE8','\xB5',
'\xD9','\x09','\x80','\x95','\xD9','\xCA','\xA5','\xD5','\x18','\x69','\x28','\x85','\xD5','\xB5','\xD9','\x30','\x03','\xCA','\xD0','\xF9','\x4C','\xF0','\xE9','\xC6','\xD6','\x20','\x7C','\xE8','\xA9','\x00','\x85','\xD3',
'\x60','\xA6','\xD6','\xD0','\x06','\x86','\xD3','\x68','\x68','\xD0','\x9D','\xCA','\x86','\xD6','\x20','\x6C','\xE5','\xA4','\xD5','\x84','\xD3','\x60','\x48','\x85','\xD7','\x8A','\x48','\x98','\x48','\xA9','\x00','\x85',
'\xD0','\xA4','\xD3','\xA5','\xD7','\x10','\x03','\x4C','\xD4','\xE7','\xC9','\x0D','\xD0','\x03','\x4C','\x91','\xE8','\xC9','\x20','\x90','\x10','\xC9','\x60','\x90','\x04','\x29','\xDF','\xD0','\x02','\x29','\x3F','\x20',
'\x84','\xE6','\x4C','\x93','\xE6','\xA6','\xD8','\xF0','\x03','\x4C','\x97','\xE6','\xC9','\x14','\xD0','\x2E','\x98','\xD0','\x06','\x20','\x01','\xE7','\x4C','\x73','\xE7','\x20','\xA1','\xE8','\x88','\x84','\xD3','\x20',
'\x24','\xEA','\xC8','\xB1','\xD1','\x88','\x91','\xD1','\xC8','\xB1','\xF3','\x88','\x91','\xF3','\xC8','\xC4','\xD5','\xD0','\xEF','\xA9','\x20','\x91','\xD1','\xAD','\x86','\x02','\x91','\xF3','\x10','\x4D','\xA6','\xD4',
'\xF0','\x03','\x4C','\x97','\xE6','\xC9','\x12','\xD0','\x02','\x85','\xC7','\xC9','\x13','\xD0','\x03','\x20','\x66','\xE5','\xC9','\x1D','\xD0','\x17','\xC8','\x20','\xB3','\xE8','\x84','\xD3','\x88','\xC4','\xD5','\x90',
'\x09','\xC6','\xD6','\x20','\x7C','\xE8','\xA0','\x00','\x84','\xD3','\x4C','\xA8','\xE6','\xC9','\x11','\xD0','\x1D','\x18','\x98','\x69','\x28','\xA8','\xE6','\xD6','\xC5','\xD5','\x90','\xEC','\xF0','\xEA','\xC6','\xD6',
'\xE9','\x28','\x90','\x04','\x85','\xD3','\xD0','\xF8','\x20','\x7C','\xE8','\x4C','\xA8','\xE6','\x20','\xCB','\xE8','\x4C','\x44','\xEC','\x29','\x7F','\xC9','\x7F','\xD0','\x02','\xA9','\x5E','\xC9','\x20','\x90','\x03',
'\x4C','\x91','\xE6','\xC9','\x0D','\xD0','\x03','\x4C','\x91','\xE8','\xA6','\xD4','\xD0','\x3F','\xC9','\x14','\xD0','\x37','\xA4','\xD5','\xB1','\xD1','\xC9','\x20','\xD0','\x04','\xC4','\xD3','\xD0','\x07','\xC0','\x4F',
'\xF0','\x24','\x20','\x65','\xE9','\xA4','\xD5','\x20','\x24','\xEA','\x88','\xB1','\xD1','\xC8','\x91','\xD1','\x88','\xB1','\xF3','\xC8','\x91','\xF3','\x88','\xC4','\xD3','\xD0','\xEF','\xA9','\x20','\x91','\xD1','\xAD',
'\x86','\x02','\x91','\xF3','\xE6','\xD8','\x4C','\xA8','\xE6','\xA6','\xD8','\xF0','\x05','\x09','\x40','\x4C','\x97','\xE6','\xC9','\x11','\xD0','\x16','\xA6','\xD6','\xF0','\x37','\xC6','\xD6','\xA5','\xD3','\x38','\xE9',
'\x28','\x90','\x04','\x85','\xD3','\x10','\x2A','\x20','\x6C','\xE5','\xD0','\x25','\xC9','\x12','\xD0','\x04','\xA9','\x00','\x85','\xC7','\xC9','\x1D','\xD0','\x12','\x98','\xF0','\x09','\x20','\xA1','\xE8','\x88','\x84',
'\xD3','\x4C','\xA8','\xE6','\x20','\x01','\xE7','\x4C','\xA8','\xE6','\xC9','\x13','\xD0','\x06','\x20','\x44','\xE5','\x4C','\xA8','\xE6','\x09','\x80','\x20','\xCB','\xE8','\x4C','\x4F','\xEC','\x46','\xC9','\xA6','\xD6',
'\xE8','\xE0','\x19','\xD0','\x03','\x20','\xEA','\xE8','\xB5','\xD9','\x10','\xF4','\x86','\xD6','\x4C','\x6C','\xE5','\xA2','\x00','\x86','\xD8','\x86','\xC7','\x86','\xD4','\x86','\xD3','\x20','\x7C','\xE8','\x4C','\xA8',
'\xE6','\xA2','\x02','\xA9','\x00','\xC5','\xD3','\xF0','\x07','\x18','\x69','\x28','\xCA','\xD0','\xF6','\x60','\xC6','\xD6','\x60','\xA2','\x02','\xA9','\x27','\xC5','\xD3','\xF0','\x07','\x18','\x69','\x28','\xCA','\xD0',
'\xF6','\x60','\xA6','\xD6','\xE0','\x19','\xF0','\x02','\xE6','\xD6','\x60','\xA2','\x0F','\xDD','\xDA','\xE8','\xF0','\x04','\xCA','\x10','\xF8','\x60','\x8E','\x86','\x02','\x60','\x90','\x05','\x1C','\x9F','\x9C','\x1E',
'\x1F','\x9E','\x81','\x95','\x96','\x97','\x98','\x99','\x9A','\x9B','\xA5','\xAC','\x48','\xA5','\xAD','\x48','\xA5','\xAE','\x48','\xA5','\xAF','\x48','\xA2','\xFF','\xC6','\xD6','\xC6','\xC9','\xCE','\xA5','\x02','\xE8',
'\x20','\xF0','\xE9','\xE0','\x18','\xB0','\x0C','\xBD','\xF1','\xEC','\x85','\xAC','\xB5','\xDA','\x20','\xC8','\xE9','\x30','\xEC','\x20','\xFF','\xE9','\xA2','\x00','\xB5','\xD9','\x29','\x7F','\xB4','\xDA','\x10','\x02',
'\x09','\x80','\x95','\xD9','\xE8','\xE0','\x18','\xD0','\xEF','\xA5','\xF1','\x09','\x80','\x85','\xF1','\xA5','\xD9','\x10','\xC3','\xE6','\xD6','\xEE','\xA5','\x02','\xA9','\x7F','\x8D','\x00','\xDC','\xAD','\x01','\xDC',
'\xC9','\xFB','\x08','\xA9','\x7F','\x8D','\x00','\xDC','\x28','\xD0','\x0B','\xA0','\x00','\xEA','\xCA','\xD0','\xFC','\x88','\xD0','\xF9','\x84','\xC6','\xA6','\xD6','\x68','\x85','\xAF','\x68','\x85','\xAE','\x68','\x85',
'\xAD','\x68','\x85','\xAC','\x60','\xA6','\xD6','\xE8','\xB5','\xD9','\x10','\xFB','\x8E','\xA5','\x02','\xE0','\x18','\xF0','\x0E','\x90','\x0C','\x20','\xEA','\xE8','\xAE','\xA5','\x02','\xCA','\xC6','\xD6','\x4C','\xDA',
'\xE6','\xA5','\xAC','\x48','\xA5','\xAD','\x48','\xA5','\xAE','\x48','\xA5','\xAF','\x48','\xA2','\x19','\xCA','\x20','\xF0','\xE9','\xEC','\xA5','\x02','\x90','\x0E','\xF0','\x0C','\xBD','\xEF','\xEC','\x85','\xAC','\xB5',
'\xD8','\x20','\xC8','\xE9','\x30','\xE9','\x20','\xFF','\xE9','\xA2','\x17','\xEC','\xA5','\x02','\x90','\x0F','\xB5','\xDA','\x29','\x7F','\xB4','\xD9','\x10','\x02','\x09','\x80','\x95','\xDA','\xCA','\xD0','\xEC','\xAE',
'\xA5','\x02','\x20','\xDA','\xE6','\x4C','\x58','\xE9','\x29','\x03','\x0D','\x88','\x02','\x85','\xAD','\x20','\xE0','\xE9','\xA0','\x27','\xB1','\xAC','\x91','\xD1','\xB1','\xAE','\x91','\xF3','\x88','\x10','\xF5','\x60',
'\x20','\x24','\xEA','\xA5','\xAC','\x85','\xAE','\xA5','\xAD','\x29','\x03','\x09','\xD8','\x85','\xAF','\x60','\xBD','\xF0','\xEC','\x85','\xD1','\xB5','\xD9','\x29','\x03','\x0D','\x88','\x02','\x85','\xD2','\x60','\xA0',
'\x27','\x20','\xF0','\xE9','\x20','\x24','\xEA','\x20','\xDA','\xE4','\xA9','\x20','\x91','\xD1','\x88','\x10','\xF6','\x60','\xEA','\xA8','\xA9','\x02','\x85','\xCD','\x20','\x24','\xEA','\x98','\xA4','\xD3','\x91','\xD1',
'\x8A','\x91','\xF3','\x60','\xA5','\xD1','\x85','\xF3','\xA5','\xD2','\x29','\x03','\x09','\xD8','\x85','\xF4','\x60','\x20','\xEA','\xFF','\xA5','\xCC','\xD0','\x29','\xC6','\xCD','\xD0','\x25','\xA9','\x14','\x85','\xCD',
'\xA4','\xD3','\x46','\xCF','\xAE','\x87','\x02','\xB1','\xD1','\xB0','\x11','\xE6','\xCF','\x85','\xCE','\x20','\x24','\xEA','\xB1','\xF3','\x8D','\x87','\x02','\xAE','\x86','\x02','\xA5','\xCE','\x49','\x80','\x20','\x1C',
'\xEA','\xA5','\x01','\x29','\x10','\xF0','\x0A','\xA0','\x00','\x84','\xC0','\xA5','\x01','\x09','\x20','\xD0','\x08','\xA5','\xC0','\xD0','\x06','\xA5','\x01','\x29','\x1F','\x85','\x01','\x20','\x87','\xEA','\xAD','\x0D',
'\xDC','\x68','\xA8','\x68','\xAA','\x68','\x40','\xA9','\x00','\x8D','\x8D','\x02','\xA0','\x40','\x84','\xCB','\x8D','\x00','\xDC','\xAE','\x01','\xDC','\xE0','\xFF','\xF0','\x61','\xA8','\xA9','\x81','\x85','\xF5','\xA9',
'\xEB','\x85','\xF6','\xA9','\xFE','\x8D','\x00','\xDC','\xA2','\x08','\x48','\xAD','\x01','\xDC','\xCD','\x01','\xDC','\xD0','\xF8','\x4A','\xB0','\x16','\x48','\xB1','\xF5','\xC9','\x05','\xB0','\x0C','\xC9','\x03','\xF0',
'\x08','\x0D','\x8D','\x02','\x8D','\x8D','\x02','\x10','\x02','\x84','\xCB','\x68','\xC8','\xC0','\x41','\xB0','\x0B','\xCA','\xD0','\xDF','\x38','\x68','\x2A','\x8D','\x00','\xDC','\xD0','\xCC','\x68','\x6C','\x8F','\x02',
'\xA4','\xCB','\xB1','\xF5','\xAA','\xC4','\xC5','\xF0','\x07','\xA0','\x10','\x8C','\x8C','\x02','\xD0','\x36','\x29','\x7F','\x2C','\x8A','\x02','\x30','\x16','\x70','\x49','\xC9','\x7F','\xF0','\x29','\xC9','\x14','\xF0',
'\x0C','\xC9','\x20','\xF0','\x08','\xC9','\x1D','\xF0','\x04','\xC9','\x11','\xD0','\x35','\xAC','\x8C','\x02','\xF0','\x05','\xCE','\x8C','\x02','\xD0','\x2B','\xCE','\x8B','\x02','\xD0','\x26','\xA0','\x04','\x8C','\x8B',
'\x02','\xA4','\xC6','\x88','\x10','\x1C','\xA4','\xCB','\x84','\xC5','\xAC','\x8D','\x02','\x8C','\x8E','\x02','\xE0','\xFF','\xF0','\x0E','\x8A','\xA6','\xC6','\xEC','\x89','\x02','\xB0','\x06','\x9D','\x77','\x02','\xE8',
'\x86','\xC6','\xA9','\x7F','\x8D','\x00','\xDC','\x60','\xAD','\x8D','\x02','\xC9','\x03','\xD0','\x15','\xCD','\x8E','\x02','\xF0','\xEE','\xAD','\x91','\x02','\x30','\x1D','\xAD','\x18','\xD0','\x49','\x02','\x8D','\x18',
'\xD0','\x4C','\x76','\xEB','\x0A','\xC9','\x08','\x90','\x02','\xA9','\x06','\xAA','\xBD','\x79','\xEB','\x85','\xF5','\xBD','\x7A','\xEB','\x85','\xF6','\x4C','\xE0','\xEA','\x81','\xEB','\xC2','\xEB','\x03','\xEC','\x78',
'\xEC','\x14','\x0D','\x1D','\x88','\x85','\x86','\x87','\x11','\x33','\x57','\x41','\x34','\x5A','\x53','\x45','\x01','\x35','\x52','\x44','\x36','\x43','\x46','\x54','\x58','\x37','\x59','\x47','\x38','\x42','\x48','\x55',
'\x56','\x39','\x49','\x4A','\x30','\x4D','\x4B','\x4F','\x4E','\x2B','\x50','\x4C','\x2D','\x2E','\x3A','\x40','\x2C','\x5C','\x2A','\x3B','\x13','\x01','\x3D','\x5E','\x2F','\x31','\x5F','\x04','\x32','\x20','\x02','\x51',
'\x03','\xFF','\x94','\x8D','\x9D','\x8C','\x89','\x8A','\x8B','\x91','\x23','\xD7','\xC1','\x24','\xDA','\xD3','\xC5','\x01','\x25','\xD2','\xC4','\x26','\xC3','\xC6','\xD4','\xD8','\x27','\xD9','\xC7','\x28','\xC2','\xC8',
'\xD5','\xD6','\x29','\xC9','\xCA','\x30','\xCD','\xCB','\xCF','\xCE','\xDB','\xD0','\xCC','\xDD','\x3E','\x5B','\xBA','\x3C','\xA9','\xC0','\x5D','\x93','\x01','\x3D','\xDE','\x3F','\x21','\x5F','\x04','\x22','\xA0','\x02',
'\xD1','\x83','\xFF','\x94','\x8D','\x9D','\x8C','\x89','\x8A','\x8B','\x91','\x96','\xB3','\xB0','\x97','\xAD','\xAE','\xB1','\x01','\x98','\xB2','\xAC','\x99','\xBC','\xBB','\xA3','\xBD','\x9A','\xB7','\xA5','\x9B','\xBF',
'\xB4','\xB8','\xBE','\x29','\xA2','\xB5','\x30','\xA7','\xA1','\xB9','\xAA','\xA6','\xAF','\xB6','\xDC','\x3E','\x5B','\xA4','\x3C','\xA8','\xDF','\x5D','\x93','\x01','\x3D','\xDE','\x3F','\x81','\x5F','\x04','\x95','\xA0',
'\x02','\xAB','\x83','\xFF','\xC9','\x0E','\xD0','\x07','\xAD','\x18','\xD0','\x09','\x02','\xD0','\x09','\xC9','\x8E','\xD0','\x0B','\xAD','\x18','\xD0','\x29','\xFD','\x8D','\x18','\xD0','\x4C','\xA8','\xE6','\xC9','\x08',
'\xD0','\x07','\xA9','\x80','\x0D','\x91','\x02','\x30','\x09','\xC9','\x09','\xD0','\xEE','\xA9','\x7F','\x2D','\x91','\x02','\x8D','\x91','\x02','\x4C','\xA8','\xE6','\xFF','\xFF','\xFF','\xFF','\xFF','\xFF','\xFF','\xFF',
'\x1C','\x17','\x01','\x9F','\x1A','\x13','\x05','\xFF','\x9C','\x12','\x04','\x1E','\x03','\x06','\x14','\x18','\x1F','\x19','\x07','\x9E','\x02','\x08','\x15','\x16','\x12','\x09','\x0A','\x92','\x0D','\x0B','\x0F','\x0E',
'\xFF','\x10','\x0C','\xFF','\xFF','\x1B','\x00','\xFF','\x1C','\xFF','\x1D','\xFF','\xFF','\x1F','\x1E','\xFF','\x90','\x06','\xFF','\x05','\xFF','\xFF','\x11','\xFF','\xFF','\x00','\x00','\x00','\x00','\x00','\x00','\x00',
'\x00','\x00','\x00','\x00','\x00','\x00','\x00','\x00','\x00','\x00','\x9B','\x37','\x00','\x00','\x00','\x08','\x00','\x14','\x0F','\x00','\x00','\x00','\x00','\x00','\x00','\x0E','\x06','\x01','\x02','\x03','\x04','\x00',
'\x01','\x02','\x03','\x04','\x05','\x06','\x07','\x4C','\x4F','\x41','\x44','\x0D','\x52','\x55','\x4E','\x0D','\x00','\x28','\x50','\x78','\xA0','\xC8','\xF0','\x18','\x40','\x68','\x90','\xB8','\xE0','\x08','\x30','\x58',
'\x80','\xA8','\xD0','\xF8','\x20','\x48','\x70','\x98','\xC0','\x09','\x40','\x2C','\x09','\x20','\x20','\xA4','\xF0','\x48','\x24','\x94','\x10','\x0A','\x38','\x66','\xA3','\x20','\x40','\xED','\x46','\x94','\x46','\xA3',
'\x68','\x85','\x95','\x78','\x20','\x97','\xEE','\xC9','\x3F','\xD0','\x03','\x20','\x85','\xEE','\xAD','\x00','\xDD','\x09','\x08','\x8D','\x00','\xDD','\x78','\x20','\x8E','\xEE','\x20','\x97','\xEE','\x20','\xB3','\xEE',
'\x78','\x20','\x97','\xEE','\x20','\xA9','\xEE','\xB0','\x64','\x20','\x85','\xEE','\x24','\xA3','\x10','\x0A','\x20','\xA9','\xEE','\x90','\xFB','\x20','\xA9','\xEE','\xB0','\xFB','\x20','\xA9','\xEE','\x90','\xFB','\x20',
'\x8E','\xEE','\xA9','\x08','\x85','\xA5','\xAD','\x00','\xDD','\xCD','\x00','\xDD','\xD0','\xF8','\x0A','\x90','\x3F','\x66','\x95','\xB0','\x05','\x20','\xA0','\xEE','\xD0','\x03','\x20','\x97','\xEE','\x20','\x85','\xEE',
'\xEA','\xEA','\xEA','\xEA','\xAD','\x00','\xDD','\x29','\xDF','\x09','\x10','\x8D','\x00','\xDD','\xC6','\xA5','\xD0','\xD4','\xA9','\x04','\x8D','\x07','\xDC','\xA9','\x19','\x8D','\x0F','\xDC','\xAD','\x0D','\xDC','\xAD',
'\x0D','\xDC','\x29','\x02','\xD0','\x0A','\x20','\xA9','\xEE','\xB0','\xF4','\x58','\x60','\xA9','\x80','\x2C','\xA9','\x03','\x20','\x1C','\xFE','\x58','\x18','\x90','\x4A','\x85','\x95','\x20','\x36','\xED','\xAD','\x00',
'\xDD','\x29','\xF7','\x8D','\x00','\xDD','\x60','\x85','\x95','\x20','\x36','\xED','\x78','\x20','\xA0','\xEE','\x20','\xBE','\xED','\x20','\x85','\xEE','\x20','\xA9','\xEE','\x30','\xFB','\x58','\x60','\x24','\x94','\x30',
'\x05','\x38','\x66','\x94','\xD0','\x05','\x48','\x20','\x40','\xED','\x68','\x85','\x95','\x18','\x60','\x78','\x20','\x8E','\xEE','\xAD','\x00','\xDD','\x09','\x08','\x8D','\x00','\xDD','\xA9','\x5F','\x2C','\xA9','\x3F',
'\x20','\x11','\xED','\x20','\xBE','\xED','\x8A','\xA2','\x0A','\xCA','\xD0','\xFD','\xAA','\x20','\x85','\xEE','\x4C','\x97','\xEE','\x78','\xA9','\x00','\x85','\xA5','\x20','\x85','\xEE','\x20','\xA9','\xEE','\x10','\xFB',
'\xA9','\x01','\x8D','\x07','\xDC','\xA9','\x19','\x8D','\x0F','\xDC','\x20','\x97','\xEE','\xAD','\x0D','\xDC','\xAD','\x0D','\xDC','\x29','\x02','\xD0','\x07','\x20','\xA9','\xEE','\x30','\xF4','\x10','\x18','\xA5','\xA5',
'\xF0','\x05','\xA9','\x02','\x4C','\xB2','\xED','\x20','\xA0','\xEE','\x20','\x85','\xEE','\xA9','\x40','\x20','\x1C','\xFE','\xE6','\xA5','\xD0','\xCA','\xA9','\x08','\x85','\xA5','\xAD','\x00','\xDD','\xCD','\x00','\xDD',
'\xD0','\xF8','\x0A','\x10','\xF5','\x66','\xA4','\xAD','\x00','\xDD','\xCD','\x00','\xDD','\xD0','\xF8','\x0A','\x30','\xF5','\xC6','\xA5','\xD0','\xE4','\x20','\xA0','\xEE','\x24','\x90','\x50','\x03','\x20','\x06','\xEE',
'\xA5','\xA4','\x58','\x18','\x60','\xAD','\x00','\xDD','\x29','\xEF','\x8D','\x00','\xDD','\x60','\xAD','\x00','\xDD','\x09','\x10','\x8D','\x00','\xDD','\x60','\xAD','\x00','\xDD','\x29','\xDF','\x8D','\x00','\xDD','\x60',
'\xAD','\x00','\xDD','\x09','\x20','\x8D','\x00','\xDD','\x60','\xAD','\x00','\xDD','\xCD','\x00','\xDD','\xD0','\xF8','\x0A','\x60','\x8A','\xA2','\xB8','\xCA','\xD0','\xFD','\xAA','\x60','\xA5','\xB4','\xF0','\x47','\x30',
'\x3F','\x46','\xB6','\xA2','\x00','\x90','\x01','\xCA','\x8A','\x45','\xBD','\x85','\xBD','\xC6','\xB4','\xF0','\x06','\x8A','\x29','\x04','\x85','\xB5','\x60','\xA9','\x20','\x2C','\x94','\x02','\xF0','\x14','\x30','\x1C',
'\x70','\x14','\xA5','\xBD','\xD0','\x01','\xCA','\xC6','\xB4','\xAD','\x93','\x02','\x10','\xE3','\xC6','\xB4','\xD0','\xDF','\xE6','\xB4','\xD0','\xF0','\xA5','\xBD','\xF0','\xED','\xD0','\xEA','\x70','\xE9','\x50','\xE6',
'\xE6','\xB4','\xA2','\xFF','\xD0','\xCB','\xAD','\x94','\x02','\x4A','\x90','\x07','\x2C','\x01','\xDD','\x10','\x1D','\x50','\x1E','\xA9','\x00','\x85','\xBD','\x85','\xB5','\xAE','\x98','\x02','\x86','\xB4','\xAC','\x9D',
'\x02','\xCC','\x9E','\x02','\xF0','\x13','\xB1','\xF9','\x85','\xB6','\xEE','\x9D','\x02','\x60','\xA9','\x40','\x2C','\xA9','\x10','\x0D','\x97','\x02','\x8D','\x97','\x02','\xA9','\x01','\x8D','\x0D','\xDD','\x4D','\xA1',
'\x02','\x09','\x80','\x8D','\xA1','\x02','\x8D','\x0D','\xDD','\x60','\xA2','\x09','\xA9','\x20','\x2C','\x93','\x02','\xF0','\x01','\xCA','\x50','\x02','\xCA','\xCA','\x60','\xA6','\xA9','\xD0','\x33','\xC6','\xA8','\xF0',
'\x36','\x30','\x0D','\xA5','\xA7','\x45','\xAB','\x85','\xAB','\x46','\xA7','\x66','\xAA','\x60','\xC6','\xA8','\xA5','\xA7','\xF0','\x67','\xAD','\x93','\x02','\x0A','\xA9','\x01','\x65','\xA8','\xD0','\xEF','\xA9','\x90',
'\x8D','\x0D','\xDD','\x0D','\xA1','\x02','\x8D','\xA1','\x02','\x85','\xA9','\xA9','\x02','\x4C','\x3B','\xEF','\xA5','\xA7','\xD0','\xEA','\x4C','\xD3','\xE4','\xAC','\x9B','\x02','\xC8','\xCC','\x9C','\x02','\xF0','\x2A',
'\x8C','\x9B','\x02','\x88','\xA5','\xAA','\xAE','\x98','\x02','\xE0','\x09','\xF0','\x04','\x4A','\xE8','\xD0','\xF8','\x91','\xF7','\xA9','\x20','\x2C','\x94','\x02','\xF0','\xB4','\x30','\xB1','\xA5','\xA7','\x45','\xAB',
'\xF0','\x03','\x70','\xA9','\x2C','\x50','\xA6','\xA9','\x01','\x2C','\xA9','\x04','\x2C','\xA9','\x80','\x2C','\xA9','\x02','\x0D','\x97','\x02','\x8D','\x97','\x02','\x4C','\x7E','\xEF','\xA5','\xAA','\xD0','\xF1','\xF0',
'\xEC','\x85','\x9A','\xAD','\x94','\x02','\x4A','\x90','\x29','\xA9','\x02','\x2C','\x01','\xDD','\x10','\x1D','\xD0','\x20','\xAD','\xA1','\x02','\x29','\x02','\xD0','\xF9','\x2C','\x01','\xDD','\x70','\xFB','\xAD','\x01',
'\xDD','\x09','\x02','\x8D','\x01','\xDD','\x2C','\x01','\xDD','\x70','\x07','\x30','\xF9','\xA9','\x40','\x8D','\x97','\x02','\x18','\x60','\x20','\x28','\xF0','\xAC','\x9E','\x02','\xC8','\xCC','\x9D','\x02','\xF0','\xF4',
'\x8C','\x9E','\x02','\x88','\xA5','\x9E','\x91','\xF9','\xAD','\xA1','\x02','\x4A','\xB0','\x1E','\xA9','\x10','\x8D','\x0E','\xDD','\xAD','\x99','\x02','\x8D','\x04','\xDD','\xAD','\x9A','\x02','\x8D','\x05','\xDD','\xA9',
'\x81','\x20','\x3B','\xEF','\x20','\x06','\xEF','\xA9','\x11','\x8D','\x0E','\xDD','\x60','\x85','\x99','\xAD','\x94','\x02','\x4A','\x90','\x28','\x29','\x08','\xF0','\x24','\xA9','\x02','\x2C','\x01','\xDD','\x10','\xAD',
'\xF0','\x22','\xAD','\xA1','\x02','\x4A','\xB0','\xFA','\xAD','\x01','\xDD','\x29','\xFD','\x8D','\x01','\xDD','\xAD','\x01','\xDD','\x29','\x04','\xF0','\xF9','\xA9','\x90','\x18','\x4C','\x3B','\xEF','\xAD','\xA1','\x02',
'\x29','\x12','\xF0','\xF3','\x18','\x60','\xAD','\x97','\x02','\xAC','\x9C','\x02','\xCC','\x9B','\x02','\xF0','\x0B','\x29','\xF7','\x8D','\x97','\x02','\xB1','\xF7','\xEE','\x9C','\x02','\x60','\x09','\x08','\x8D','\x97',
'\x02','\xA9','\x00','\x60','\x48','\xAD','\xA1','\x02','\xF0','\x11','\xAD','\xA1','\x02','\x29','\x03','\xD0','\xF9','\xA9','\x10','\x8D','\x0D','\xDD','\xA9','\x00','\x8D','\xA1','\x02','\x68','\x60','\x0D','\x49','\x2F',
'\x4F','\x20','\x45','\x52','\x52','\x4F','\x52','\x20','\xA3','\x0D','\x53','\x45','\x41','\x52','\x43','\x48','\x49','\x4E','\x47','\xA0','\x46','\x4F','\x52','\xA0','\x0D','\x50','\x52','\x45','\x53','\x53','\x20','\x50',
'\x4C','\x41','\x59','\x20','\x4F','\x4E','\x20','\x54','\x41','\x50','\xC5','\x50','\x52','\x45','\x53','\x53','\x20','\x52','\x45','\x43','\x4F','\x52','\x44','\x20','\x26','\x20','\x50','\x4C','\x41','\x59','\x20','\x4F',
'\x4E','\x20','\x54','\x41','\x50','\xC5','\x0D','\x4C','\x4F','\x41','\x44','\x49','\x4E','\xC7','\x0D','\x53','\x41','\x56','\x49','\x4E','\x47','\xA0','\x0D','\x56','\x45','\x52','\x49','\x46','\x59','\x49','\x4E','\xC7',
'\x0D','\x46','\x4F','\x55','\x4E','\x44','\xA0','\x0D','\x4F','\x4B','\x8D','\x24','\x9D','\x10','\x0D','\xB9','\xBD','\xF0','\x08','\x29','\x7F','\x20','\xD2','\xFF','\xC8','\x28','\x10','\xF3','\x18','\x60','\xA5','\x99',
'\xD0','\x08','\xA5','\xC6','\xF0','\x0F','\x78','\x4C','\xB4','\xE5','\xC9','\x02','\xD0','\x18','\x84','\x97','\x20','\x86','\xF0','\xA4','\x97','\x18','\x60','\xA5','\x99','\xD0','\x0B','\xA5','\xD3','\x85','\xCA','\xA5',
'\xD6','\x85','\xC9','\x4C','\x32','\xE6','\xC9','\x03','\xD0','\x09','\x85','\xD0','\xA5','\xD5','\x85','\xC8','\x4C','\x32','\xE6','\xB0','\x38','\xC9','\x02','\xF0','\x3F','\x86','\x97','\x20','\x99','\xF1','\xB0','\x16',
'\x48','\x20','\x99','\xF1','\xB0','\x0D','\xD0','\x05','\xA9','\x40','\x20','\x1C','\xFE','\xC6','\xA6','\xA6','\x97','\x68','\x60','\xAA','\x68','\x8A','\xA6','\x97','\x60','\x20','\x0D','\xF8','\xD0','\x0B','\x20','\x41',
'\xF8','\xB0','\x11','\xA9','\x00','\x85','\xA6','\xF0','\xF0','\xB1','\xB2','\x18','\x60','\xA5','\x90','\xF0','\x04','\xA9','\x0D','\x18','\x60','\x4C','\x13','\xEE','\x20','\x4E','\xF1','\xB0','\xF7','\xC9','\x00','\xD0',
'\xF2','\xAD','\x97','\x02','\x29','\x60','\xD0','\xE9','\xF0','\xEE','\x48','\xA5','\x9A','\xC9','\x03','\xD0','\x04','\x68','\x4C','\x16','\xE7','\x90','\x04','\x68','\x4C','\xDD','\xED','\x4A','\x68','\x85','\x9E','\x8A',
'\x48','\x98','\x48','\x90','\x23','\x20','\x0D','\xF8','\xD0','\x0E','\x20','\x64','\xF8','\xB0','\x0E','\xA9','\x02','\xA0','\x00','\x91','\xB2','\xC8','\x84','\xA6','\xA5','\x9E','\x91','\xB2','\x18','\x68','\xA8','\x68',
'\xAA','\xA5','\x9E','\x90','\x02','\xA9','\x00','\x60','\x20','\x17','\xF0','\x4C','\xFC','\xF1','\x20','\x0F','\xF3','\xF0','\x03','\x4C','\x01','\xF7','\x20','\x1F','\xF3','\xA5','\xBA','\xF0','\x16','\xC9','\x03','\xF0',
'\x12','\xB0','\x14','\xC9','\x02','\xD0','\x03','\x4C','\x4D','\xF0','\xA6','\xB9','\xE0','\x60','\xF0','\x03','\x4C','\x0A','\xF7','\x85','\x99','\x18','\x60','\xAA','\x20','\x09','\xED','\xA5','\xB9','\x10','\x06','\x20',
'\xCC','\xED','\x4C','\x48','\xF2','\x20','\xC7','\xED','\x8A','\x24','\x90','\x10','\xE6','\x4C','\x07','\xF7','\x20','\x0F','\xF3','\xF0','\x03','\x4C','\x01','\xF7','\x20','\x1F','\xF3','\xA5','\xBA','\xD0','\x03','\x4C',
'\x0D','\xF7','\xC9','\x03','\xF0','\x0F','\xB0','\x11','\xC9','\x02','\xD0','\x03','\x4C','\xE1','\xEF','\xA6','\xB9','\xE0','\x60','\xF0','\xEA','\x85','\x9A','\x18','\x60','\xAA','\x20','\x0C','\xED','\xA5','\xB9','\x10',
'\x05','\x20','\xBE','\xED','\xD0','\x03','\x20','\xB9','\xED','\x8A','\x24','\x90','\x10','\xE7','\x4C','\x07','\xF7','\x20','\x14','\xF3','\xF0','\x02','\x18','\x60','\x20','\x1F','\xF3','\x8A','\x48','\xA5','\xBA','\xF0',
'\x50','\xC9','\x03','\xF0','\x4C','\xB0','\x47','\xC9','\x02','\xD0','\x1D','\x68','\x20','\xF2','\xF2','\x20','\x83','\xF4','\x20','\x27','\xFE','\xA5','\xF8','\xF0','\x01','\xC8','\xA5','\xFA','\xF0','\x01','\xC8','\xA9',
'\x00','\x85','\xF8','\x85','\xFA','\x4C','\x7D','\xF4','\xA5','\xB9','\x29','\x0F','\xF0','\x23','\x20','\xD0','\xF7','\xA9','\x00','\x38','\x20','\xDD','\xF1','\x20','\x64','\xF8','\x90','\x04','\x68','\xA9','\x00','\x60',
'\xA5','\xB9','\xC9','\x62','\xD0','\x0B','\xA9','\x05','\x20','\x6A','\xF7','\x4C','\xF1','\xF2','\x20','\x42','\xF6','\x68','\xAA','\xC6','\x98','\xE4','\x98','\xF0','\x14','\xA4','\x98','\xB9','\x59','\x02','\x9D','\x59',
'\x02','\xB9','\x63','\x02','\x9D','\x63','\x02','\xB9','\x6D','\x02','\x9D','\x6D','\x02','\x18','\x60','\xA9','\x00','\x85','\x90','\x8A','\xA6','\x98','\xCA','\x30','\x15','\xDD','\x59','\x02','\xD0','\xF8','\x60','\xBD',
'\x59','\x02','\x85','\xB8','\xBD','\x63','\x02','\x85','\xBA','\xBD','\x6D','\x02','\x85','\xB9','\x60','\xA9','\x00','\x85','\x98','\xA2','\x03','\xE4','\x9A','\xB0','\x03','\x20','\xFE','\xED','\xE4','\x99','\xB0','\x03',
'\x20','\xEF','\xED','\x86','\x9A','\xA9','\x00','\x85','\x99','\x60','\xA6','\xB8','\xD0','\x03','\x4C','\x0A','\xF7','\x20','\x0F','\xF3','\xD0','\x03','\x4C','\xFE','\xF6','\xA6','\x98','\xE0','\x0A','\x90','\x03','\x4C',
'\xFB','\xF6','\xE6','\x98','\xA5','\xB8','\x9D','\x59','\x02','\xA5','\xB9','\x09','\x60','\x85','\xB9','\x9D','\x6D','\x02','\xA5','\xBA','\x9D','\x63','\x02','\xF0','\x5A','\xC9','\x03','\xF0','\x56','\x90','\x05','\x20',
'\xD5','\xF3','\x90','\x4F','\xC9','\x02','\xD0','\x03','\x4C','\x09','\xF4','\x20','\xD0','\xF7','\xB0','\x03','\x4C','\x13','\xF7','\xA5','\xB9','\x29','\x0F','\xD0','\x1F','\x20','\x17','\xF8','\xB0','\x36','\x20','\xAF',
'\xF5','\xA5','\xB7','\xF0','\x0A','\x20','\xEA','\xF7','\x90','\x18','\xF0','\x28','\x4C','\x04','\xF7','\x20','\x2C','\xF7','\xF0','\x20','\x90','\x0C','\xB0','\xF4','\x20','\x38','\xF8','\xB0','\x17','\xA9','\x04','\x20',
'\x6A','\xF7','\xA9','\xBF','\xA4','\xB9','\xC0','\x60','\xF0','\x07','\xA0','\x00','\xA9','\x02','\x91','\xB2','\x98','\x85','\xA6','\x18','\x60','\xA5','\xB9','\x30','\xFA','\xA4','\xB7','\xF0','\xF6','\xA9','\x00','\x85',
'\x90','\xA5','\xBA','\x20','\x0C','\xED','\xA5','\xB9','\x09','\xF0','\x20','\xB9','\xED','\xA5','\x90','\x10','\x05','\x68','\x68','\x4C','\x07','\xF7','\xA5','\xB7','\xF0','\x0C','\xA0','\x00','\xB1','\xBB','\x20','\xDD',
'\xED','\xC8','\xC4','\xB7','\xD0','\xF6','\x4C','\x54','\xF6','\x20','\x83','\xF4','\x8C','\x97','\x02','\xC4','\xB7','\xF0','\x0A','\xB1','\xBB','\x99','\x93','\x02','\xC8','\xC0','\x04','\xD0','\xF2','\x20','\x4A','\xEF',
'\x8E','\x98','\x02','\xAD','\x93','\x02','\x29','\x0F','\xF0','\x1C','\x0A','\xAA','\xAD','\xA6','\x02','\xD0','\x09','\xBC','\xC1','\xFE','\xBD','\xC0','\xFE','\x4C','\x40','\xF4','\xBC','\xEB','\xE4','\xBD','\xEA','\xE4',
'\x8C','\x96','\x02','\x8D','\x95','\x02','\xAD','\x95','\x02','\x0A','\x20','\x2E','\xFF','\xAD','\x94','\x02','\x4A','\x90','\x09','\xAD','\x01','\xDD','\x0A','\xB0','\x03','\x20','\x0D','\xF0','\xAD','\x9B','\x02','\x8D',
'\x9C','\x02','\xAD','\x9E','\x02','\x8D','\x9D','\x02','\x20','\x27','\xFE','\xA5','\xF8','\xD0','\x05','\x88','\x84','\xF8','\x86','\xF7','\xA5','\xFA','\xD0','\x05','\x88','\x84','\xFA','\x86','\xF9','\x38','\xA9','\xF0',
'\x4C','\x2D','\xFE','\xA9','\x7F','\x8D','\x0D','\xDD','\xA9','\x06','\x8D','\x03','\xDD','\x8D','\x01','\xDD','\xA9','\x04','\x0D','\x00','\xDD','\x8D','\x00','\xDD','\xA0','\x00','\x8C','\xA1','\x02','\x60','\x86','\xC3',
'\x84','\xC4','\x6C','\x30','\x03','\x85','\x93','\xA9','\x00','\x85','\x90','\xA5','\xBA','\xD0','\x03','\x4C','\x13','\xF7','\xC9','\x03','\xF0','\xF9','\x90','\x7B','\xA4','\xB7','\xD0','\x03','\x4C','\x10','\xF7','\xA6',
'\xB9','\x20','\xAF','\xF5','\xA9','\x60','\x85','\xB9','\x20','\xD5','\xF3','\xA5','\xBA','\x20','\x09','\xED','\xA5','\xB9','\x20','\xC7','\xED','\x20','\x13','\xEE','\x85','\xAE','\xA5','\x90','\x4A','\x4A','\xB0','\x50',
'\x20','\x13','\xEE','\x85','\xAF','\x8A','\xD0','\x08','\xA5','\xC3','\x85','\xAE','\xA5','\xC4','\x85','\xAF','\x20','\xD2','\xF5','\xA9','\xFD','\x25','\x90','\x85','\x90','\x20','\xE1','\xFF','\xD0','\x03','\x4C','\x33',
'\xF6','\x20','\x13','\xEE','\xAA','\xA5','\x90','\x4A','\x4A','\xB0','\xE8','\x8A','\xA4','\x93','\xF0','\x0C','\xA0','\x00','\xD1','\xAE','\xF0','\x08','\xA9','\x10','\x20','\x1C','\xFE','\x2C','\x91','\xAE','\xE6','\xAE',
'\xD0','\x02','\xE6','\xAF','\x24','\x90','\x50','\xCB','\x20','\xEF','\xED','\x20','\x42','\xF6','\x90','\x79','\x4C','\x04','\xF7','\x4A','\xB0','\x03','\x4C','\x13','\xF7','\x20','\xD0','\xF7','\xB0','\x03','\x4C','\x13',
'\xF7','\x20','\x17','\xF8','\xB0','\x68','\x20','\xAF','\xF5','\xA5','\xB7','\xF0','\x09','\x20','\xEA','\xF7','\x90','\x0B','\xF0','\x5A','\xB0','\xDA','\x20','\x2C','\xF7','\xF0','\x53','\xB0','\xD3','\xA5','\x90','\x29',
'\x10','\x38','\xD0','\x4A','\xE0','\x01','\xF0','\x11','\xE0','\x03','\xD0','\xDD','\xA0','\x01','\xB1','\xB2','\x85','\xC3','\xC8','\xB1','\xB2','\x85','\xC4','\xB0','\x04','\xA5','\xB9','\xD0','\xEF','\xA0','\x03','\xB1',
'\xB2','\xA0','\x01','\xF1','\xB2','\xAA','\xA0','\x04','\xB1','\xB2','\xA0','\x02','\xF1','\xB2','\xA8','\x18','\x8A','\x65','\xC3','\x85','\xAE','\x98','\x65','\xC4','\x85','\xAF','\xA5','\xC3','\x85','\xC1','\xA5','\xC4',
'\x85','\xC2','\x20','\xD2','\xF5','\x20','\x4A','\xF8','\x24','\x18','\xA6','\xAE','\xA4','\xAF','\x60','\xA5','\x9D','\x10','\x1E','\xA0','\x0C','\x20','\x2F','\xF1','\xA5','\xB7','\xF0','\x15','\xA0','\x17','\x20','\x2F',
'\xF1','\xA4','\xB7','\xF0','\x0C','\xA0','\x00','\xB1','\xBB','\x20','\xD2','\xFF','\xC8','\xC4','\xB7','\xD0','\xF6','\x60','\xA0','\x49','\xA5','\x93','\xF0','\x02','\xA0','\x59','\x4C','\x2B','\xF1','\x86','\xAE','\x84',
'\xAF','\xAA','\xB5','\x00','\x85','\xC1','\xB5','\x01','\x85','\xC2','\x6C','\x32','\x03','\xA5','\xBA','\xD0','\x03','\x4C','\x13','\xF7','\xC9','\x03','\xF0','\xF9','\x90','\x5F','\xA9','\x61','\x85','\xB9','\xA4','\xB7',
'\xD0','\x03','\x4C','\x10','\xF7','\x20','\xD5','\xF3','\x20','\x8F','\xF6','\xA5','\xBA','\x20','\x0C','\xED','\xA5','\xB9','\x20','\xB9','\xED','\xA0','\x00','\x20','\x8E','\xFB','\xA5','\xAC','\x20','\xDD','\xED','\xA5',
'\xAD','\x20','\xDD','\xED','\x20','\xD1','\xFC','\xB0','\x16','\xB1','\xAC','\x20','\xDD','\xED','\x20','\xE1','\xFF','\xD0','\x07','\x20','\x42','\xF6','\xA9','\x00','\x38','\x60','\x20','\xDB','\xFC','\xD0','\xE5','\x20',
'\xFE','\xED','\x24','\xB9','\x30','\x11','\xA5','\xBA','\x20','\x0C','\xED','\xA5','\xB9','\x29','\xEF','\x09','\xE0','\x20','\xB9','\xED','\x20','\xFE','\xED','\x18','\x60','\x4A','\xB0','\x03','\x4C','\x13','\xF7','\x20',
'\xD0','\xF7','\x90','\x8D','\x20','\x38','\xF8','\xB0','\x25','\x20','\x8F','\xF6','\xA2','\x03','\xA5','\xB9','\x29','\x01','\xD0','\x02','\xA2','\x01','\x8A','\x20','\x6A','\xF7','\xB0','\x12','\x20','\x67','\xF8','\xB0',
'\x0D','\xA5','\xB9','\x29','\x02','\xF0','\x06','\xA9','\x05','\x20','\x6A','\xF7','\x24','\x18','\x60','\xA5','\x9D','\x10','\xFB','\xA0','\x51','\x20','\x2F','\xF1','\x4C','\xC1','\xF5','\xA2','\x00','\xE6','\xA2','\xD0',
'\x06','\xE6','\xA1','\xD0','\x02','\xE6','\xA0','\x38','\xA5','\xA2','\xE9','\x01','\xA5','\xA1','\xE9','\x1A','\xA5','\xA0','\xE9','\x4F','\x90','\x06','\x86','\xA0','\x86','\xA1','\x86','\xA2','\xAD','\x01','\xDC','\xCD',
'\x01','\xDC','\xD0','\xF8','\xAA','\x30','\x13','\xA2','\xBD','\x8E','\x00','\xDC','\xAE','\x01','\xDC','\xEC','\x01','\xDC','\xD0','\xF8','\x8D','\x00','\xDC','\xE8','\xD0','\x02','\x85','\x91','\x60','\x78','\xA5','\xA2',
'\xA6','\xA1','\xA4','\xA0','\x78','\x85','\xA2','\x86','\xA1','\x84','\xA0','\x58','\x60','\xA5','\x91','\xC9','\x7F','\xD0','\x07','\x08','\x20','\xCC','\xFF','\x85','\xC6','\x28','\x60','\xA9','\x01','\x2C','\xA9','\x02',
'\x2C','\xA9','\x03','\x2C','\xA9','\x04','\x2C','\xA9','\x05','\x2C','\xA9','\x06','\x2C','\xA9','\x07','\x2C','\xA9','\x08','\x2C','\xA9','\x09','\x48','\x20','\xCC','\xFF','\xA0','\x00','\x24','\x9D','\x50','\x0A','\x20',
'\x2F','\xF1','\x68','\x48','\x09','\x30','\x20','\xD2','\xFF','\x68','\x38','\x60','\xA5','\x93','\x48','\x20','\x41','\xF8','\x68','\x85','\x93','\xB0','\x32','\xA0','\x00','\xB1','\xB2','\xC9','\x05','\xF0','\x2A','\xC9',
'\x01','\xF0','\x08','\xC9','\x03','\xF0','\x04','\xC9','\x04','\xD0','\xE1','\xAA','\x24','\x9D','\x10','\x17','\xA0','\x63','\x20','\x2F','\xF1','\xA0','\x05','\xB1','\xB2','\x20','\xD2','\xFF','\xC8','\xC0','\x15','\xD0',
'\xF6','\xA5','\xA1','\x20','\xE0','\xE4','\xEA','\x18','\x88','\x60','\x85','\x9E','\x20','\xD0','\xF7','\x90','\x5E','\xA5','\xC2','\x48','\xA5','\xC1','\x48','\xA5','\xAF','\x48','\xA5','\xAE','\x48','\xA0','\xBF','\xA9',
'\x20','\x91','\xB2','\x88','\xD0','\xFB','\xA5','\x9E','\x91','\xB2','\xC8','\xA5','\xC1','\x91','\xB2','\xC8','\xA5','\xC2','\x91','\xB2','\xC8','\xA5','\xAE','\x91','\xB2','\xC8','\xA5','\xAF','\x91','\xB2','\xC8','\x84',
'\x9F','\xA0','\x00','\x84','\x9E','\xA4','\x9E','\xC4','\xB7','\xF0','\x0C','\xB1','\xBB','\xA4','\x9F','\x91','\xB2','\xE6','\x9E','\xE6','\x9F','\xD0','\xEE','\x20','\xD7','\xF7','\xA9','\x69','\x85','\xAB','\x20','\x6B',
'\xF8','\xA8','\x68','\x85','\xAE','\x68','\x85','\xAF','\x68','\x85','\xC1','\x68','\x85','\xC2','\x98','\x60','\xA6','\xB2','\xA4','\xB3','\xC0','\x02','\x60','\x20','\xD0','\xF7','\x8A','\x85','\xC1','\x18','\x69','\xC0',
'\x85','\xAE','\x98','\x85','\xC2','\x69','\x00','\x85','\xAF','\x60','\x20','\x2C','\xF7','\xB0','\x1D','\xA0','\x05','\x84','\x9F','\xA0','\x00','\x84','\x9E','\xC4','\xB7','\xF0','\x10','\xB1','\xBB','\xA4','\x9F','\xD1',
'\xB2','\xD0','\xE7','\xE6','\x9E','\xE6','\x9F','\xA4','\x9E','\xD0','\xEC','\x18','\x60','\x20','\xD0','\xF7','\xE6','\xA6','\xA4','\xA6','\xC0','\xC0','\x60','\x20','\x2E','\xF8','\xF0','\x1A','\xA0','\x1B','\x20','\x2F',
'\xF1','\x20','\xD0','\xF8','\x20','\x2E','\xF8','\xD0','\xF8','\xA0','\x6A','\x4C','\x2F','\xF1','\xA9','\x10','\x24','\x01','\xD0','\x02','\x24','\x01','\x18','\x60','\x20','\x2E','\xF8','\xF0','\xF9','\xA0','\x2E','\xD0',
'\xDD','\xA9','\x00','\x85','\x90','\x85','\x93','\x20','\xD7','\xF7','\x20','\x17','\xF8','\xB0','\x1F','\x78','\xA9','\x00','\x85','\xAA','\x85','\xB4','\x85','\xB0','\x85','\x9E','\x85','\x9F','\x85','\x9C','\xA9','\x90',
'\xA2','\x0E','\xD0','\x11','\x20','\xD7','\xF7','\xA9','\x14','\x85','\xAB','\x20','\x38','\xF8','\xB0','\x6C','\x78','\xA9','\x82','\xA2','\x08','\xA0','\x7F','\x8C','\x0D','\xDC','\x8D','\x0D','\xDC','\xAD','\x0E','\xDC',
'\x09','\x19','\x8D','\x0F','\xDC','\x29','\x91','\x8D','\xA2','\x02','\x20','\xA4','\xF0','\xAD','\x11','\xD0','\x29','\xEF','\x8D','\x11','\xD0','\xAD','\x14','\x03','\x8D','\x9F','\x02','\xAD','\x15','\x03','\x8D','\xA0',
'\x02','\x20','\xBD','\xFC','\xA9','\x02','\x85','\xBE','\x20','\x97','\xFB','\xA5','\x01','\x29','\x1F','\x85','\x01','\x85','\xC0','\xA2','\xFF','\xA0','\xFF','\x88','\xD0','\xFD','\xCA','\xD0','\xF8','\x58','\xAD','\xA0',
'\x02','\xCD','\x15','\x03','\x18','\xF0','\x15','\x20','\xD0','\xF8','\x20','\xBC','\xF6','\x4C','\xBE','\xF8','\x20','\xE1','\xFF','\x18','\xD0','\x0B','\x20','\x93','\xFC','\x38','\x68','\x68','\xA9','\x00','\x8D','\xA0',
'\x02','\x60','\x86','\xB1','\xA5','\xB0','\x0A','\x0A','\x18','\x65','\xB0','\x18','\x65','\xB1','\x85','\xB1','\xA9','\x00','\x24','\xB0','\x30','\x01','\x2A','\x06','\xB1','\x2A','\x06','\xB1','\x2A','\xAA','\xAD','\x06',
'\xDC','\xC9','\x16','\x90','\xF9','\x65','\xB1','\x8D','\x04','\xDC','\x8A','\x6D','\x07','\xDC','\x8D','\x05','\xDC','\xAD','\xA2','\x02','\x8D','\x0E','\xDC','\x8D','\xA4','\x02','\xAD','\x0D','\xDC','\x29','\x10','\xF0',
'\x09','\xA9','\xF9','\x48','\xA9','\x2A','\x48','\x4C','\x43','\xFF','\x58','\x60','\xAE','\x07','\xDC','\xA0','\xFF','\x98','\xED','\x06','\xDC','\xEC','\x07','\xDC','\xD0','\xF2','\x86','\xB1','\xAA','\x8C','\x06','\xDC',
'\x8C','\x07','\xDC','\xA9','\x19','\x8D','\x0F','\xDC','\xAD','\x0D','\xDC','\x8D','\xA3','\x02','\x98','\xE5','\xB1','\x86','\xB1','\x4A','\x66','\xB1','\x4A','\x66','\xB1','\xA5','\xB0','\x18','\x69','\x3C','\xC5','\xB1',
'\xB0','\x4A','\xA6','\x9C','\xF0','\x03','\x4C','\x60','\xFA','\xA6','\xA3','\x30','\x1B','\xA2','\x00','\x69','\x30','\x65','\xB0','\xC5','\xB1','\xB0','\x1C','\xE8','\x69','\x26','\x65','\xB0','\xC5','\xB1','\xB0','\x17',
'\x69','\x2C','\x65','\xB0','\xC5','\xB1','\x90','\x03','\x4C','\x10','\xFA','\xA5','\xB4','\xF0','\x1D','\x85','\xA8','\xD0','\x19','\xE6','\xA9','\xB0','\x02','\xC6','\xA9','\x38','\xE9','\x13','\xE5','\xB1','\x65','\x92',
'\x85','\x92','\xA5','\xA4','\x49','\x01','\x85','\xA4','\xF0','\x2B','\x86','\xD7','\xA5','\xB4','\xF0','\x22','\xAD','\xA3','\x02','\x29','\x01','\xD0','\x05','\xAD','\xA4','\x02','\xD0','\x16','\xA9','\x00','\x85','\xA4',
'\x8D','\xA4','\x02','\xA5','\xA3','\x10','\x30','\x30','\xBF','\xA2','\xA6','\x20','\xE2','\xF8','\xA5','\x9B','\xD0','\xB9','\x4C','\xBC','\xFE','\xA5','\x92','\xF0','\x07','\x30','\x03','\xC6','\xB0','\x2C','\xE6','\xB0',
'\xA9','\x00','\x85','\x92','\xE4','\xD7','\xD0','\x0F','\x8A','\xD0','\xA0','\xA5','\xA9','\x30','\xBD','\xC9','\x10','\x90','\xB9','\x85','\x96','\xB0','\xB5','\x8A','\x45','\x9B','\x85','\x9B','\xA5','\xB4','\xF0','\xD2',
'\xC6','\xA3','\x30','\xC5','\x46','\xD7','\x66','\xBF','\xA2','\xDA','\x20','\xE2','\xF8','\x4C','\xBC','\xFE','\xA5','\x96','\xF0','\x04','\xA5','\xB4','\xF0','\x07','\xA5','\xA3','\x30','\x03','\x4C','\x97','\xF9','\x46',
'\xB1','\xA9','\x93','\x38','\xE5','\xB1','\x65','\xB0','\x0A','\xAA','\x20','\xE2','\xF8','\xE6','\x9C','\xA5','\xB4','\xD0','\x11','\xA5','\x96','\xF0','\x26','\x85','\xA8','\xA9','\x00','\x85','\x96','\xA9','\x81','\x8D',
'\x0D','\xDC','\x85','\xB4','\xA5','\x96','\x85','\xB5','\xF0','\x09','\xA9','\x00','\x85','\xB4','\xA9','\x01','\x8D','\x0D','\xDC','\xA5','\xBF','\x85','\xBD','\xA5','\xA8','\x05','\xA9','\x85','\xB6','\x4C','\xBC','\xFE',
'\x20','\x97','\xFB','\x85','\x9C','\xA2','\xDA','\x20','\xE2','\xF8','\xA5','\xBE','\xF0','\x02','\x85','\xA7','\xA9','\x0F','\x24','\xAA','\x10','\x17','\xA5','\xB5','\xD0','\x0C','\xA6','\xBE','\xCA','\xD0','\x0B','\xA9',
'\x08','\x20','\x1C','\xFE','\xD0','\x04','\xA9','\x00','\x85','\xAA','\x4C','\xBC','\xFE','\x70','\x31','\xD0','\x18','\xA5','\xB5','\xD0','\xF5','\xA5','\xB6','\xD0','\xF1','\xA5','\xA7','\x4A','\xA5','\xBD','\x30','\x03',
'\x90','\x18','\x18','\xB0','\x15','\x29','\x0F','\x85','\xAA','\xC6','\xAA','\xD0','\xDD','\xA9','\x40','\x85','\xAA','\x20','\x8E','\xFB','\xA9','\x00','\x85','\xAB','\xF0','\xD0','\xA9','\x80','\x85','\xAA','\xD0','\xCA',
'\xA5','\xB5','\xF0','\x0A','\xA9','\x04','\x20','\x1C','\xFE','\xA9','\x00','\x4C','\x4A','\xFB','\x20','\xD1','\xFC','\x90','\x03','\x4C','\x48','\xFB','\xA6','\xA7','\xCA','\xF0','\x2D','\xA5','\x93','\xF0','\x0C','\xA0',
'\x00','\xA5','\xBD','\xD1','\xAC','\xF0','\x04','\xA9','\x01','\x85','\xB6','\xA5','\xB6','\xF0','\x4B','\xA2','\x3D','\xE4','\x9E','\x90','\x3E','\xA6','\x9E','\xA5','\xAD','\x9D','\x01','\x01','\xA5','\xAC','\x9D','\x00',
'\x01','\xE8','\xE8','\x86','\x9E','\x4C','\x3A','\xFB','\xA6','\x9F','\xE4','\x9E','\xF0','\x35','\xA5','\xAC','\xDD','\x00','\x01','\xD0','\x2E','\xA5','\xAD','\xDD','\x01','\x01','\xD0','\x27','\xE6','\x9F','\xE6','\x9F',
'\xA5','\x93','\xF0','\x0B','\xA5','\xBD','\xA0','\x00','\xD1','\xAC','\xF0','\x17','\xC8','\x84','\xB6','\xA5','\xB6','\xF0','\x07','\xA9','\x10','\x20','\x1C','\xFE','\xD0','\x09','\xA5','\x93','\xD0','\x05','\xA8','\xA5',
'\xBD','\x91','\xAC','\x20','\xDB','\xFC','\xD0','\x43','\xA9','\x80','\x85','\xAA','\x78','\xA2','\x01','\x8E','\x0D','\xDC','\xAE','\x0D','\xDC','\xA6','\xBE','\xCA','\x30','\x02','\x86','\xBE','\xC6','\xA7','\xF0','\x08',
'\xA5','\x9E','\xD0','\x27','\x85','\xBE','\xF0','\x23','\x20','\x93','\xFC','\x20','\x8E','\xFB','\xA0','\x00','\x84','\xAB','\xB1','\xAC','\x45','\xAB','\x85','\xAB','\x20','\xDB','\xFC','\x20','\xD1','\xFC','\x90','\xF2',
'\xA5','\xAB','\x45','\xBD','\xF0','\x05','\xA9','\x20','\x20','\x1C','\xFE','\x4C','\xBC','\xFE','\xA5','\xC2','\x85','\xAD','\xA5','\xC1','\x85','\xAC','\x60','\xA9','\x08','\x85','\xA3','\xA9','\x00','\x85','\xA4','\x85',
'\xA8','\x85','\x9B','\x85','\xA9','\x60','\xA5','\xBD','\x4A','\xA9','\x60','\x90','\x02','\xA9','\xB0','\xA2','\x00','\x8D','\x06','\xDC','\x8E','\x07','\xDC','\xAD','\x0D','\xDC','\xA9','\x19','\x8D','\x0F','\xDC','\xA5',
'\x01','\x49','\x08','\x85','\x01','\x29','\x08','\x60','\x38','\x66','\xB6','\x30','\x3C','\xA5','\xA8','\xD0','\x12','\xA9','\x10','\xA2','\x01','\x20','\xB1','\xFB','\xD0','\x2F','\xE6','\xA8','\xA5','\xB6','\x10','\x29',
'\x4C','\x57','\xFC','\xA5','\xA9','\xD0','\x09','\x20','\xAD','\xFB','\xD0','\x1D','\xE6','\xA9','\xD0','\x19','\x20','\xA6','\xFB','\xD0','\x14','\xA5','\xA4','\x49','\x01','\x85','\xA4','\xF0','\x0F','\xA5','\xBD','\x49',
'\x01','\x85','\xBD','\x29','\x01','\x45','\x9B','\x85','\x9B','\x4C','\xBC','\xFE','\x46','\xBD','\xC6','\xA3','\xA5','\xA3','\xF0','\x3A','\x10','\xF3','\x20','\x97','\xFB','\x58','\xA5','\xA5','\xF0','\x12','\xA2','\x00',
'\x86','\xD7','\xC6','\xA5','\xA6','\xBE','\xE0','\x02','\xD0','\x02','\x09','\x80','\x85','\xBD','\xD0','\xD9','\x20','\xD1','\xFC','\x90','\x0A','\xD0','\x91','\xE6','\xAD','\xA5','\xD7','\x85','\xBD','\xB0','\xCA','\xA0',
'\x00','\xB1','\xAC','\x85','\xBD','\x45','\xD7','\x85','\xD7','\x20','\xDB','\xFC','\xD0','\xBB','\xA5','\x9B','\x49','\x01','\x85','\xBD','\x4C','\xBC','\xFE','\xC6','\xBE','\xD0','\x03','\x20','\xCA','\xFC','\xA9','\x50',
'\x85','\xA7','\xA2','\x08','\x78','\x20','\xBD','\xFC','\xD0','\xEA','\xA9','\x78','\x20','\xAF','\xFB','\xD0','\xE3','\xC6','\xA7','\xD0','\xDF','\x20','\x97','\xFB','\xC6','\xAB','\x10','\xD8','\xA2','\x0A','\x20','\xBD',
'\xFC','\x58','\xE6','\xAB','\xA5','\xBE','\xF0','\x30','\x20','\x8E','\xFB','\xA2','\x09','\x86','\xA5','\x86','\xB6','\xD0','\x83','\x08','\x78','\xAD','\x11','\xD0','\x09','\x10','\x8D','\x11','\xD0','\x20','\xCA','\xFC',
'\xA9','\x7F','\x8D','\x0D','\xDC','\x20','\xDD','\xFD','\xAD','\xA0','\x02','\xF0','\x09','\x8D','\x15','\x03','\xAD','\x9F','\x02','\x8D','\x14','\x03','\x28','\x60','\x20','\x93','\xFC','\xF0','\x97','\xBD','\x93','\xFD',
'\x8D','\x14','\x03','\xBD','\x94','\xFD','\x8D','\x15','\x03','\x60','\xA5','\x01','\x09','\x20','\x85','\x01','\x60','\x38','\xA5','\xAC','\xE5','\xAE','\xA5','\xAD','\xE5','\xAF','\x60','\xE6','\xAC','\xD0','\x02','\xE6',
'\xAD','\x60','\xA2','\xFF','\x78','\x9A','\xD8','\x20','\x02','\xFD','\xD0','\x03','\x6C','\x00','\x80','\x8E','\x16','\xD0','\x20','\xA3','\xFD','\x20','\x50','\xFD','\x20','\x15','\xFD','\x20','\x5B','\xFF','\x58','\x6C',
'\x00','\xA0','\xA2','\x05','\xBD','\x0F','\xFD','\xDD','\x03','\x80','\xD0','\x03','\xCA','\xD0','\xF5','\x60','\xC3','\xC2','\xCD','\x38','\x30','\xA2','\x30','\xA0','\xFD','\x18','\x86','\xC3','\x84','\xC4','\xA0','\x1F',
'\xB9','\x14','\x03','\xB0','\x02','\xB1','\xC3','\x91','\xC3','\x99','\x14','\x03','\x88','\x10','\xF1','\x60','\x31','\xEA','\x66','\xFE','\x47','\xFE','\x4A','\xF3','\x91','\xF2','\x0E','\xF2','\x50','\xF2','\x33','\xF3',
'\x57','\xF1','\xCA','\xF1','\xED','\xF6','\x3E','\xF1','\x2F','\xF3','\x66','\xFE','\xA5','\xF4','\xED','\xF5','\xA9','\x00','\xA8','\x99','\x02','\x00','\x99','\x00','\x02','\x99','\x00','\x03','\xC8','\xD0','\xF4','\xA2',
'\x3C','\xA0','\x03','\x86','\xB2','\x84','\xB3','\xA8','\xA9','\x03','\x85','\xC2','\xE6','\xC2','\xB1','\xC1','\xAA','\xA9','\x55','\x91','\xC1','\xD1','\xC1','\xD0','\x0F','\x2A','\x91','\xC1','\xD1','\xC1','\xD0','\x08',
'\x8A','\x91','\xC1','\xC8','\xD0','\xE8','\xF0','\xE4','\x98','\xAA','\xA4','\xC2','\x18','\x20','\x2D','\xFE','\xA9','\x08','\x8D','\x82','\x02','\xA9','\x04','\x8D','\x88','\x02','\x60','\x6A','\xFC','\xCD','\xFB','\x31',
'\xEA','\x2C','\xF9','\xA9','\x7F','\x8D','\x0D','\xDC','\x8D','\x0D','\xDD','\x8D','\x00','\xDC','\xA9','\x08','\x8D','\x0E','\xDC','\x8D','\x0E','\xDD','\x8D','\x0F','\xDC','\x8D','\x0F','\xDD','\xA2','\x00','\x8E','\x03',
'\xDC','\x8E','\x03','\xDD','\x8E','\x18','\xD4','\xCA','\x8E','\x02','\xDC','\xA9','\x07','\x8D','\x00','\xDD','\xA9','\x3F','\x8D','\x02','\xDD','\xA9','\xE7','\x85','\x01','\xA9','\x2F','\x85','\x00','\xAD','\xA6','\x02',
'\xF0','\x0A','\xA9','\x25','\x8D','\x04','\xDC','\xA9','\x40','\x4C','\xF3','\xFD','\xA9','\x95','\x8D','\x04','\xDC','\xA9','\x42','\x8D','\x05','\xDC','\x4C','\x6E','\xFF','\x85','\xB7','\x86','\xBB','\x84','\xBC','\x60',
'\x85','\xB8','\x86','\xBA','\x84','\xB9','\x60','\xA5','\xBA','\xC9','\x02','\xD0','\x0D','\xAD','\x97','\x02','\x48','\xA9','\x00','\x8D','\x97','\x02','\x68','\x60','\x85','\x9D','\xA5','\x90','\x05','\x90','\x85','\x90',
'\x60','\x8D','\x85','\x02','\x60','\x90','\x06','\xAE','\x83','\x02','\xAC','\x84','\x02','\x8E','\x83','\x02','\x8C','\x84','\x02','\x60','\x90','\x06','\xAE','\x81','\x02','\xAC','\x82','\x02','\x8E','\x81','\x02','\x8C',
'\x82','\x02','\x60','\x78','\x6C','\x18','\x03','\x48','\x8A','\x48','\x98','\x48','\xA9','\x7F','\x8D','\x0D','\xDD','\xAC','\x0D','\xDD','\x30','\x1C','\x20','\x02','\xFD','\xD0','\x03','\x6C','\x02','\x80','\x20','\xBC',
'\xF6','\x20','\xE1','\xFF','\xD0','\x0C','\x20','\x15','\xFD','\x20','\xA3','\xFD','\x20','\x18','\xE5','\x6C','\x02','\xA0','\x98','\x2D','\xA1','\x02','\xAA','\x29','\x01','\xF0','\x28','\xAD','\x00','\xDD','\x29','\xFB',
'\x05','\xB5','\x8D','\x00','\xDD','\xAD','\xA1','\x02','\x8D','\x0D','\xDD','\x8A','\x29','\x12','\xF0','\x0D','\x29','\x02','\xF0','\x06','\x20','\xD6','\xFE','\x4C','\x9D','\xFE','\x20','\x07','\xFF','\x20','\xBB','\xEE',
'\x4C','\xB6','\xFE','\x8A','\x29','\x02','\xF0','\x06','\x20','\xD6','\xFE','\x4C','\xB6','\xFE','\x8A','\x29','\x10','\xF0','\x03','\x20','\x07','\xFF','\xAD','\xA1','\x02','\x8D','\x0D','\xDD','\x68','\xA8','\x68','\xAA',
'\x68','\x40','\xC1','\x27','\x3E','\x1A','\xC5','\x11','\x74','\x0E','\xED','\x0C','\x45','\x06','\xF0','\x02','\x46','\x01','\xB8','\x00','\x71','\x00','\xAD','\x01','\xDD','\x29','\x01','\x85','\xA7','\xAD','\x06','\xDD',
'\xE9','\x1C','\x6D','\x99','\x02','\x8D','\x06','\xDD','\xAD','\x07','\xDD','\x6D','\x9A','\x02','\x8D','\x07','\xDD','\xA9','\x11','\x8D','\x0F','\xDD','\xAD','\xA1','\x02','\x8D','\x0D','\xDD','\xA9','\xFF','\x8D','\x06',
'\xDD','\x8D','\x07','\xDD','\x4C','\x59','\xEF','\xAD','\x95','\x02','\x8D','\x06','\xDD','\xAD','\x96','\x02','\x8D','\x07','\xDD','\xA9','\x11','\x8D','\x0F','\xDD','\xA9','\x12','\x4D','\xA1','\x02','\x8D','\xA1','\x02',
'\xA9','\xFF','\x8D','\x06','\xDD','\x8D','\x07','\xDD','\xAE','\x98','\x02','\x86','\xA8','\x60','\xAA','\xAD','\x96','\x02','\x2A','\xA8','\x8A','\x69','\xC8','\x8D','\x99','\x02','\x98','\x69','\x00','\x8D','\x9A','\x02',
'\x60','\xEA','\xEA','\x08','\x68','\x29','\xEF','\x48','\x48','\x8A','\x48','\x98','\x48','\xBA','\xBD','\x04','\x01','\x29','\x10','\xF0','\x03','\x6C','\x16','\x03','\x6C','\x14','\x03','\x20','\x18','\xE5','\xAD','\x12',
'\xD0','\xD0','\xFB','\xAD','\x19','\xD0','\x29','\x01','\x8D','\xA6','\x02','\x4C','\xDD','\xFD','\xA9','\x81','\x8D','\x0D','\xDC','\xAD','\x0E','\xDC','\x29','\x80','\x09','\x11','\x8D','\x0E','\xDC','\x4C','\x8E','\xEE',
'\x03','\x4C','\x5B','\xFF','\x4C','\xA3','\xFD','\x4C','\x50','\xFD','\x4C','\x15','\xFD','\x4C','\x1A','\xFD','\x4C','\x18','\xFE','\x4C','\xB9','\xED','\x4C','\xC7','\xED','\x4C','\x25','\xFE','\x4C','\x34','\xFE','\x4C',
'\x87','\xEA','\x4C','\x21','\xFE','\x4C','\x13','\xEE','\x4C','\xDD','\xED','\x4C','\xEF','\xED','\x4C','\xFE','\xED','\x4C','\x0C','\xED','\x4C','\x09','\xED','\x4C','\x07','\xFE','\x4C','\x00','\xFE','\x4C','\xF9','\xFD',
'\x6C','\x1A','\x03','\x6C','\x1C','\x03','\x6C','\x1E','\x03','\x6C','\x20','\x03','\x6C','\x22','\x03','\x6C','\x24','\x03','\x6C','\x26','\x03','\x4C','\x9E','\xF4','\x4C','\xDD','\xF5','\x4C','\xE4','\xF6','\x4C','\xDD',
'\xF6','\x6C','\x28','\x03','\x6C','\x2A','\x03','\x6C','\x2C','\x03','\x4C','\x9B','\xF6','\x4C','\x05','\xE5','\x4C','\x0A','\xE5','\x4C','\x00','\xE5','\x52','\x52','\x42','\x59','\x43','\xFE','\xE2','\xFC','\x48','\xFF'
};

const byte chargen_rom[] = {
'\x3C','\x66','\x6E','\x6E','\x60','\x62','\x3C','\x00','\x18','\x3C','\x66','\x7E','\x66','\x66','\x66','\x00','\x7C','\x66','\x66','\x7C','\x66','\x66','\x7C','\x00','\x3C','\x66','\x60','\x60','\x60','\x66','\x3C','\x00',
'\x78','\x6C','\x66','\x66','\x66','\x6C','\x78','\x00','\x7E','\x60','\x60','\x78','\x60','\x60','\x7E','\x00','\x7E','\x60','\x60','\x78','\x60','\x60','\x60','\x00','\x3C','\x66','\x60','\x6E','\x66','\x66','\x3C','\x00',
'\x66','\x66','\x66','\x7E','\x66','\x66','\x66','\x00','\x3C','\x18','\x18','\x18','\x18','\x18','\x3C','\x00','\x1E','\x0C','\x0C','\x0C','\x0C','\x6C','\x38','\x00','\x66','\x6C','\x78','\x70','\x78','\x6C','\x66','\x00',
'\x60','\x60','\x60','\x60','\x60','\x60','\x7E','\x00','\x63','\x77','\x7F','\x6B','\x63','\x63','\x63','\x00','\x66','\x76','\x7E','\x7E','\x6E','\x66','\x66','\x00','\x3C','\x66','\x66','\x66','\x66','\x66','\x3C','\x00',
'\x7C','\x66','\x66','\x7C','\x60','\x60','\x60','\x00','\x3C','\x66','\x66','\x66','\x66','\x3C','\x0E','\x00','\x7C','\x66','\x66','\x7C','\x78','\x6C','\x66','\x00','\x3C','\x66','\x60','\x3C','\x06','\x66','\x3C','\x00',
'\x7E','\x18','\x18','\x18','\x18','\x18','\x18','\x00','\x66','\x66','\x66','\x66','\x66','\x66','\x3C','\x00','\x66','\x66','\x66','\x66','\x66','\x3C','\x18','\x00','\x63','\x63','\x63','\x6B','\x7F','\x77','\x63','\x00',
'\x66','\x66','\x3C','\x18','\x3C','\x66','\x66','\x00','\x66','\x66','\x66','\x3C','\x18','\x18','\x18','\x00','\x7E','\x06','\x0C','\x18','\x30','\x60','\x7E','\x00','\x3C','\x30','\x30','\x30','\x30','\x30','\x3C','\x00',
'\x0C','\x12','\x30','\x7C','\x30','\x62','\xFC','\x00','\x3C','\x0C','\x0C','\x0C','\x0C','\x0C','\x3C','\x00','\x00','\x18','\x3C','\x7E','\x18','\x18','\x18','\x18','\x00','\x10','\x30','\x7F','\x7F','\x30','\x10','\x00',
'\x00','\x00','\x00','\x00','\x00','\x00','\x00','\x00','\x18','\x18','\x18','\x18','\x00','\x00','\x18','\x00','\x66','\x66','\x66','\x00','\x00','\x00','\x00','\x00','\x66','\x66','\xFF','\x66','\xFF','\x66','\x66','\x00',
'\x18','\x3E','\x60','\x3C','\x06','\x7C','\x18','\x00','\x62','\x66','\x0C','\x18','\x30','\x66','\x46','\x00','\x3C','\x66','\x3C','\x38','\x67','\x66','\x3F','\x00','\x06','\x0C','\x18','\x00','\x00','\x00','\x00','\x00',
'\x0C','\x18','\x30','\x30','\x30','\x18','\x0C','\x00','\x30','\x18','\x0C','\x0C','\x0C','\x18','\x30','\x00','\x00','\x66','\x3C','\xFF','\x3C','\x66','\x00','\x00','\x00','\x18','\x18','\x7E','\x18','\x18','\x00','\x00',
'\x00','\x00','\x00','\x00','\x00','\x18','\x18','\x30','\x00','\x00','\x00','\x7E','\x00','\x00','\x00','\x00','\x00','\x00','\x00','\x00','\x00','\x18','\x18','\x00','\x00','\x03','\x06','\x0C','\x18','\x30','\x60','\x00',
'\x3C','\x66','\x6E','\x76','\x66','\x66','\x3C','\x00','\x18','\x18','\x38','\x18','\x18','\x18','\x7E','\x00','\x3C','\x66','\x06','\x0C','\x30','\x60','\x7E','\x00','\x3C','\x66','\x06','\x1C','\x06','\x66','\x3C','\x00',
'\x06','\x0E','\x1E','\x66','\x7F','\x06','\x06','\x00','\x7E','\x60','\x7C','\x06','\x06','\x66','\x3C','\x00','\x3C','\x66','\x60','\x7C','\x66','\x66','\x3C','\x00','\x7E','\x66','\x0C','\x18','\x18','\x18','\x18','\x00',
'\x3C','\x66','\x66','\x3C','\x66','\x66','\x3C','\x00','\x3C','\x66','\x66','\x3E','\x06','\x66','\x3C','\x00','\x00','\x00','\x18','\x00','\x00','\x18','\x00','\x00','\x00','\x00','\x18','\x00','\x00','\x18','\x18','\x30',
'\x0E','\x18','\x30','\x60','\x30','\x18','\x0E','\x00','\x00','\x00','\x7E','\x00','\x7E','\x00','\x00','\x00','\x70','\x18','\x0C','\x06','\x0C','\x18','\x70','\x00','\x3C','\x66','\x06','\x0C','\x18','\x00','\x18','\x00',
'\x00','\x00','\x00','\xFF','\xFF','\x00','\x00','\x00','\x08','\x1C','\x3E','\x7F','\x7F','\x1C','\x3E','\x00','\x18','\x18','\x18','\x18','\x18','\x18','\x18','\x18','\x00','\x00','\x00','\xFF','\xFF','\x00','\x00','\x00',
'\x00','\x00','\xFF','\xFF','\x00','\x00','\x00','\x00','\x00','\xFF','\xFF','\x00','\x00','\x00','\x00','\x00','\x00','\x00','\x00','\x00','\xFF','\xFF','\x00','\x00','\x30','\x30','\x30','\x30','\x30','\x30','\x30','\x30',
'\x0C','\x0C','\x0C','\x0C','\x0C','\x0C','\x0C','\x0C','\x00','\x00','\x00','\xE0','\xF0','\x38','\x18','\x18','\x18','\x18','\x1C','\x0F','\x07','\x00','\x00','\x00','\x18','\x18','\x38','\xF0','\xE0','\x00','\x00','\x00',
'\xC0','\xC0','\xC0','\xC0','\xC0','\xC0','\xFF','\xFF','\xC0','\xE0','\x70','\x38','\x1C','\x0E','\x07','\x03','\x03','\x07','\x0E','\x1C','\x38','\x70','\xE0','\xC0','\xFF','\xFF','\xC0','\xC0','\xC0','\xC0','\xC0','\xC0',
'\xFF','\xFF','\x03','\x03','\x03','\x03','\x03','\x03','\x00','\x3C','\x7E','\x7E','\x7E','\x7E','\x3C','\x00','\x00','\x00','\x00','\x00','\x00','\xFF','\xFF','\x00','\x36','\x7F','\x7F','\x7F','\x3E','\x1C','\x08','\x00',
'\x60','\x60','\x60','\x60','\x60','\x60','\x60','\x60','\x00','\x00','\x00','\x07','\x0F','\x1C','\x18','\x18','\xC3','\xE7','\x7E','\x3C','\x3C','\x7E','\xE7','\xC3','\x00','\x3C','\x7E','\x66','\x66','\x7E','\x3C','\x00',
'\x18','\x18','\x66','\x66','\x18','\x18','\x3C','\x00','\x06','\x06','\x06','\x06','\x06','\x06','\x06','\x06','\x08','\x1C','\x3E','\x7F','\x3E','\x1C','\x08','\x00','\x18','\x18','\x18','\xFF','\xFF','\x18','\x18','\x18',
'\xC0','\xC0','\x30','\x30','\xC0','\xC0','\x30','\x30','\x18','\x18','\x18','\x18','\x18','\x18','\x18','\x18','\x00','\x00','\x03','\x3E','\x76','\x36','\x36','\x00','\xFF','\x7F','\x3F','\x1F','\x0F','\x07','\x03','\x01',
'\x00','\x00','\x00','\x00','\x00','\x00','\x00','\x00','\xF0','\xF0','\xF0','\xF0','\xF0','\xF0','\xF0','\xF0','\x00','\x00','\x00','\x00','\xFF','\xFF','\xFF','\xFF','\xFF','\x00','\x00','\x00','\x00','\x00','\x00','\x00',
'\x00','\x00','\x00','\x00','\x00','\x00','\x00','\xFF','\xC0','\xC0','\xC0','\xC0','\xC0','\xC0','\xC0','\xC0','\xCC','\xCC','\x33','\x33','\xCC','\xCC','\x33','\x33','\x03','\x03','\x03','\x03','\x03','\x03','\x03','\x03',
'\x00','\x00','\x00','\x00','\xCC','\xCC','\x33','\x33','\xFF','\xFE','\xFC','\xF8','\xF0','\xE0','\xC0','\x80','\x03','\x03','\x03','\x03','\x03','\x03','\x03','\x03','\x18','\x18','\x18','\x1F','\x1F','\x18','\x18','\x18',
'\x00','\x00','\x00','\x00','\x0F','\x0F','\x0F','\x0F','\x18','\x18','\x18','\x1F','\x1F','\x00','\x00','\x00','\x00','\x00','\x00','\xF8','\xF8','\x18','\x18','\x18','\x00','\x00','\x00','\x00','\x00','\x00','\xFF','\xFF',
'\x00','\x00','\x00','\x1F','\x1F','\x18','\x18','\x18','\x18','\x18','\x18','\xFF','\xFF','\x00','\x00','\x00','\x00','\x00','\x00','\xFF','\xFF','\x18','\x18','\x18','\x18','\x18','\x18','\xF8','\xF8','\x18','\x18','\x18',
'\xC0','\xC0','\xC0','\xC0','\xC0','\xC0','\xC0','\xC0','\xE0','\xE0','\xE0','\xE0','\xE0','\xE0','\xE0','\xE0','\x07','\x07','\x07','\x07','\x07','\x07','\x07','\x07','\xFF','\xFF','\x00','\x00','\x00','\x00','\x00','\x00',
'\xFF','\xFF','\xFF','\x00','\x00','\x00','\x00','\x00','\x00','\x00','\x00','\x00','\x00','\xFF','\xFF','\xFF','\x03','\x03','\x03','\x03','\x03','\x03','\xFF','\xFF','\x00','\x00','\x00','\x00','\xF0','\xF0','\xF0','\xF0',
'\x0F','\x0F','\x0F','\x0F','\x00','\x00','\x00','\x00','\x18','\x18','\x18','\xF8','\xF8','\x00','\x00','\x00','\xF0','\xF0','\xF0','\xF0','\x00','\x00','\x00','\x00','\xF0','\xF0','\xF0','\xF0','\x0F','\x0F','\x0F','\x0F',
'\xC3','\x99','\x91','\x91','\x9F','\x99','\xC3','\xFF','\xE7','\xC3','\x99','\x81','\x99','\x99','\x99','\xFF','\x83','\x99','\x99','\x83','\x99','\x99','\x83','\xFF','\xC3','\x99','\x9F','\x9F','\x9F','\x99','\xC3','\xFF',
'\x87','\x93','\x99','\x99','\x99','\x93','\x87','\xFF','\x81','\x9F','\x9F','\x87','\x9F','\x9F','\x81','\xFF','\x81','\x9F','\x9F','\x87','\x9F','\x9F','\x9F','\xFF','\xC3','\x99','\x9F','\x91','\x99','\x99','\xC3','\xFF',
'\x99','\x99','\x99','\x81','\x99','\x99','\x99','\xFF','\xC3','\xE7','\xE7','\xE7','\xE7','\xE7','\xC3','\xFF','\xE1','\xF3','\xF3','\xF3','\xF3','\x93','\xC7','\xFF','\x99','\x93','\x87','\x8F','\x87','\x93','\x99','\xFF',
'\x9F','\x9F','\x9F','\x9F','\x9F','\x9F','\x81','\xFF','\x9C','\x88','\x80','\x94','\x9C','\x9C','\x9C','\xFF','\x99','\x89','\x81','\x81','\x91','\x99','\x99','\xFF','\xC3','\x99','\x99','\x99','\x99','\x99','\xC3','\xFF',
'\x83','\x99','\x99','\x83','\x9F','\x9F','\x9F','\xFF','\xC3','\x99','\x99','\x99','\x99','\xC3','\xF1','\xFF','\x83','\x99','\x99','\x83','\x87','\x93','\x99','\xFF','\xC3','\x99','\x9F','\xC3','\xF9','\x99','\xC3','\xFF',
'\x81','\xE7','\xE7','\xE7','\xE7','\xE7','\xE7','\xFF','\x99','\x99','\x99','\x99','\x99','\x99','\xC3','\xFF','\x99','\x99','\x99','\x99','\x99','\xC3','\xE7','\xFF','\x9C','\x9C','\x9C','\x94','\x80','\x88','\x9C','\xFF',
'\x99','\x99','\xC3','\xE7','\xC3','\x99','\x99','\xFF','\x99','\x99','\x99','\xC3','\xE7','\xE7','\xE7','\xFF','\x81','\xF9','\xF3','\xE7','\xCF','\x9F','\x81','\xFF','\xC3','\xCF','\xCF','\xCF','\xCF','\xCF','\xC3','\xFF',
'\xF3','\xED','\xCF','\x83','\xCF','\x9D','\x03','\xFF','\xC3','\xF3','\xF3','\xF3','\xF3','\xF3','\xC3','\xFF','\xFF','\xE7','\xC3','\x81','\xE7','\xE7','\xE7','\xE7','\xFF','\xEF','\xCF','\x80','\x80','\xCF','\xEF','\xFF',
'\xFF','\xFF','\xFF','\xFF','\xFF','\xFF','\xFF','\xFF','\xE7','\xE7','\xE7','\xE7','\xFF','\xFF','\xE7','\xFF','\x99','\x99','\x99','\xFF','\xFF','\xFF','\xFF','\xFF','\x99','\x99','\x00','\x99','\x00','\x99','\x99','\xFF',
'\xE7','\xC1','\x9F','\xC3','\xF9','\x83','\xE7','\xFF','\x9D','\x99','\xF3','\xE7','\xCF','\x99','\xB9','\xFF','\xC3','\x99','\xC3','\xC7','\x98','\x99','\xC0','\xFF','\xF9','\xF3','\xE7','\xFF','\xFF','\xFF','\xFF','\xFF',
'\xF3','\xE7','\xCF','\xCF','\xCF','\xE7','\xF3','\xFF','\xCF','\xE7','\xF3','\xF3','\xF3','\xE7','\xCF','\xFF','\xFF','\x99','\xC3','\x00','\xC3','\x99','\xFF','\xFF','\xFF','\xE7','\xE7','\x81','\xE7','\xE7','\xFF','\xFF',
'\xFF','\xFF','\xFF','\xFF','\xFF','\xE7','\xE7','\xCF','\xFF','\xFF','\xFF','\x81','\xFF','\xFF','\xFF','\xFF','\xFF','\xFF','\xFF','\xFF','\xFF','\xE7','\xE7','\xFF','\xFF','\xFC','\xF9','\xF3','\xE7','\xCF','\x9F','\xFF',
'\xC3','\x99','\x91','\x89','\x99','\x99','\xC3','\xFF','\xE7','\xE7','\xC7','\xE7','\xE7','\xE7','\x81','\xFF','\xC3','\x99','\xF9','\xF3','\xCF','\x9F','\x81','\xFF','\xC3','\x99','\xF9','\xE3','\xF9','\x99','\xC3','\xFF',
'\xF9','\xF1','\xE1','\x99','\x80','\xF9','\xF9','\xFF','\x81','\x9F','\x83','\xF9','\xF9','\x99','\xC3','\xFF','\xC3','\x99','\x9F','\x83','\x99','\x99','\xC3','\xFF','\x81','\x99','\xF3','\xE7','\xE7','\xE7','\xE7','\xFF',
'\xC3','\x99','\x99','\xC3','\x99','\x99','\xC3','\xFF','\xC3','\x99','\x99','\xC1','\xF9','\x99','\xC3','\xFF','\xFF','\xFF','\xE7','\xFF','\xFF','\xE7','\xFF','\xFF','\xFF','\xFF','\xE7','\xFF','\xFF','\xE7','\xE7','\xCF',
'\xF1','\xE7','\xCF','\x9F','\xCF','\xE7','\xF1','\xFF','\xFF','\xFF','\x81','\xFF','\x81','\xFF','\xFF','\xFF','\x8F','\xE7','\xF3','\xF9','\xF3','\xE7','\x8F','\xFF','\xC3','\x99','\xF9','\xF3','\xE7','\xFF','\xE7','\xFF',
'\xFF','\xFF','\xFF','\x00','\x00','\xFF','\xFF','\xFF','\xF7','\xE3','\xC1','\x80','\x80','\xE3','\xC1','\xFF','\xE7','\xE7','\xE7','\xE7','\xE7','\xE7','\xE7','\xE7','\xFF','\xFF','\xFF','\x00','\x00','\xFF','\xFF','\xFF',
'\xFF','\xFF','\x00','\x00','\xFF','\xFF','\xFF','\xFF','\xFF','\x00','\x00','\xFF','\xFF','\xFF','\xFF','\xFF','\xFF','\xFF','\xFF','\xFF','\x00','\x00','\xFF','\xFF','\xCF','\xCF','\xCF','\xCF','\xCF','\xCF','\xCF','\xCF',
'\xF3','\xF3','\xF3','\xF3','\xF3','\xF3','\xF3','\xF3','\xFF','\xFF','\xFF','\x1F','\x0F','\xC7','\xE7','\xE7','\xE7','\xE7','\xE3','\xF0','\xF8','\xFF','\xFF','\xFF','\xE7','\xE7','\xC7','\x0F','\x1F','\xFF','\xFF','\xFF',
'\x3F','\x3F','\x3F','\x3F','\x3F','\x3F','\x00','\x00','\x3F','\x1F','\x8F','\xC7','\xE3','\xF1','\xF8','\xFC','\xFC','\xF8','\xF1','\xE3','\xC7','\x8F','\x1F','\x3F','\x00','\x00','\x3F','\x3F','\x3F','\x3F','\x3F','\x3F',
'\x00','\x00','\xFC','\xFC','\xFC','\xFC','\xFC','\xFC','\xFF','\xC3','\x81','\x81','\x81','\x81','\xC3','\xFF','\xFF','\xFF','\xFF','\xFF','\xFF','\x00','\x00','\xFF','\xC9','\x80','\x80','\x80','\xC1','\xE3','\xF7','\xFF',
'\x9F','\x9F','\x9F','\x9F','\x9F','\x9F','\x9F','\x9F','\xFF','\xFF','\xFF','\xF8','\xF0','\xE3','\xE7','\xE7','\x3C','\x18','\x81','\xC3','\xC3','\x81','\x18','\x3C','\xFF','\xC3','\x81','\x99','\x99','\x81','\xC3','\xFF',
'\xE7','\xE7','\x99','\x99','\xE7','\xE7','\xC3','\xFF','\xF9','\xF9','\xF9','\xF9','\xF9','\xF9','\xF9','\xF9','\xF7','\xE3','\xC1','\x80','\xC1','\xE3','\xF7','\xFF','\xE7','\xE7','\xE7','\x00','\x00','\xE7','\xE7','\xE7',
'\x3F','\x3F','\xCF','\xCF','\x3F','\x3F','\xCF','\xCF','\xE7','\xE7','\xE7','\xE7','\xE7','\xE7','\xE7','\xE7','\xFF','\xFF','\xFC','\xC1','\x89','\xC9','\xC9','\xFF','\x00','\x80','\xC0','\xE0','\xF0','\xF8','\xFC','\xFE',
'\xFF','\xFF','\xFF','\xFF','\xFF','\xFF','\xFF','\xFF','\x0F','\x0F','\x0F','\x0F','\x0F','\x0F','\x0F','\x0F','\xFF','\xFF','\xFF','\xFF','\x00','\x00','\x00','\x00','\x00','\xFF','\xFF','\xFF','\xFF','\xFF','\xFF','\xFF',
'\xFF','\xFF','\xFF','\xFF','\xFF','\xFF','\xFF','\x00','\x3F','\x3F','\x3F','\x3F','\x3F','\x3F','\x3F','\x3F','\x33','\x33','\xCC','\xCC','\x33','\x33','\xCC','\xCC','\xFC','\xFC','\xFC','\xFC','\xFC','\xFC','\xFC','\xFC',
'\xFF','\xFF','\xFF','\xFF','\x33','\x33','\xCC','\xCC','\x00','\x01','\x03','\x07','\x0F','\x1F','\x3F','\x7F','\xFC','\xFC','\xFC','\xFC','\xFC','\xFC','\xFC','\xFC','\xE7','\xE7','\xE7','\xE0','\xE0','\xE7','\xE7','\xE7',
'\xFF','\xFF','\xFF','\xFF','\xF0','\xF0','\xF0','\xF0','\xE7','\xE7','\xE7','\xE0','\xE0','\xFF','\xFF','\xFF','\xFF','\xFF','\xFF','\x07','\x07','\xE7','\xE7','\xE7','\xFF','\xFF','\xFF','\xFF','\xFF','\xFF','\x00','\x00',
'\xFF','\xFF','\xFF','\xE0','\xE0','\xE7','\xE7','\xE7','\xE7','\xE7','\xE7','\x00','\x00','\xFF','\xFF','\xFF','\xFF','\xFF','\xFF','\x00','\x00','\xE7','\xE7','\xE7','\xE7','\xE7','\xE7','\x07','\x07','\xE7','\xE7','\xE7',
'\x3F','\x3F','\x3F','\x3F','\x3F','\x3F','\x3F','\x3F','\x1F','\x1F','\x1F','\x1F','\x1F','\x1F','\x1F','\x1F','\xF8','\xF8','\xF8','\xF8','\xF8','\xF8','\xF8','\xF8','\x00','\x00','\xFF','\xFF','\xFF','\xFF','\xFF','\xFF',
'\x00','\x00','\x00','\xFF','\xFF','\xFF','\xFF','\xFF','\xFF','\xFF','\xFF','\xFF','\xFF','\x00','\x00','\x00','\xFC','\xFC','\xFC','\xFC','\xFC','\xFC','\x00','\x00','\xFF','\xFF','\xFF','\xFF','\x0F','\x0F','\x0F','\x0F',
'\xF0','\xF0','\xF0','\xF0','\xFF','\xFF','\xFF','\xFF','\xE7','\xE7','\xE7','\x07','\x07','\xFF','\xFF','\xFF','\x0F','\x0F','\x0F','\x0F','\xFF','\xFF','\xFF','\xFF','\x0F','\x0F','\x0F','\x0F','\xF0','\xF0','\xF0','\xF0',
'\x3C','\x66','\x6E','\x6E','\x60','\x62','\x3C','\x00','\x00','\x00','\x3C','\x06','\x3E','\x66','\x3E','\x00','\x00','\x60','\x60','\x7C','\x66','\x66','\x7C','\x00','\x00','\x00','\x3C','\x60','\x60','\x60','\x3C','\x00',
'\x00','\x06','\x06','\x3E','\x66','\x66','\x3E','\x00','\x00','\x00','\x3C','\x66','\x7E','\x60','\x3C','\x00','\x00','\x0E','\x18','\x3E','\x18','\x18','\x18','\x00','\x00','\x00','\x3E','\x66','\x66','\x3E','\x06','\x7C',
'\x00','\x60','\x60','\x7C','\x66','\x66','\x66','\x00','\x00','\x18','\x00','\x38','\x18','\x18','\x3C','\x00','\x00','\x06','\x00','\x06','\x06','\x06','\x06','\x3C','\x00','\x60','\x60','\x6C','\x78','\x6C','\x66','\x00',
'\x00','\x38','\x18','\x18','\x18','\x18','\x3C','\x00','\x00','\x00','\x66','\x7F','\x7F','\x6B','\x63','\x00','\x00','\x00','\x7C','\x66','\x66','\x66','\x66','\x00','\x00','\x00','\x3C','\x66','\x66','\x66','\x3C','\x00',
'\x00','\x00','\x7C','\x66','\x66','\x7C','\x60','\x60','\x00','\x00','\x3E','\x66','\x66','\x3E','\x06','\x06','\x00','\x00','\x7C','\x66','\x60','\x60','\x60','\x00','\x00','\x00','\x3E','\x60','\x3C','\x06','\x7C','\x00',
'\x00','\x18','\x7E','\x18','\x18','\x18','\x0E','\x00','\x00','\x00','\x66','\x66','\x66','\x66','\x3E','\x00','\x00','\x00','\x66','\x66','\x66','\x3C','\x18','\x00','\x00','\x00','\x63','\x6B','\x7F','\x3E','\x36','\x00',
'\x00','\x00','\x66','\x3C','\x18','\x3C','\x66','\x00','\x00','\x00','\x66','\x66','\x66','\x3E','\x0C','\x78','\x00','\x00','\x7E','\x0C','\x18','\x30','\x7E','\x00','\x3C','\x30','\x30','\x30','\x30','\x30','\x3C','\x00',
'\x0C','\x12','\x30','\x7C','\x30','\x62','\xFC','\x00','\x3C','\x0C','\x0C','\x0C','\x0C','\x0C','\x3C','\x00','\x00','\x18','\x3C','\x7E','\x18','\x18','\x18','\x18','\x00','\x10','\x30','\x7F','\x7F','\x30','\x10','\x00',
'\x00','\x00','\x00','\x00','\x00','\x00','\x00','\x00','\x18','\x18','\x18','\x18','\x00','\x00','\x18','\x00','\x66','\x66','\x66','\x00','\x00','\x00','\x00','\x00','\x66','\x66','\xFF','\x66','\xFF','\x66','\x66','\x00',
'\x18','\x3E','\x60','\x3C','\x06','\x7C','\x18','\x00','\x62','\x66','\x0C','\x18','\x30','\x66','\x46','\x00','\x3C','\x66','\x3C','\x38','\x67','\x66','\x3F','\x00','\x06','\x0C','\x18','\x00','\x00','\x00','\x00','\x00',
'\x0C','\x18','\x30','\x30','\x30','\x18','\x0C','\x00','\x30','\x18','\x0C','\x0C','\x0C','\x18','\x30','\x00','\x00','\x66','\x3C','\xFF','\x3C','\x66','\x00','\x00','\x00','\x18','\x18','\x7E','\x18','\x18','\x00','\x00',
'\x00','\x00','\x00','\x00','\x00','\x18','\x18','\x30','\x00','\x00','\x00','\x7E','\x00','\x00','\x00','\x00','\x00','\x00','\x00','\x00','\x00','\x18','\x18','\x00','\x00','\x03','\x06','\x0C','\x18','\x30','\x60','\x00',
'\x3C','\x66','\x6E','\x76','\x66','\x66','\x3C','\x00','\x18','\x18','\x38','\x18','\x18','\x18','\x7E','\x00','\x3C','\x66','\x06','\x0C','\x30','\x60','\x7E','\x00','\x3C','\x66','\x06','\x1C','\x06','\x66','\x3C','\x00',
'\x06','\x0E','\x1E','\x66','\x7F','\x06','\x06','\x00','\x7E','\x60','\x7C','\x06','\x06','\x66','\x3C','\x00','\x3C','\x66','\x60','\x7C','\x66','\x66','\x3C','\x00','\x7E','\x66','\x0C','\x18','\x18','\x18','\x18','\x00',
'\x3C','\x66','\x66','\x3C','\x66','\x66','\x3C','\x00','\x3C','\x66','\x66','\x3E','\x06','\x66','\x3C','\x00','\x00','\x00','\x18','\x00','\x00','\x18','\x00','\x00','\x00','\x00','\x18','\x00','\x00','\x18','\x18','\x30',
'\x0E','\x18','\x30','\x60','\x30','\x18','\x0E','\x00','\x00','\x00','\x7E','\x00','\x7E','\x00','\x00','\x00','\x70','\x18','\x0C','\x06','\x0C','\x18','\x70','\x00','\x3C','\x66','\x06','\x0C','\x18','\x00','\x18','\x00',
'\x00','\x00','\x00','\xFF','\xFF','\x00','\x00','\x00','\x18','\x3C','\x66','\x7E','\x66','\x66','\x66','\x00','\x7C','\x66','\x66','\x7C','\x66','\x66','\x7C','\x00','\x3C','\x66','\x60','\x60','\x60','\x66','\x3C','\x00',
'\x78','\x6C','\x66','\x66','\x66','\x6C','\x78','\x00','\x7E','\x60','\x60','\x78','\x60','\x60','\x7E','\x00','\x7E','\x60','\x60','\x78','\x60','\x60','\x60','\x00','\x3C','\x66','\x60','\x6E','\x66','\x66','\x3C','\x00',
'\x66','\x66','\x66','\x7E','\x66','\x66','\x66','\x00','\x3C','\x18','\x18','\x18','\x18','\x18','\x3C','\x00','\x1E','\x0C','\x0C','\x0C','\x0C','\x6C','\x38','\x00','\x66','\x6C','\x78','\x70','\x78','\x6C','\x66','\x00',
'\x60','\x60','\x60','\x60','\x60','\x60','\x7E','\x00','\x63','\x77','\x7F','\x6B','\x63','\x63','\x63','\x00','\x66','\x76','\x7E','\x7E','\x6E','\x66','\x66','\x00','\x3C','\x66','\x66','\x66','\x66','\x66','\x3C','\x00',
'\x7C','\x66','\x66','\x7C','\x60','\x60','\x60','\x00','\x3C','\x66','\x66','\x66','\x66','\x3C','\x0E','\x00','\x7C','\x66','\x66','\x7C','\x78','\x6C','\x66','\x00','\x3C','\x66','\x60','\x3C','\x06','\x66','\x3C','\x00',
'\x7E','\x18','\x18','\x18','\x18','\x18','\x18','\x00','\x66','\x66','\x66','\x66','\x66','\x66','\x3C','\x00','\x66','\x66','\x66','\x66','\x66','\x3C','\x18','\x00','\x63','\x63','\x63','\x6B','\x7F','\x77','\x63','\x00',
'\x66','\x66','\x3C','\x18','\x3C','\x66','\x66','\x00','\x66','\x66','\x66','\x3C','\x18','\x18','\x18','\x00','\x7E','\x06','\x0C','\x18','\x30','\x60','\x7E','\x00','\x18','\x18','\x18','\xFF','\xFF','\x18','\x18','\x18',
'\xC0','\xC0','\x30','\x30','\xC0','\xC0','\x30','\x30','\x18','\x18','\x18','\x18','\x18','\x18','\x18','\x18','\x33','\x33','\xCC','\xCC','\x33','\x33','\xCC','\xCC','\x33','\x99','\xCC','\x66','\x33','\x99','\xCC','\x66',
'\x00','\x00','\x00','\x00','\x00','\x00','\x00','\x00','\xF0','\xF0','\xF0','\xF0','\xF0','\xF0','\xF0','\xF0','\x00','\x00','\x00','\x00','\xFF','\xFF','\xFF','\xFF','\xFF','\x00','\x00','\x00','\x00','\x00','\x00','\x00',
'\x00','\x00','\x00','\x00','\x00','\x00','\x00','\xFF','\xC0','\xC0','\xC0','\xC0','\xC0','\xC0','\xC0','\xC0','\xCC','\xCC','\x33','\x33','\xCC','\xCC','\x33','\x33','\x03','\x03','\x03','\x03','\x03','\x03','\x03','\x03',
'\x00','\x00','\x00','\x00','\xCC','\xCC','\x33','\x33','\xCC','\x99','\x33','\x66','\xCC','\x99','\x33','\x66','\x03','\x03','\x03','\x03','\x03','\x03','\x03','\x03','\x18','\x18','\x18','\x1F','\x1F','\x18','\x18','\x18',
'\x00','\x00','\x00','\x00','\x0F','\x0F','\x0F','\x0F','\x18','\x18','\x18','\x1F','\x1F','\x00','\x00','\x00','\x00','\x00','\x00','\xF8','\xF8','\x18','\x18','\x18','\x00','\x00','\x00','\x00','\x00','\x00','\xFF','\xFF',
'\x00','\x00','\x00','\x1F','\x1F','\x18','\x18','\x18','\x18','\x18','\x18','\xFF','\xFF','\x00','\x00','\x00','\x00','\x00','\x00','\xFF','\xFF','\x18','\x18','\x18','\x18','\x18','\x18','\xF8','\xF8','\x18','\x18','\x18',
'\xC0','\xC0','\xC0','\xC0','\xC0','\xC0','\xC0','\xC0','\xE0','\xE0','\xE0','\xE0','\xE0','\xE0','\xE0','\xE0','\x07','\x07','\x07','\x07','\x07','\x07','\x07','\x07','\xFF','\xFF','\x00','\x00','\x00','\x00','\x00','\x00',
'\xFF','\xFF','\xFF','\x00','\x00','\x00','\x00','\x00','\x00','\x00','\x00','\x00','\x00','\xFF','\xFF','\xFF','\x01','\x03','\x06','\x6C','\x78','\x70','\x60','\x00','\x00','\x00','\x00','\x00','\xF0','\xF0','\xF0','\xF0',
'\x0F','\x0F','\x0F','\x0F','\x00','\x00','\x00','\x00','\x18','\x18','\x18','\xF8','\xF8','\x00','\x00','\x00','\xF0','\xF0','\xF0','\xF0','\x00','\x00','\x00','\x00','\xF0','\xF0','\xF0','\xF0','\x0F','\x0F','\x0F','\x0F',
'\xC3','\x99','\x91','\x91','\x9F','\x99','\xC3','\xFF','\xFF','\xFF','\xC3','\xF9','\xC1','\x99','\xC1','\xFF','\xFF','\x9F','\x9F','\x83','\x99','\x99','\x83','\xFF','\xFF','\xFF','\xC3','\x9F','\x9F','\x9F','\xC3','\xFF',
'\xFF','\xF9','\xF9','\xC1','\x99','\x99','\xC1','\xFF','\xFF','\xFF','\xC3','\x99','\x81','\x9F','\xC3','\xFF','\xFF','\xF1','\xE7','\xC1','\xE7','\xE7','\xE7','\xFF','\xFF','\xFF','\xC1','\x99','\x99','\xC1','\xF9','\x83',
'\xFF','\x9F','\x9F','\x83','\x99','\x99','\x99','\xFF','\xFF','\xE7','\xFF','\xC7','\xE7','\xE7','\xC3','\xFF','\xFF','\xF9','\xFF','\xF9','\xF9','\xF9','\xF9','\xC3','\xFF','\x9F','\x9F','\x93','\x87','\x93','\x99','\xFF',
'\xFF','\xC7','\xE7','\xE7','\xE7','\xE7','\xC3','\xFF','\xFF','\xFF','\x99','\x80','\x80','\x94','\x9C','\xFF','\xFF','\xFF','\x83','\x99','\x99','\x99','\x99','\xFF','\xFF','\xFF','\xC3','\x99','\x99','\x99','\xC3','\xFF',
'\xFF','\xFF','\x83','\x99','\x99','\x83','\x9F','\x9F','\xFF','\xFF','\xC1','\x99','\x99','\xC1','\xF9','\xF9','\xFF','\xFF','\x83','\x99','\x9F','\x9F','\x9F','\xFF','\xFF','\xFF','\xC1','\x9F','\xC3','\xF9','\x83','\xFF',
'\xFF','\xE7','\x81','\xE7','\xE7','\xE7','\xF1','\xFF','\xFF','\xFF','\x99','\x99','\x99','\x99','\xC1','\xFF','\xFF','\xFF','\x99','\x99','\x99','\xC3','\xE7','\xFF','\xFF','\xFF','\x9C','\x94','\x80','\xC1','\xC9','\xFF',
'\xFF','\xFF','\x99','\xC3','\xE7','\xC3','\x99','\xFF','\xFF','\xFF','\x99','\x99','\x99','\xC1','\xF3','\x87','\xFF','\xFF','\x81','\xF3','\xE7','\xCF','\x81','\xFF','\xC3','\xCF','\xCF','\xCF','\xCF','\xCF','\xC3','\xFF',
'\xF3','\xED','\xCF','\x83','\xCF','\x9D','\x03','\xFF','\xC3','\xF3','\xF3','\xF3','\xF3','\xF3','\xC3','\xFF','\xFF','\xE7','\xC3','\x81','\xE7','\xE7','\xE7','\xE7','\xFF','\xEF','\xCF','\x80','\x80','\xCF','\xEF','\xFF',
'\xFF','\xFF','\xFF','\xFF','\xFF','\xFF','\xFF','\xFF','\xE7','\xE7','\xE7','\xE7','\xFF','\xFF','\xE7','\xFF','\x99','\x99','\x99','\xFF','\xFF','\xFF','\xFF','\xFF','\x99','\x99','\x00','\x99','\x00','\x99','\x99','\xFF',
'\xE7','\xC1','\x9F','\xC3','\xF9','\x83','\xE7','\xFF','\x9D','\x99','\xF3','\xE7','\xCF','\x99','\xB9','\xFF','\xC3','\x99','\xC3','\xC7','\x98','\x99','\xC0','\xFF','\xF9','\xF3','\xE7','\xFF','\xFF','\xFF','\xFF','\xFF',
'\xF3','\xE7','\xCF','\xCF','\xCF','\xE7','\xF3','\xFF','\xCF','\xE7','\xF3','\xF3','\xF3','\xE7','\xCF','\xFF','\xFF','\x99','\xC3','\x00','\xC3','\x99','\xFF','\xFF','\xFF','\xE7','\xE7','\x81','\xE7','\xE7','\xFF','\xFF',
'\xFF','\xFF','\xFF','\xFF','\xFF','\xE7','\xE7','\xCF','\xFF','\xFF','\xFF','\x81','\xFF','\xFF','\xFF','\xFF','\xFF','\xFF','\xFF','\xFF','\xFF','\xE7','\xE7','\xFF','\xFF','\xFC','\xF9','\xF3','\xE7','\xCF','\x9F','\xFF',
'\xC3','\x99','\x91','\x89','\x99','\x99','\xC3','\xFF','\xE7','\xE7','\xC7','\xE7','\xE7','\xE7','\x81','\xFF','\xC3','\x99','\xF9','\xF3','\xCF','\x9F','\x81','\xFF','\xC3','\x99','\xF9','\xE3','\xF9','\x99','\xC3','\xFF',
'\xF9','\xF1','\xE1','\x99','\x80','\xF9','\xF9','\xFF','\x81','\x9F','\x83','\xF9','\xF9','\x99','\xC3','\xFF','\xC3','\x99','\x9F','\x83','\x99','\x99','\xC3','\xFF','\x81','\x99','\xF3','\xE7','\xE7','\xE7','\xE7','\xFF',
'\xC3','\x99','\x99','\xC3','\x99','\x99','\xC3','\xFF','\xC3','\x99','\x99','\xC1','\xF9','\x99','\xC3','\xFF','\xFF','\xFF','\xE7','\xFF','\xFF','\xE7','\xFF','\xFF','\xFF','\xFF','\xE7','\xFF','\xFF','\xE7','\xE7','\xCF',
'\xF1','\xE7','\xCF','\x9F','\xCF','\xE7','\xF1','\xFF','\xFF','\xFF','\x81','\xFF','\x81','\xFF','\xFF','\xFF','\x8F','\xE7','\xF3','\xF9','\xF3','\xE7','\x8F','\xFF','\xC3','\x99','\xF9','\xF3','\xE7','\xFF','\xE7','\xFF',
'\xFF','\xFF','\xFF','\x00','\x00','\xFF','\xFF','\xFF','\xE7','\xC3','\x99','\x81','\x99','\x99','\x99','\xFF','\x83','\x99','\x99','\x83','\x99','\x99','\x83','\xFF','\xC3','\x99','\x9F','\x9F','\x9F','\x99','\xC3','\xFF',
'\x87','\x93','\x99','\x99','\x99','\x93','\x87','\xFF','\x81','\x9F','\x9F','\x87','\x9F','\x9F','\x81','\xFF','\x81','\x9F','\x9F','\x87','\x9F','\x9F','\x9F','\xFF','\xC3','\x99','\x9F','\x91','\x99','\x99','\xC3','\xFF',
'\x99','\x99','\x99','\x81','\x99','\x99','\x99','\xFF','\xC3','\xE7','\xE7','\xE7','\xE7','\xE7','\xC3','\xFF','\xE1','\xF3','\xF3','\xF3','\xF3','\x93','\xC7','\xFF','\x99','\x93','\x87','\x8F','\x87','\x93','\x99','\xFF',
'\x9F','\x9F','\x9F','\x9F','\x9F','\x9F','\x81','\xFF','\x9C','\x88','\x80','\x94','\x9C','\x9C','\x9C','\xFF','\x99','\x89','\x81','\x81','\x91','\x99','\x99','\xFF','\xC3','\x99','\x99','\x99','\x99','\x99','\xC3','\xFF',
'\x83','\x99','\x99','\x83','\x9F','\x9F','\x9F','\xFF','\xC3','\x99','\x99','\x99','\x99','\xC3','\xF1','\xFF','\x83','\x99','\x99','\x83','\x87','\x93','\x99','\xFF','\xC3','\x99','\x9F','\xC3','\xF9','\x99','\xC3','\xFF',
'\x81','\xE7','\xE7','\xE7','\xE7','\xE7','\xE7','\xFF','\x99','\x99','\x99','\x99','\x99','\x99','\xC3','\xFF','\x99','\x99','\x99','\x99','\x99','\xC3','\xE7','\xFF','\x9C','\x9C','\x9C','\x94','\x80','\x88','\x9C','\xFF',
'\x99','\x99','\xC3','\xE7','\xC3','\x99','\x99','\xFF','\x99','\x99','\x99','\xC3','\xE7','\xE7','\xE7','\xFF','\x81','\xF9','\xF3','\xE7','\xCF','\x9F','\x81','\xFF','\xE7','\xE7','\xE7','\x00','\x00','\xE7','\xE7','\xE7',
'\x3F','\x3F','\xCF','\xCF','\x3F','\x3F','\xCF','\xCF','\xE7','\xE7','\xE7','\xE7','\xE7','\xE7','\xE7','\xE7','\xCC','\xCC','\x33','\x33','\xCC','\xCC','\x33','\x33','\xCC','\x66','\x33','\x99','\xCC','\x66','\x33','\x99',
'\xFF','\xFF','\xFF','\xFF','\xFF','\xFF','\xFF','\xFF','\x0F','\x0F','\x0F','\x0F','\x0F','\x0F','\x0F','\x0F','\xFF','\xFF','\xFF','\xFF','\x00','\x00','\x00','\x00','\x00','\xFF','\xFF','\xFF','\xFF','\xFF','\xFF','\xFF',
'\xFF','\xFF','\xFF','\xFF','\xFF','\xFF','\xFF','\x00','\x3F','\x3F','\x3F','\x3F','\x3F','\x3F','\x3F','\x3F','\x33','\x33','\xCC','\xCC','\x33','\x33','\xCC','\xCC','\xFC','\xFC','\xFC','\xFC','\xFC','\xFC','\xFC','\xFC',
'\xFF','\xFF','\xFF','\xFF','\x33','\x33','\xCC','\xCC','\x33','\x66','\xCC','\x99','\x33','\x66','\xCC','\x99','\xFC','\xFC','\xFC','\xFC','\xFC','\xFC','\xFC','\xFC','\xE7','\xE7','\xE7','\xE0','\xE0','\xE7','\xE7','\xE7',
'\xFF','\xFF','\xFF','\xFF','\xF0','\xF0','\xF0','\xF0','\xE7','\xE7','\xE7','\xE0','\xE0','\xFF','\xFF','\xFF','\xFF','\xFF','\xFF','\x07','\x07','\xE7','\xE7','\xE7','\xFF','\xFF','\xFF','\xFF','\xFF','\xFF','\x00','\x00',
'\xFF','\xFF','\xFF','\xE0','\xE0','\xE7','\xE7','\xE7','\xE7','\xE7','\xE7','\x00','\x00','\xFF','\xFF','\xFF','\xFF','\xFF','\xFF','\x00','\x00','\xE7','\xE7','\xE7','\xE7','\xE7','\xE7','\x07','\x07','\xE7','\xE7','\xE7',
'\x3F','\x3F','\x3F','\x3F','\x3F','\x3F','\x3F','\x3F','\x1F','\x1F','\x1F','\x1F','\x1F','\x1F','\x1F','\x1F','\xF8','\xF8','\xF8','\xF8','\xF8','\xF8','\xF8','\xF8','\x00','\x00','\xFF','\xFF','\xFF','\xFF','\xFF','\xFF',
'\x00','\x00','\x00','\xFF','\xFF','\xFF','\xFF','\xFF','\xFF','\xFF','\xFF','\xFF','\xFF','\x00','\x00','\x00','\xFE','\xFC','\xF9','\x93','\x87','\x8F','\x9F','\xFF','\xFF','\xFF','\xFF','\xFF','\x0F','\x0F','\x0F','\x0F',
'\xF0','\xF0','\xF0','\xF0','\xFF','\xFF','\xFF','\xFF','\xE7','\xE7','\xE7','\x07','\x07','\xFF','\xFF','\xFF','\x0F','\x0F','\x0F','\x0F','\xFF','\xFF','\xFF','\xFF','\x0F','\x0F','\x0F','\x0F','\xF0','\xF0','\xF0','\xF0'
};

static byte ram[64 * 1024];
static byte color_nybles[1024];

// note ram starts at 0x0000
static const int basic_addr = 0xA000;
static const int kernal_addr = 0xE000;
static const int io_addr = 0xD000;
static const int io_size = 0x1000;
static const int color_addr = 0xD800;
static const int open_addr = 0xC000;
static const int open_size = 0x1000;

static byte io[io_size];

void C64_Init(const char* basic_file, const char* chargen_file, const char* kernal_file)
{
	//File_ReadAllBytes(basic_rom, sizeof(basic_rom), basic_file);
	//File_ReadAllBytes(chargen_rom, sizeof(chargen_rom), chargen_file);
	//File_ReadAllBytes(kernal_rom, sizeof(kernal_rom), kernal_file);

	for (int i = 0; i < sizeof(ram); ++i)
		ram[i] = 0;
	for (int i = 0; i < sizeof(color_nybles); ++i)
		color_nybles[i] = 0;
	for (int i = 0; i < sizeof(io); ++i)
		io[i] = 0;

	// initialize DDR and memory mapping to defaults
	ram[0] = 0xEF;
	ram[1] = 0x07;
}

int C64ColorToLCDColor(byte value)
{
	switch (value & 0xF)
	{
		case 0: return LCD_COLOR_BLACK;
		case 1: return LCD_COLOR_WHITE;
		case 2: return LCD_COLOR_RED;
		case 3: return LCD_COLOR_CYAN;
		case 4: return LCD_COLOR_DARKMAGENTA; // PURPLE
		case 5: return LCD_COLOR_GREEN;
		case 6: return LCD_COLOR_BLUE;
		case 7: return LCD_COLOR_YELLOW;
		case 8: return LCD_COLOR_ORANGE;
		case 9: return LCD_COLOR_BROWN;
		case 10: return LCD_COLOR_LIGHTRED;
		case 11: return LCD_COLOR_DARKGRAY;
		case 12: return LCD_COLOR_DARKCYAN; // MED GRAY
		case 13: return LCD_COLOR_LIGHTGREEN;
		case 14: return LCD_COLOR_LIGHTBLUE;
		case 15: return LCD_COLOR_LIGHTGRAY;
		default: return 0;
	}
}

void DrawChar(byte c, int col, int row, int fg, int bg)
{
	int offset = ((io[0x18] & 2) == 0) ? 0 : (8*256);
	const byte* shape = &chargen_rom[c*8+offset];
	int x0 = 20 + row*8;
	int y0 = col*8;
	for (int row_i=0; row_i<8; ++row_i)
	{
		int mask = 128;
		for (int col_i=0; col_i<8; ++col_i)
		{
			lcd.DrawPixel(x0+row_i, 320 - (y0+col_i), ((shape[row_i] & mask) == 0) ? bg : fg);
			mask = mask >> 1;
		}
	}
}

static void DrawChar(int offset)
{
	int col = offset % 40;
	int row = offset / 40;
	int fg = C64ColorToLCDColor(color_nybles[offset]);
	int bg = C64ColorToLCDColor(io[0x21]);
	DrawChar(ram[1024+offset], col, row, fg, bg);
}

static void RedrawScreen()
{
	int bg = C64ColorToLCDColor(io[0x21]);
	int offset = 0;
	for (int row = 0; row < 25; ++row)
	{
		for (int col = 0; col < 40; ++col)
		{
			int fg = C64ColorToLCDColor(color_nybles[offset]);
			DrawChar(ram[1024 + offset], col, row, fg, bg);
			++offset;
		}
	}
}

byte GetMemory(ushort addr)
{
	if (addr <= sizeof(ram) - 1
		  && (
			  addr < basic_addr // always RAM
			  || (addr >= open_addr && addr < open_addr + open_size) // always open RAM C000.CFFF
			  || (((ram[1] & 3) != 3) && addr >= basic_addr && addr < basic_addr + sizeof(basic_rom)) // RAM banked instead of BASIC
			  || (((ram[1] & 2) == 0) && addr >= kernal_addr && addr <= kernal_addr + sizeof(kernal_rom) - 1) // RAM banked instead of KERNAL
			  || (((ram[1] & 3) == 0) && addr >= io_addr && addr < io_addr + io_size) // RAM banked instead of IO
		  )
		)
		return ram[addr];
	else if (addr >= basic_addr && addr < basic_addr + sizeof(basic_rom))
		return basic_rom[addr - basic_addr];
	else if (addr >= io_addr && addr < io_addr + io_size)
	{
		if ((ram[1] & 4) == 0)
			return chargen_rom[addr - io_addr];
		else if (addr >= color_addr && addr < color_addr + sizeof(color_nybles))
			return color_nybles[addr - color_addr] | 0xF0;
		else if (addr == 0xDC01)
		{
			int value = 0;
			
			for (int i=0; i<9; ++i)
			{
				int scan_code = scan_codes[i] & 127; // remove any modifiers
				if (scan_code < 64)
				{			
					int col = scan_code / 8;
					int row = scan_code % 8;
					
					if ((io[0xC00] & (1 << col)) == 0)
						value |= (1 << row);
				}
			}
			
			return ~value;
		}
		else
			return io[addr - io_addr];
	}
	else if (addr >= kernal_addr && addr <= kernal_addr + sizeof(kernal_rom)-1)
		return kernal_rom[addr - kernal_addr];
	else
		return 0xFF;
}

void SetMemory(ushort addr, byte value)
{
	if (addr <= sizeof(ram) - 1
		&& (
			addr < io_addr // RAM, including open RAM, and RAM under BASIC
			|| (addr >= kernal_addr && addr <= kernal_addr + sizeof(kernal_rom) - 1) // RAM under KERNAL
			|| (((ram[1] & 7) == 0) && addr >= io_addr && addr < io_addr + io_size) // RAM banked in instead of IO
			)
		)
	{
		ram[addr] = value;
		if (addr >= 1024 && addr < 2024)
			DrawChar(addr - 1024);
	}
	else if (addr == 0xD018) // VIC-II Chip Memory Control Register
	{
		io[addr - io_addr] = value;
		RedrawScreen(); // upper to lower or lower to upper
	}
	else if (addr == 0xD020) // border
	{
		lcd.SetTextColor(C64ColorToLCDColor(value));
		lcd.FillRect(0, 0, 20, 320);
		lcd.FillRect(220, 0, 20, 320);
		io[addr - io_addr] = value & 0xF;
	}	
	else if (addr == 0xD021) // background
	{
		lcd.SetBackColor(C64ColorToLCDColor(value));
		lcd.SetTextColor(C64ColorToLCDColor(value));
		io[addr - io_addr] = value & 0xF;
		RedrawScreen();
	}
	else if (addr >= color_addr && addr < color_addr + sizeof(color_nybles))
	{
		int offset = addr - color_addr;
		color_nybles[offset] = value;
		DrawChar(offset);
	}
	else if (addr == 0xDC00)
	{
		io[addr - io_addr] = value;
	}	
	//else if (addr >= io_addr && addr < io_addr + io.Length)
	//    io[addr - io_addr] = value;
}
