// ----------------
// CTIA Helmet Demo
// ----------------

#include "mbed.h"
#include "mDot.h"
#include "rtos.h"
#include <H3LIS331DL.h>
#include <string>
#include <vector>

#define G_THRESHOLD 5

//please get these value by running H3LIS331DL_AdjVal
#define VAL_X_AXIS  71 //68
#define VAL_Y_AXIS  97 //38
#define VAL_Z_AXIS  722 //664
#define GAIN 0.003

// these options must match the settings on your Conduit
static std::string config_network_name = "Escalation";
static std::string config_network_pass = "Escalation";
static uint8_t config_frequency_sub_band = 3;

//Globals
mDot* dot;
H3LIS331DL* h3lis=new H3LIS331DL(I2C_SDA, I2C_SCL);
DigitalOut led(LED1);
Ticker ledTick;
Ticker heart;
Timer timer;
volatile bool timeToSendHeartbeat = true;

//Function prototypes
void log_error(mDot* dot, const char* msg, int32_t retval);
bool setFrequencySubBand(uint8_t subBand);
bool setNetworkName(const std::string name);
bool setNetworkPassphrase(const std::string passphrase);
bool setAck(uint8_t retries);
bool joinNetwork();
bool sendData(const std::string text);
H3LIS331DL* rebuild(H3LIS331DL* h3lis );


void ledTock() {
    led = !led;
}

void beat() {
  timeToSendHeartbeat = true;
}
int begin;


int main() {
    int32_t ret;
    unsigned char id;
    std::vector<uint8_t> data;
    char data_str[40];
    int16_t x=0, y=0, z=0, x_max=0, y_max=0, z_max=0;
    uint16_t vector_sum=0, vector_sum_max=0;

    ledTick.attach(&ledTock, 0.1);

    // get the mDot handle
    dot = mDot::getInstance();


    // reset to default config so we know what state we're in
    dot->resetConfig();
    dot->getDeviceId();
    
    printf("\r\n\r\n");
    printf("=======================\r\n");
    printf("   CTIA Helment Demo\r\n");
    printf("=======================\r\n");

    printf("Library version: %s\r\n\r\n", dot->getId().c_str());

    // set up the mDot with our network information
    setFrequencySubBand(config_frequency_sub_band);
    setNetworkName(config_network_name);
    setNetworkPassphrase(config_network_pass);
    setAck(0);  //disable acks

    printf("Initializing accelerometer...\r\n");
    // H3LIS331DL_FULLSCALE_4 = +- 200g, H3LIS331DL_FULLSCALE_2 = +- 100g
    h3lis->init(H3LIS331DL_ODR_1000Hz, H3LIS331DL_NORMAL, H3LIS331DL_FULLSCALE_2);
//    h3lis.init(H3LIS331DL_ODR_100Hz, H3LIS331DL_NORMAL, H3LIS331DL_FULLSCALE_2);


    if (!h3lis->getWHO_AM_I(&id)) {
        printf("Failed to initialize accelerometer!\r\n");
        return 0;
    }

    // Join network
    while (!joinNetwork()) { wait(5); };
    ledTick.detach();
    led = 1;

    heart.attach(&beat, 10);
    timer.start();
    while (true) {
        begin = timer.read_us();
        
        h3lis->readXYZ(&x,&y,&z);
        x = (x - VAL_X_AXIS) * GAIN;
        y = (y - VAL_Y_AXIS) * GAIN;
        z = (z - VAL_Z_AXIS) * GAIN;
        
        // printf takes about 50 ms !
        // printf("Reading: X:% d  Y:% d  Z:% d, timer: % d\r\n", x, y, z, begin);

        vector_sum = sqrt((double)(x*x + y*y + z*z));

        // If impact occuring
        if (vector_sum > G_THRESHOLD) {
            //Update peak forces if needed
            if (vector_sum > vector_sum_max) {
                vector_sum_max = vector_sum;
                x_max = x;
                y_max = y;
                z_max = z;
            }
            printf("Reading: X:% d  Y:% d  Z:% d\r\n", x, y, z);
        }
        // If impact detected
        else if (vector_sum_max > 0) {
            printf("IMPACT: X:% d  Y:% d  Z:% d\r\n", x_max, y_max, z_max);
            sprintf(data_str, "%d,%d,%d", x_max, y_max, z_max);

            // send the data
            sendData(data_str);

            // Reset maximums
            vector_sum_max = 0;
            x_max = 0;
            y_max = 0;
            z_max = 0;
            timeToSendHeartbeat = false;
        }
        //Idle
        else {
          if (timeToSendHeartbeat) {
            sendData("heartbeat");
            timeToSendHeartbeat = false;
          }
        }
        wait_ms(10); //Accelerometer reading updates every 1ms
    }

    return 0;
}

void log_error(mDot* dot, const char* msg, int32_t retval) {
    printf("%s - %ld:%s, %s\r\n", msg, retval, mDot::getReturnCodeString(retval).c_str(), dot->getLastError().c_str());
}

bool setFrequencySubBand(uint8_t subBand)
{
    int32_t ret;
    printf("Setting frequency sub band to '%d'...\r\n", subBand);
    if ((ret = dot->setFrequencySubBand(subBand)) != mDot::MDOT_OK) {
        log_error(dot, "Failed to set frequency sub band", ret);
        return false;
    }
    return true;
}

bool setNetworkName(const std::string name)
{
    int32_t ret;
    printf("Setting network name to '%s'...\r\n", name.c_str());
    if ((ret = dot->setNetworkName(name)) != mDot::MDOT_OK) {
        log_error(dot, "Failed to set network name", ret);
        return false;
    }
    return true;
}

bool setNetworkPassphrase(const std::string passphrase)
{
    int32_t ret;
    printf("Setting passphrase...\r\n");
    if ((ret = dot->setNetworkPassphrase(passphrase)) != mDot::MDOT_OK) {
        log_error(dot, "Failed to set network password", ret);
        return false;
    }
    return true;
}

bool setAck(uint8_t retries)
{
    int32_t ret;
    printf("Setting ack...\r\n");
    if ((ret = dot->setAck(retries)) != mDot::MDOT_OK)
    {
        log_error(dot, "Failed to set ack", ret);
        return false;
    }
    return true;
}

bool joinNetwork()
{
    int32_t ret;
    printf("\r\nJoining network...  ");
    // reset to default config so we know what state we're in


    if ((ret = dot->joinNetwork()) != mDot::MDOT_OK) {
        log_error(dot, "Failed", ret);
        dot->resetConfig();
        dot->getDeviceId();
        osDelay(200);
        return false;
    }
    printf("Network Joined!\r\n");
    return true;
}

bool sendData(const std::string text)
{
    if (dot->getNextTxMs() != 0) {
        printf("Sending in %lu ms...\r\n", dot->getNextTxMs());
        return false;
    }

    int32_t ret;
    printf("Sending: '%s'   ", text.c_str());
    std::vector<uint8_t> data(text.begin(), text.end());
    if ((ret = dot->send(data, 1)) != mDot::MDOT_OK)
    {
        log_error(dot, "Failed to send data", ret);
        return false;
    }
    printf("Data sent!\r\n");
    return true;
}

H3LIS331DL* rebuild(H3LIS331DL* h3lis ){
    H3LIS331DL* new_h3lis = new H3LIS331DL( I2C_SDA, I2C_SCL );
    delete h3lis;
    new_h3lis->init(H3LIS331DL_ODR_1000Hz, H3LIS331DL_NORMAL, H3LIS331DL_FULLSCALE_2);

    return new_h3lis;
}


