/* mbed MCP23S17 Library, for driving the MCP23S17 16-Bit I/O Expander with Serial Interface (SPI)
 * Copyright (c) 2011, Created by Steen Joergensen (stjo2809) inspired by Romilly Cocking MCP23S17 library
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#include "mbed.h"

#ifndef MBED_MCP23S17_H
#define MBED_MCP23S17_H

//=============================================================================
// All Registers and there Address if BANK = 0
//=============================================================================

#define MCP23S17_IODIRA   0x00       // Controls the direction of the data I/O on Port A       
#define MCP23S17_IODIRB   0x01       // Controls the direction of the data I/O on Port B
#define MCP23S17_IPOLA    0x02       // Configure the polarity on the corresponding GPIO (Port A)
#define MCP23S17_IPOLB    0x03       // Configure the polarity on the corresponding GPIO (Port B)
#define MCP23S17_GPINTENA 0x04       // Controls the interrupt-on change feature for each pin for Port A
#define MCP23S17_GPINTENB 0x05       // Controls the interrupt-on change feature for each pin for Port B
#define MCP23S17_DEFVALA  0x06       // The default comparison value if the INTCONA is set to "1" for Port A
#define MCP23S17_DEFVALB  0x07       // The default comparison value if the INTCONA is set to "1" for Port B
#define MCP23S17_INTCONA  0x08       // Controls how the associated pin value is compared for the interrupt-on-change feature for Port A
#define MCP23S17_INTCONB  0x09       // Controls how the associated pin value is compared for the interrupt-on-change feature for Port B
#define MCP23S17_IOCON    0x0A       // Contains several bits for configuring the device
#define MCP23S17_GPPUA    0x0C       // Controls the pull-up resistors for the port pins for port A
#define MCP23S17_GPPUB    0x0D       // Controls the pull-up resistors for the port pins for port B
#define MCP23S17_INTFA    0x0E       // READ ONLY // reflects the interrupt condition on port A pins of any pin that is enabled for interrupts via the GPINTEN register.
#define MCP23S17_INTFB    0x0F       // READ ONLY // reflects the interrupt condition on port B pins of any pin that is enabled for interrupts via the GPINTEN register.
#define MCP23S17_INTCAPA  0x10       // READ ONLY // captures the GPIO port A value at the time the interrupt occurred
#define MCP23S17_INTCAPB  0x11       // READ ONLY // captures the GPIO port B value at the time the interrupt occurred
#define MCP23S17_GPIOA    0x12       // Reflects the value on the port A (doing write function it only read input)
#define MCP23S17_GPIOB    0x13       // Reflects the value on the port B (doing write function it only read input)
#define MCP23S17_OLATA    0x14       // A write to this register modifies the output latches that modifies the pins configured as outputs for Port A
#define MCP23S17_OLATB    0x15       // A write to this register modifies the output latches that modifies the pins configured as outputs for Port B

//=============================================================================
// Declaration of variables & custom #defines
//=============================================================================

#define INTERRUPT_MIRROR_BIT   0x40
#define INTERRUPT_POLARITY_BIT 0x02

enum Port { PORT_A, PORT_B };
enum Polarity { ACTIVE_LOW, ACTIVE_HIGH };

//=============================================================================
// Functions Declaration
//=============================================================================

/** Interface to the MCP23S17 16-Bit I/O Expander with Serial Interface (SPI) 
 *
 *  Using the driver:
 *   - remenber to setup SPI in main routine.
 *   - remenber to setup interrupt pin or pins in main routine (if you are using interrupts).
 *
 *  Defaults in this driver:
 *   - as default is hardware adressing "On" and if disable use "0" in hardwareaddress when creating the instance.
 *   - as default is interrupt pins "Active High".
 *   - as default is INTA is associated with PortA and INTB is associated with PortB. 
 *
 *  Limitations of using this driver:
 *   - can't use Open-Drain output.
 *   - can't use Sequential Operation mode bit.
 *   - can't use BANK 1 addressing.
 *
 */
class MCP23S17 {
public:
    /** Create an instance of the MCP23S17 connected to specfied SPI pins, with the specified address.
     *
     * @param hardwareaddress The SPI hardware address 0-7 for this MCP23S17.
     * @param spi The mbed SPI instance (make in main routine)
     * @param nCs The SPI chip select pin.
     * @param nReset The Hardware reset pin.
     */
    MCP23S17(int hardwareaddress, SPI& spi, PinName nCs, PinName nReset);

    /** Read the port IO pins level.
     *
     * @param Port The selected port to read from.
     * @return The 8 bits read, but only the value of the inputs (outputs is read as "0").
     */
    char read(Port port);

    /** Write to the port IO pins.
     *
     * @param Port The selected port to write to.
     * @param data The 8 bits to write to the IO port, but will only change the output.
     */
    void write(Port port, char data);

    /** Resetting the MCP23S17.
     *
     * Reset has to be pull down for min. 1uS to insure correct reset.
     * This function pull down the reset pin for 5uS.
     */
    void reset();
    
    /** Read a register.
     *
     * @param address Enter register navn to read from.
     * @return The 8 bits read, but only the value of the inputs (outputs is read as "0").
     */
    char read_register(char address);

    /** Configure the control setting (IOCON register).
     *
     * @param data The 8 bits to write to IOCON register.
     */
    void config_control_register(char data);

    /** Configure if the port pins has Pull ups or not (GPPUn register).
     *
     * @param Port The selected port to write to.
     * @param data The 8 bits to write to GPPUn register.
     */
    void config_pullups(Port port, char data);

    /** Configure the direction of the port pins (IODIRn register).
     *
     * @param Port The selected port to write to.
     * @param data The 8 bits to write to IODIRn register.
     */
    void config_direction(Port port, char data);

    /** Configure the input polarity of the port pins (IPOLn register).
     *
     * @param Port The selected port to write to.
     * @param data The 8 bits to write to IPOLn register.
     */
    void config_polarity(Port port, char data);

    /** Configure if the port pins is interruptable or not (GPINTENn register).
     *
     * @param Port The selected port to write to.
     * @param data The 8 bits to write to GPINTENn register.
     */
    void config_interrupt_enable(Port port, char data);

    /** Configure if the interrupts is compared to default value register or against the previous value (INTCONn register).
     *
     * @param Port The selected port to write to.
     * @param data The 8 bits to write to INTCONn register.
     */
    void config_interrupt_control(Port port, char data);

    /** Configure if the interrupts pins is connected internally or separated into intA and intB (default = separated).
     *
     * @param mirror Write True or False (True = interconnected).
     */
    void config_mirror_interrupt(bool mirror);

    /** Configure the default value that the interrupt value is compared to if it setup for use (DEFVALn register).
     *
     * @param Port The selected port to write to.
     * @param data The 8 bits to write to DEFVALn register.
     */
    void config_defaultvalue(Port port, char data);

    /** Configure if the interrupts is active high or active low.
     *
     * @param polarity Write ACTIVE_LOW or ACTIVE_HIGH.
     */
    void config_interrupt_polarity(Polarity polarity);

    /** Read the interrupt flags for the port.
     *
     * @param Port The selected port to read from.
     * @return The 8 bits read, but only value for the inputs (outputs is read as "0").
     */
    char read_interrupt_flag(Port port);

    /** Read the value of the port captured the time the interrupt occurred (use this is for fast changing inputs).
     *
     * @param Port The selected port to read from.
     * @return The 8 bits read, but only value for the inputs (outputs is read as "0").
     */
    char read_interrupt_capture(Port port);

private:
    SPI& _spi;
    DigitalOut _nCs;
    DigitalOut _nReset;
    int _hardwareaddress;
    char _writeopcode;
    char _readopcode;
    void _initialization();
    void _make_opcode(int _hardwareaddress);
    char _read(char address);                          // _read function is overloaded
    char _read(Port port, char address);        
    void _write(char address, char data);              // _write function is overloaded
    void _write(Port port, char address, char data);
    
};

#endif

