/* mbed MCP23S17 Library, for driving the MCP23S17 16-Bit I/O Expander with Serial Interface (SPI)
 * Copyright (c) 2011, Created by Steen Joergensen (stjo2809) inspired by Romilly Cocking MCP23S17 library
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
 
 #include "mbed.h"
 #include "MCP23S17.h"
 
//=============================================================================
// Public functions
//=============================================================================
    
    MCP23S17::MCP23S17(int hardwareaddress, SPI& spi, PinName nCs, PinName nReset) : _spi(spi), _nCs(nCs), _nReset(nReset)
    {
        _hardwareaddress = hardwareaddress;
        _make_opcode(_hardwareaddress);
        _initialization();        
    }

    char MCP23S17::read(Port port)
    {
        return _read(port, MCP23S17_INTCAPA );    
    }

    void MCP23S17::write(Port port, char data)
    {
        _write(port, MCP23S17_OLATA, data);
    }

    void MCP23S17::reset()
    {
        _nReset = 0;
        wait_us(5);
        _nReset = 1;
        _initialization();
    }
    
    char MCP23S17::read_register(char address)
    {
        return _read(address);     
    }

    void MCP23S17::config_control_register(char data)
    {
        _write(MCP23S17_IOCON, data);
    }

    void MCP23S17::config_pullups(Port port, char data)
    {
        _write(port, MCP23S17_GPPUA, data);
    }

    void MCP23S17::config_direction(Port port, char data)
    {
        _write(port, MCP23S17_IODIRA, data);
    }

    void MCP23S17::config_polarity(Port port, char data)
    {
        _write(port, MCP23S17_IPOLA, data);
    }

    void MCP23S17::config_interrupt_enable(Port port, char data)
    {
        _write(port, MCP23S17_GPINTENA, data);
    }

    void MCP23S17::config_interrupt_control(Port port, char data)
    {
        _write(port, MCP23S17_INTCONA , data);
    }

    void MCP23S17::config_mirror_interrupt(bool mirror)
    {
        char kopi_iocon = _read(MCP23S17_IOCON);
        if (mirror)
        {
            kopi_iocon = kopi_iocon | INTERRUPT_MIRROR_BIT;
        } 
        else
        {
            kopi_iocon = kopi_iocon & ~INTERRUPT_MIRROR_BIT;
        }
        _write(MCP23S17_IOCON, kopi_iocon);
    }

    void MCP23S17::config_defaultvalue(Port port, char data)
    {
        _write(port, MCP23S17_DEFVALA, data);
    }

    void MCP23S17::config_interrupt_polarity(Polarity polarity)
    {
        char kopi_iocon = _read(MCP23S17_IOCON);
        if (polarity == ACTIVE_LOW)
        {
            kopi_iocon = kopi_iocon | INTERRUPT_POLARITY_BIT;
        } 
        else
        {
            kopi_iocon = kopi_iocon & ~INTERRUPT_POLARITY_BIT;
        }
        _write(MCP23S17_IOCON, kopi_iocon);
    }

    char MCP23S17::read_interrupt_flag(Port port)
    {
        return _read(port, MCP23S17_INTFA);
    }

    char MCP23S17::read_interrupt_capture(Port port)
    {
        return _read(port, MCP23S17_INTCAPA );
    } 
 
//=============================================================================
// Private functions
//=============================================================================

    void MCP23S17::_initialization()
    {
        _write(MCP23S17_IOCON, 0x2A); // setup af control register (BANK = 0, MIRROR = 0, SEQOP = 1, DISSLW = 0, HAEN = 1, ODR = 0, INTPOL = 1, NC = 0)
    }
    
    void MCP23S17::_make_opcode(int _hardwareaddress)
    {
        switch(_hardwareaddress)
        {
            case 0:
            _writeopcode = 0x40;
            _readopcode = 0x41;
            break;

            case 1:
            _writeopcode = 0x42;
            _readopcode = 0x43;
            break;
            
            case 2:
            _writeopcode = 0x44;
            _readopcode = 0x45;
            break;
            
            case 3:
            _writeopcode = 0x46;
            _readopcode = 0x47;
            break;
            
            case 4:
            _writeopcode = 0x48;
            _readopcode = 0x49;
            break;                        
                        
            case 5:
            _writeopcode = 0x4A;
            _readopcode = 0x4B;
            break;
           
            case 6:
            _writeopcode = 0x4C;
            _readopcode = 0x4D;
            break;
            
            case 7:
            _writeopcode = 0x4E;
            _readopcode = 0x4F;
            break;                                
        }
    }
    
    char MCP23S17::_read(char address)                         // _read function is overloaded
    {
        _nCs = 0;
        _spi.write(_readopcode);
        _spi.write(address);
        char response = _spi.write(0xFF);                      // 0xFF data to get response
        _nCs = 1;
        return response;
    }
                                    
    char MCP23S17::_read(Port port, char address)              // _read function is overloaded
    {
        return _read(address + (char) port);
    }
            
    void MCP23S17::_write(char address, char data)             // _write function is overloaded
    {
        _nCs = 0;
        _spi.write(_writeopcode);
        _spi.write(address);
        _spi.write(data);
        _nCs = 1;
    }
    
    void MCP23S17::_write(Port port, char address, char data)  // _write function is overloaded
    {
        return _write(address + (char) port, data);
    }
    