
/*
Copyright (c) 2010 Donatien Garnier (donatiengar [at] gmail [dot] com)
 
Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:
 
The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.
 
THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/

#include "SerialBuf.h"
#include "mbed.h"

//#define __DEBUG
#include "dbg/dbg.h"

#include "netCfg.h"
#if NET_GPRS

#if NET_USB_SERIAL
#define m_pStream( a ) (m_pSerial?m_pSerial->a:m_pUsbSerial->a)
#else
#define m_pStream( a ) (m_pSerial->a)
#endif

SerialBuf::SerialBuf(int len) : m_trmt(false), m_pSerial(NULL), m_intCanReadData(true), m_readMode(false) //Buffer length
#if NET_USB_SERIAL
, m_pUsbSerial(NULL) 
#endif
{
  m_buf = new char[len];
  m_bufLen=len;
  m_pRead=m_buf;
  m_pReadStart=m_buf;
  m_pReadByInt=m_buf;
  m_pWrite=m_buf;
}

SerialBuf::~SerialBuf()
{
  delete[] m_buf;
}

void SerialBuf::attach(Serial* pSerial)
{
  m_pSerial = pSerial;
  m_pSerial->attach<SerialBuf>(this, &SerialBuf::onSerialInterrupt);
//  onSerialInterrupt(); //Flush hw buffer into current buf
}

#if NET_USB_SERIAL
void SerialBuf::attach(UsbSerial* pUsbSerial)
{
  m_pUsbSerial = pUsbSerial;
  //m_pUsbSerial->attach<SerialBuf>(this, &SerialBuf::onSerialInterrupt);
  m_usbTick.attach_us<SerialBuf>(this, &SerialBuf::onSerialInterrupt, 10000); 
}
#endif

void SerialBuf::detach()
{
  if(m_pSerial)
  {
    m_pSerial->attach(0);
    m_pSerial = NULL;
  }
  #if NET_USB_SERIAL
  else if(m_pUsbSerial)
  {
    m_usbTick.detach();
    m_pUsbSerial = NULL;
  }
  #endif
}


void SerialBuf::onSerialInterrupt() //Callback from m_pSerial
{
  //DBG("\r\n[INT]");
 // Timer tmr;
 // tmr.start();
//  static volatile bool incompleteData = true;
//  static volatile char* pLastIntReadPos = m_buf;
  if(!(m_pStream(readable())))
  {
    return;
  }
  int len=0;
  do
  {
    if(room()>0)
    {
      len++;
#ifdef __DEBUGVERBOSE
      char c = m_pStream(getc());
      DBG("\r\n[%c]",c);
      put(c);
#else
      put(m_pStream(getc()));
#endif
    }
    else
    {
      DBG("\r\nWARN: SerialBuf Overrun");
      m_pStream(getc());
    }
  } while(m_pStream(readable()));
  //  DBG("\r\n[/INT]=*%d*\r\n w len=*%d*",tmr.read_us(),len);

  if( m_intCanReadData )
  {
    volatile bool u_readMode = m_readMode; //Save User context on interrupt
    volatile bool handled = onRead();
    if(handled)
    {
      m_pReadByInt = m_pRead;
      if(m_pRead==m_pReadStart)
      {
        //Data has been processed
      }
      else
      {
        m_pRead = m_pReadStart;
        m_intCanReadData = false; 
        //Data has to be processed in user space
      }
    }
    setReadMode( u_readMode );
  }


#if 0  
  if( incompleteData  || ( pLastIntReadPos != m_pRead ) )
  {
    bool u_readMode = m_readMode; //Save User context on interrupt
    incompleteData = onRead();
    if(!incompleteData)
      m_pRead = m_pReadStart;
    pLastIntReadPos = m_pRead;
    
  }
#endif
  
}

char SerialBuf::getc()
{
//  DBG("\r\n\[GETC]");
#if 0
  if(m_trmt) //Was transmitting
  {
    DBG("\r\n<\r\n");
    m_trmt=false;
  }
#endif
  char c;
  c = get();
  DBG("%c", c);
//  DBG("\r\n[/GETC]");
  return c;
}

void SerialBuf::putc(char c)
{
#if 0
  if(!m_trmt) //Was receiving
  {
    DBG("\r\n>\r\n");
    m_trmt=true;
  }
#endif
//  m_pSerial->writeable();
//  while(!m_pSerial->writeable() /*|| m_pSerial->readable()*/)
//  {
//    wait_us(100);
//    DBG("\r\nWait...\r\n");
//  }
/*
  NVIC_DisableIRQ(UART1_IRQn);
  NVIC_DisableIRQ(UART2_IRQn);
  NVIC_DisableIRQ(UART3_IRQn);
  */
//  onSerialInterrupt();
  m_pStream(putc(c));
/*  NVIC_EnableIRQ(UART1_IRQn);
  NVIC_EnableIRQ(UART2_IRQn);
  NVIC_EnableIRQ(UART3_IRQn);
  */
  DBG("%c", c);
}

/* Buffer Stuff */

bool SerialBuf::readable()
{
//  onSerialInterrupt(); //Flush hw buffer into current buf
 // DBG("\r\nLen=%d",len());
  return (len()>0);
}

bool SerialBuf::writeable()
{
  return m_pStream(writeable());
}

void SerialBuf::setReadMode(bool readMode) //If true, keeps chars in buf when read, false by default
{
  if(m_readMode == true && readMode == false)
  {
    //Trash bytes that have been read
    flushRead();
  }
  m_readMode=readMode;
}

void SerialBuf::flushRead() //Delete chars that have been read (only useful with readMode = true)
{
  m_pReadStart = m_pRead;
}

void SerialBuf::resetRead() //Go back to initial read position (only useful with readMode = true)
{
  m_pRead = m_pReadStart;
}

bool SerialBuf::onRead()
{
  return false;
  //Nothing here
}

char SerialBuf::get() //Get a char from buf
{
  //WARN: Must call len() before
  char c = *m_pRead;
  m_pRead++;
  
  if(m_pRead>=m_buf+m_bufLen)
    m_pRead=m_buf;
    
  if(!m_readMode) //If readmode=false, trash this char
    m_pReadStart=m_pRead;
    
  if(m_pRead==m_pReadByInt) //Next message can be processed by interrupt
    m_intCanReadData = true;
/*  else if(m_intCanReadData) //Increment this address
    m_pReadByInt=m_pRead;*/
  return c;
}

void SerialBuf::put(char c) //Put a char in buf
{
  //WARN: Must call room() before
  *m_pWrite = c;
  m_pWrite++;
  if(m_pWrite>=m_buf+m_bufLen)
    m_pWrite=m_buf;
}

int SerialBuf::room() //Return room available in buf
{
  //return m_bufLen - len() - 1; //-1 is to avoid loop
  if ( m_pReadStart > m_pWrite )
    return ( m_pReadStart - m_pWrite - 1 );
  else
    return m_bufLen - ( m_pWrite - m_pReadStart ) - 1;
}

int SerialBuf::len() //Return chars length in buf
{
  if ( m_pWrite >= m_pRead )
    return ( m_pWrite - m_pRead );
  else
    return m_bufLen - ( m_pRead - m_pWrite ); // = ( ( m_buf + m_bufLen) - m_pRead ) + ( m_pWrite - m_buf )
}

#endif
