//=====================================================================================================
// Mag_track.cpp
//=====================================================================================================
//
// Implementation of Magnetic tracking algorithm.
//
// Date         Author          Notes
// 30/01/2020   DAFG            Initial release
//
//=====================================================================================================
// Header files
    #include "mbed.h"
    #include "Mag_track.h"
    #include <math.h>
    #include "MadgwickAHRS.h"
    #include "Scattered_interpolator.h"

//----------------------------------------------------------------------------------------------------
// Variable declaration
    ///////////////////////////////////////////////////////////
    //  Filter Code Definitions
    //////////////////////////////////////////////////////////////
    
    // Maximum number of inputs that can be handled
    // in one function call
    #define MAX_INPUT_LEN   16
    // Maximum length of filter than can be handled
    #define MAX_FLT_LEN     84
    // Buffer to hold all of the input samples
    #define BUFFER_LEN      (MAX_FLT_LEN + MAX_INPUT_LEN)
    
    // Arrays to hold input samples
    int16_t lp_insamp[ 3*BUFFER_LEN ];
    int16_t insamp[ 3*BUFFER_LEN ];
    
    #define FILTER_LEN_BP  71        
    // fs = 800 Hz; Band pass 85-115 Hz
    int16_t coeffs_bp[ FILTER_LEN_BP ] =
    {
         30, 0, -28, -39, -27, 0, 21, 21, 4, 0, 31, 81, 91, 0, -182, -340, -307, 0,
         465, 786, 653, 0, -861, -1371, -1079, 0, 1292, 1970, 1487, 0, -1648, -2421,
         -1763, 0, 1823, 2588, 1823, 0, -1763, -2421, -1648, 0, 1487, 1970, 1292, 0,
         -1079, -1371, -861, 0, 653, 786, 465, 0, -307, -340, -182, 0, 91, 81, 31, 0,
         4, 21, 21, 0, -27, -39, -28, 0, 30
    };
    
    #define FILTER_LEN_LP 82
    // fs = 800 Hz; low pass fc 10-35 Hz
    int16_t coeffs_lp[FILTER_LEN_LP] =
    {
        -7,-5,-7,-9,-11,-13,-15,-15,-15,-14,-11,-5,3,15,30,49,73,102,136,176,221,272,328,390,
        456,527,601,678,757,836,914,990,1064,1133,1197,1254,1303,1343,1374,1395,1406,1406,1395,
        1374,1343,1303,1254,1197,1133,1064,990,914,836,757,678,601,527,456,390,328,272,221,176,
        136,102,73,49,30,15,3,-5,-11,-14,-15,-15,-15,-13,-11,-9,-7,-5,-7
    };
    // I2C buffers
    char data_write[2]; // 2 bytes for writing I2C
    char data_read[12]; // 12 bytes for storing I2C sensor data
    char rawData_a[6];  // x/y/z accel register data stored here
    char rawData_g[6];  // x/y/z accel register data stored here
    // Sensor Offsets
    int16_t offset_mag[3] = {-226,-135,1506}; //{-267, -100, 1552};
    int16_t offset_acc[3] = {-17,170,602};//{-123, 123, 760};
    int16_t offset_gyr[3] = {-32,-42,-5};//{-40,-37,27};
    // Sensor Settings Data
    int KMX62_ADDR = 28; // KMX62 address
    int MPU_ADDR = 208; // 0x68 = 104 *2
    char REG_MPU[6] = {0x3b,0x43,0x1c,0x1b,0x1a,0x1d};       //Acc data, gyr data, acc_config, gyro_config, mpu_config
    char REG[8] = {0x0A,0x10,0x38,0x3a,0x77,0x78,0x79,0x7E}; //Acc data, Mag data, ODC, Ctrl2 REG, buff_ctrl 1, 2 and 3, buff read
    char set_up[5] = {0x77,0x1f,0x64,0x02,0x0E};             //ODC, Ctrl2, buff_ctrl set up
    char set_up_mpu[4] = {0x08,0x10,0x00,0x00};              //Acc_config, gyro_config, set up and acc_config_2
    // DSP buffers
    double E_c[4] = {9.402,-4.057,0.8096,0.1276};       //Envelope extraction amplitude regularization coeff
    int16_t E[3] = {0,0,0};
    double E_D;
    int16_t Test_point[2] ;
    int16_t TRIAG = 0;
    int16_t inter_p[2]={0,0};
    // MCU ports setting
    I2C i2c(I2C_SDA, I2C_SCL);
    Serial PC(SERIAL_TX, SERIAL_RX);
    DigitalOut myled(LED1);
    

//---------------------------------------------------------------------------------------------------
// Function declarations

    /////// FIR init
    void firFixedInit( void )
    {
         memset( insamp, 0, sizeof( insamp ) );
         memset( lp_insamp, 0, sizeof( lp_insamp ) );
    }
    
    /////// the FIR filter function
    void firFixed( int16_t *coeffs, int16_t *input, int16_t *output,
            int length, int filterLength, int bias, int16_t *buffer )
    {
         int32_t acc;     // accumulator for MACs
         int16_t *coeffp; // pointer to coefficients
         int16_t *inputp; // pointer to input samples
         int16_t *bufferp; //pointer to input buffer
         int n;
         int k;
    
         // put the new samples at the high end of the buffer
         /*memcpy( &insamp[filterLength - 1 + bias*BUFFER_LEN], input,
                 length * sizeof(int16_t) );*/
         bufferp=buffer;
         memcpy( bufferp + filterLength - 1 + bias*BUFFER_LEN, input, length * sizeof(int16_t) );
    
         // apply the filter to each input sample
         for ( n = 0; n < length; n++ ) {
             // calculate output n
             coeffp = coeffs;
             /*inputp = &insamp[filterLength - 1 + n + bias*BUFFER_LEN];*/
             inputp = bufferp+filterLength - 1 + n + bias*BUFFER_LEN;
             // load rounding constant
             acc = 1 << 14;
             // perform the multiply-accumulate
             for ( k = 0; k < filterLength; k++ ) {
                 acc += (int32_t)(*coeffp++) * (int32_t)(*inputp--) ;
             }
             // saturate the result
             if ( acc > 0x3fffffff ) {
                 acc = 0x3fffffff;
             } else if ( acc < -0x40000000 ) {
                 acc = -0x40000000;
             }
             // convert from Q30 to Q15
             output[n] = (int16_t)(acc >> 15);
    
         }
    
         // shift input samples back in time for next time
         //memmove(&insamp[0+bias*BUFFER_LEN], &insamp[length + bias*BUFFER_LEN], (filterLength-1)*sizeof(int16_t));
         memmove( bufferp + bias*BUFFER_LEN, bufferp+length + bias*BUFFER_LEN, (filterLength-1)*sizeof(int16_t));
    
    }
    
    
    /////// SIGN FUNCTION
    int sign(int16_t x){
      if (x>0) return 1;
      if (x<0) return -1;
      return 0;
      }
    /////// QUATERNION product FUNCTION
    void Qprod(float *p, float *q, float *r ){
         r[0]= p[0] * q[0] - p[1] * q[1] - p[2] * q[2] - p[3] * q[3];
         r[1]= p[0]*q[1] + p[1]*q[0] + p[2]*q[3] - p[3]*q[2];
         r[2]= p[0]*q[2] - p[1]*q[3] + p[2]*q[0] + p[3]*q[1];
         r[3]= p[0]*q[3] + p[1]*q[2] - p[2]*q[1] + p[3]*q[0];
    }
    /////// QUATERNION rotation FUNCTION
    void Qrotate(float *v, float *q, float *out ){
    
         float v_1[4]= {0, v[0],v[1],v[2]};
         float temp[4]={0,0,0,0};
         Qprod(v_1,q,temp);
         float q_conj[4]={q[0],-q[1],-q[2],-q[3]};
         Qprod(q_conj,temp,v_1);
         out[0]= v_1[1];
         out[1]= v_1[2];
         out[2]= v_1[3];
    }
    /////// Sensor set_up function
    void sensor_setup(){
         i2c.frequency(400000);         
         /*Configure MPU9250*/
         data_write[0] = REG_MPU[4];
         data_write[1] = set_up_mpu[2];
         int status = i2c.write(MPU_ADDR, data_write, 2, false);         
         data_write[0] = REG_MPU[2];
         data_write[1] = set_up_mpu[0];
         status = i2c.write(MPU_ADDR, data_write, 2, false);
         data_write[0] = REG_MPU[5];
         data_write[1] = set_up_mpu[3];
         status = i2c.write(MPU_ADDR, data_write, 2, false);    
         data_write[0] = REG_MPU[3];
         data_write[1] = set_up_mpu[1];
         status = i2c.write(MPU_ADDR, data_write, 2, false);     
         /*Configure KMX62 Sensor*/
         // Ctrl Reg Set up
         data_write[0] = REG[3];
         data_write[1] = set_up[1];
         status = i2c.write(KMX62_ADDR, data_write, 2, false);
         if (status != 0) { // Error
             while (status != 0) {
                 PC.printf("Error 1");
                 myled = !myled;
                 wait(1);
                 status = i2c.write(KMX62_ADDR, data_write, 2, false);
             }
         }
         // ODC Reg Set up
         data_write[0] = REG[2];
         data_write[1] = set_up[0];
         status = i2c.write(KMX62_ADDR, data_write, 2, false);
         if (status != 0) { // Error
             while (status != 0) {
                 PC.printf("Error 2");
                 myled = !myled;
                 wait(1);
                 status = i2c.write(KMX62_ADDR, data_write, 2, false);
             }
         }
         // Buffer Reg Set up
         int i;
         for(i=0;i<3;i++) {
             data_write[0] = REG[4+i];
             data_write[1] = set_up[2+i];
             status = i2c.write(KMX62_ADDR, data_write, 2, false);
             if (status != 0) { // Error
                 while (status != 0) {
                     PC.printf("Error 3");
                     myled = !myled;
                     wait(1);
                     status = i2c.write(KMX62_ADDR, data_write, 2, false);
                 }
             }
         }        
    }
    
    /////// MPU sensor axes alignment and offset correction    
    void MPUrot_offset(int16_t *Data, char *rawData, int16_t *offset){
        int16_t Data_read[3];
        Data_read[1] = -(((int16_t)rawData[0] << 8 | rawData[1])-offset[0]) ;  // Turn the MSB and LSB into a signed 16-bit value
        Data_read[0] = (((int16_t)rawData[2] << 8 | rawData[3])-offset[1] );
        Data_read[2] = ((int16_t)rawData[4] << 8 | rawData[5])-offset[2] ;
        Data[0]=Data_read[0]; Data[1]=Data_read[1]; Data[2]=Data_read[2];
    }
    
    /////// Magnetic sensor offset correction
    void Mag_offset(int16_t *M_read, char *data, int16_t *offset){
        M_read[0] = (((int16_t)data[7]<<8)|data[6]) - offset[0];
        M_read[1] = (((int16_t)data[9]<<8)|data[8]) - offset[1];
        M_read[2] = (((int16_t)data[11]<<8)|data[10]) - offset[2];
    }
    /////// Magnetic sensor reading
    void Read_Mag(int16_t *M_read){
        data_write[0] = REG[0];
        i2c.write(KMX62_ADDR, data_write, 1, true); // no stop
        i2c.read(KMX62_ADDR, data_read, 12, false);
        Mag_offset(M_read, data_read, offset_mag);
        }
    /////// Acc sensor reading    
    void Read_Acc(int16_t *A_read){
        data_write[0] = REG_MPU[0];
        i2c.write(MPU_ADDR, data_write, 1, true); // no stop
        i2c.read(MPU_ADDR, rawData_a, 6 ,false);  // Read the six raw data registers into data array
        MPUrot_offset(A_read, rawData_a, offset_acc);
        }
    /////// Gyr sensor reading
    void Read_Gyr(int16_t *G_read){
        data_write[0] = REG_MPU[1];
        i2c.write(MPU_ADDR, data_write, 1, true); // no stop
        i2c.read(MPU_ADDR, rawData_g, 6 ,false);  // Read the six raw data registers into data array
        MPUrot_offset(G_read, rawData_g, offset_gyr);
        }    
    
    /////// Madgwick Quaternion update high-level function
    void QUpdate(int32_t *q, float *m, float *a, float *g, int16_t *Mag, int16_t *Acc, int16_t *Gyr, float *Q_init, int32_t *Q_out){
        float gRes = 0.0174533*0.0305175;//0.03*0.0175*2;  //1/62.5; radianti, non gradi 500=0.016
        float mRes = 0.000366;     // G, non microT
        float aRes = 0.00012207031;  //1/8192;
        float Q[4] = {0,0,0,0};
        float Q_o[4] = {0,0,0,0};
        //// scaling
        m[0] = ((float) Mag[0])*mRes;
        m[1] = ((float) Mag[1])*mRes;
        m[2] = ((float) Mag[2])*mRes;
        a[0] = ((float) Acc[0])*aRes;
        a[1] = ((float) Acc[1])*aRes;
        a[2] = ((float) Acc[2])*aRes;
        g[0] = ((float) Gyr[0])*gRes;
        g[1] = ((float) Gyr[1])*gRes;
        g[2] = ((float) Gyr[2])*gRes;
        //// quaternion calculation
        MadgwickAHRSupdate(g[0], g[1], g[2], a[0], a[1], a[2], m[0], m[1], m[2]);
        Q[0]=q0; Q[1]=q1; Q[2]=q2; Q[3]=q3;
        Qprod(Q, Q_init, Q_o);
        Q_out[0]=(int32_t)(Q_o[0]*(powf(2,30)));  
        Q_out[1]=(int32_t)(Q_o[1]*(powf(2,30)));
        Q_out[2]=(int32_t)(Q_o[2]*(powf(2,30)));
        Q_out[3]=(int32_t)(Q_o[3]*(powf(2,30)));                
        q[0]=(int32_t)(q0*(powf(2,30)));  
        q[1]=(int32_t)(q1*(powf(2,30)));
        q[2]=(int32_t)(q2*(powf(2,30)));
        q[3]=(int32_t)(q3*(powf(2,30)));
    }
    
    /////// Filtered magnetic signals envelope extraction    
    void Env_extraction(int16_t *P, int16_t *buffer_rot, int16_t *max, int16_t *p_i, double *R, double *ph, int16_t *S){
         int16_t i, k;         
         for(i=0; i<3; i++){
             max[2*i]=buffer_rot[i*8]; //maximum[i]=buffer_env[4+i*8];
             max[2*i+1]=buffer_rot[i*8];
             p_i[2*i]=0;
             p_i[2*i+1]=0;
    
             for(k=1;k<8;k++){
                 if ( buffer_rot[k + i*8] >= max[2*i] ){
                    max[2*i + 1] = max[2*i];
                    max[2*i] = buffer_rot[k + i*8];
                    p_i[2*i + 1]=p_i[2*i];
                    p_i[2*i]=k;
                 }                         
                 else {
                    if ( buffer_rot[k + i*8] >= max[2*i + 1] ){
                        max[2*i + 1] = buffer_rot[k + i*8];
                        p_i[2*i + 1]=k;
                    }  
                 }                        
             }        
             R[i]=((double)max[2*i])/((double)max[2*i + 1]);
             E[i]=(int16_t)(((double)max[2*i])*(E_c[0]*exp(E_c[1]*R[i])+E_c[2]*exp(E_c[3]*R[i])));         
             if ( (p_i[2*i] == 0) && (p_i[2*i + 1] == 7) ){
                 p_i[2*i + 1] = -1;
             }
             if ( (p_i[2*i] == 7) && (p_i[2*i + 1] == 0) ){
                 p_i[2*i + 1] = 8;
             }
             //Phase update: Y phase is updated only if its Envelope is higher than 5.5 uT : 150
             if (i == 1){
                 if(E[i]>=150){
                     ph[i]=((double)(p_i[2*i]+p_i[2*i + 1]))/2;             
                 }            
             }
             else{
                ph[i]=((double)(p_i[2*i]+p_i[2*i + 1]))/2;
             }                              
         }
         if (( fabs(ph[1]-ph[0]) >= 2 ) && ( fabs(ph[1]-ph[0]) <= 6)){
             S[0]=-1;
         }
         else{
             S[0]=1;
         }
         if (( fabs(ph[1]-ph[2]) >= 2 ) && ( fabs(ph[1]-ph[2]) <= 6)){
             S[1]=-1;
         }
         else{
             S[1]=1;
         }
         E_D = hypot( (double)E[0], (double)E[1] );
         Test_point[0]= (int16_t)E_D;
         Test_point[1]=S[1]*E[2];
         TRIAG = Find_point(Test_point, TRIAG, SI_Points, SI_ConnectivityList, inter_p);
         P[0]= S[0]*(int16_t)( (double)inter_p[0]*((double)E[0] / E_D));
         P[1]= (int16_t)( (double)inter_p[0]*((double)E[1] / E_D));         
         P[2]=inter_p[1];     
    }