﻿/*******************************************************************************
* Copyright (C) 2016 Maxim Integrated Products, Inc., All rights Reserved.
* 
* This software is protected by copyright laws of the United States and
* of foreign countries. This material may also be protected by patent laws
* and technology transfer regulations of the United States and of foreign
* countries. This software is furnished under a license agreement and/or a
* nondisclosure agreement and may only be used or reproduced in accordance
* with the terms of those agreements. Dissemination of this information to
* any party or parties not specified in the license agreement and/or
* nondisclosure agreement is expressly prohibited.
*
* The above copyright notice and this permission notice shall be included
* in all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
* OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
* IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
* OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
* ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
* OTHER DEALINGS IN THE SOFTWARE.
*
* Except as contained in this notice, the name of Maxim Integrated
* Products, Inc. shall not be used except as stated in the Maxim Integrated
* Products, Inc. Branding Policy.
*
* The mere transfer of this software does not imply any licenses
* of trade secrets, proprietary technology, copyrights, patents,
* trademarks, maskwork rights, or any other form of intellectual
* property whatsoever. Maxim Integrated Products, Inc. retains all
* ownership rights.
*******************************************************************************
*/

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace HealthSensorPlatform.Model
{
    /// <summary>
    /// Store and parse all the Pace burst read data
    /// </summary>
    public class PaceData 
    {
        private PaceRegister[] Group = new PaceRegister[6];

        /// <summary>
        /// Raw pace data from HSP firmware
        /// </summary>
        /// <param name="rawData"></param>
        public PaceData(int[] rawData)
        {
            if (rawData.Length == 18)
            {
                for (int i = 0; i < 6; i++)
                {
                    Group[i] = new PaceRegister(new int[] { rawData[3 * i + 0], rawData[3 * i + 1], rawData[3 * i + 2] });
                }
            }
            else
            {
                throw new InvalidOperationException("Pace register group expects array length of 18.");
            }
        }

        /// <summary>
        /// Calculate the time between first pace event and last event
        /// </summary>
        /// <param name="group"></param>
        /// <returns>Time count unit difference. To obtain the time value multiply the count by PACE_RES = 1/FPACE</returns>
        public int PaceLength(int group)
        {
            int start = 0, stop = 0;
            int sign = 1;

            start = Group[group].Edge[0].Data;
            sign = Group[group].Edge[0].Polarity == true ? 1 : -1;

            foreach (PaceEdge time in Group[group].Edge)
            {
                if (time.Last == true)
                {
                    stop = time.Data;
                    break;
                }
            }

            return sign * (stop - start);
        }

        public PaceRegister PaceGroup(int group)
        {
            return Group[group];
        }

        public string GroupToString(int group)
        {
            StringBuilder sb = new StringBuilder();

            foreach (PaceEdge pe in Group[group].Edge)
            {
                sb.Append(pe.Data);
                sb.Append(", ");
                sb.Append(pe.Polarity ? 'R' : 'F');
                sb.Append(", ");
                sb.Append(pe.Last ? 'Y' : 'N');
                sb.Append(", ");
            }

            return sb.ToString();
        }

        public static string GroupEmptyString()
        {
            return "0, R, Y, 0, R, Y, 0, R, Y, 0, R, Y, 0, R, Y, 0, R, Y";
        }


        ///// Inner classes of PaceData ////

        /// <summary>
        /// Pace edge information
        /// </summary>
        public class PaceEdge
        {
            public int Data;
            public bool Polarity;
            public bool Last;

            public PaceEdge(int data)
            {
                Data = data >> 2;
                Polarity = ((data & 0x02) >> 1) == 1 ? true : false;
                Last = ((data & 0x01) >> 0) == 1 ? true : false;
            }
        }

        /// <summary>
        /// Pace register information, each pace register contains up to 6 pace edge events
        /// </summary>
        public class PaceRegister
        {
            private PaceEdge[] Bytes = new PaceEdge[6];

            public PaceEdge[] Edge
            {
                get { return Bytes; }
            }

            public PaceRegister(int[] groupData)
            {
                if (groupData.Length == 3)
                {
                    for (int i = 0; i < 3; i++)
                    {
                        Bytes[2 * i] = new PaceEdge(groupData[i] >> 12);
                        Bytes[2 * i + 1] = new PaceEdge(groupData[i] & 0xFFF);
                    }
                }
            }
        }
    }
}
