#ifndef DECLARACIONES_H
#define DECLARACIONES_H
/*
 declaraciones.h
 PROGRAMA CONTROL DE SERVO DIRECCION BARCO
 Copyright (c) 2020 Pedro Campos <pedro@iotelectronic.com>
 All rights reserved.

  */


#include "mbed.h"
#include "mbed_toolchain.h"

#include "gps.h"

#define VERSION "V0.1"

//#define GPS_OUT

// TIPO DE TRAMAS ETHERNET
#define TRAMA_ESTADO_MOTOR_1    0x0001
#define TRAMA_ESTADO_MOTOR_2    0x0002
#define TRAMA_ESTADO_SERVO_DIR  0x0010  // Trama estado del servo de direccion


// TIPO DE TRAMAS COMUNICACION RS485
#define TIPO_TRAMA_ESTADO_RS485       0x10
#define TIPO_TRAMA_POSICION_RS485     0x11
#define TIPO_TRAMA_ESTADO_MANDO_RS485 0x15

#define NUL   0x00
#define STX   0x02
#define ETX   0x03
#define EOT   0x04
#define ENQ   0x05
#define ACK   0x06
#define LF    0x0A
#define CR    0x0D
#define NAK   0x15


#define TIMONES_HIST 2    // 2% histeresis timones

// CONTROL SERVO
#define TORQUE_MINIMO 15  // Depende de la resistencia e inercia del volante
#define TORQUE_MAXIMO 70  // Depende de la resistencia e inercia del volante
#define TORQUE_PILOTO 15  // Torque en piloto automático

#define DESVIACION_MAXIMA       450   // desviacion máxima mando timones
#define TORQUE_MAX_PILOTO_AUTO  300   // Torque maximo leido en piloto automatico

#define MASK_PULS_RUMBO     0x01
#define MASK_PULS_RADIO     0x02
#define MASK_PULS_BOCINA    0x04
#define MASK_PULS_AUX       0x08
#define MASK_PULS_MOT_BAB   0x10
#define MASK_PULS_MOT_EST   0x20
#define MASK_PULS_ANCLA_UP  0x40
#define MASK_PULS_ANCLA_DN  0x80


typedef uint8_t byte;


enum enEstadoDireccion { DIR_REPOSO=0, DIR_MASTER=1, DIR_SLAVE=2, DIR_RUMBO=3, DIR_PILOTO=4, DIR_ERROR=5, DIR_END_PILOT=6};
enum enEstadoServo { ESTADO_SERVO_OFF = 0, ESTADO_SERVO_ON = 1, ESTADO_SERVO_ERROR = 2 };



// ESTRUCTURA DE VARIABLES DE ESTADO DEL SERVO DE DIRECCION
struct __attribute__((packed)) st_datos_servo {       
  byte    n_servo;
  byte    estado;
  float   latitud;
  float   longitud;
  long    fecha_hora;
  float   velocidad;          // NUDOS
  float   rumbo;
  byte    ch_radio;
  byte    trim_timon_bab;
  byte    trim_trans_bab;
  byte    trim_timon_estr;
  byte    trim_trans_estr;
  byte    timon_babor;        // %de timon babor bajado, para 100% 5s, 1% = 50ms
  byte    timon_estribor;     // %de timon estribor bajado, para 100% 5s
  float   rumbo_fijado;       // Rumbo fijado al pulsar botón para RUMBO FIJO
  float   rumbo_piloto_auto;  // Rumbo fijado por piloto automático
  int     posicion_mando;     // Posicion actual del mando
};



/*
 * ESTRUCTURA COMUNICACIONES POR RS485
 */
struct __attribute__((packed)) st_trama_estado {    // ESTRUCTURA TRAMA DE ESTADO RECIBIDA DESDE SERVOS
  uint8_t bStart;    // start trama
  uint8_t nBytes;    // numero de bytes de la trama
  uint8_t tipo;      // tipo de trama
  uint8_t servo;     // Si es servo de babor = 0
  float HDOP;
  float latitud;
  float longitud;
  uint32_t timeStamp;
  float velocidad;
  float rumbo;
  uint8_t nSatelites;
  uint8_t estado;
  uint8_t timBabor;
  uint8_t timEstribor;
  uint16_t posicion;
  uint8_t trim_timon_bab;
  uint8_t trim_trans_bab;
  uint8_t trim_timon_estr;
  uint8_t trim_trans_estr;
  float rumbo_fijado;       // Rumbo fijado en RUMBO FIJO o PILOTO AUTOMATICO
  uint16_t ch_radio;
  uint16_t chk;      // checksum con XOR 0xffff desde b_start (inclusive) a chk
  uint8_t bStop;     // stop trama 
};


struct __attribute__((packed)) st_trama_posicion {    // ESTRUCTURA TRAMA DE POSICION RECIBIDA DESDE SERVO
  uint8_t bStart;    // start trama
  uint8_t nBytes;    // numero de bytes de la trama
  uint8_t tipo;      // tipo de trama
  uint8_t servo;     // Si es servo de babor = 0
  uint16_t posicion;
  uint8_t estado;
  uint16_t chk;   // checksum con XOR 0xffff desde b_start (inclusive) a chk
  uint8_t bStop;     // stop trama 
};


struct __attribute__((packed)) st_trama_estado_mando {    // ESTRUCTURA TRAMA DE POSICION ENVIADA DESDE MANDOS
  uint8_t bStart;    // start trama
  uint8_t nBytes;    // numero de bytes de la trama
  uint8_t tipo;      // tipo de trama
  uint8_t mando;     // Si es mando de babor = 0
  uint8_t estado;    // estado del mando
  uint8_t rudBab;    // trim del timon babor
  uint8_t rudEst;    // trim del timon estribor
  uint8_t traBab;    // trim de transmision babor
  uint8_t traEst;    // trim de transmision estribor
  uint8_t radioCH;   // canal radio
  uint8_t pulsado;   // estado pulsadores
  uint16_t chk;   // checksum con XOR 0xffff desde b_start (inclusive) a chk
  uint8_t bStop;     // stop trama 
};


// Time protocol implementation : Address: time.nist.gov UDPPort: 37   
typedef struct {
    uint32_t secs;         // Transmit Time-stamp seconds.
}ntp_packet;


const char str_estado[7][10] = { "CENTERED", "MASTER", "SLAVE", "HEADING", "AUT PILOT", "ERROR", "END PILOT"};

const int buffer_size = 255;     // Tamaño del buffer de rx y tx para RS485


void epoch_to_time(long epoch);
void grados(float pos, int res[]);  // devuelve grados

void procesa_servo(void);           // lee estado del servo, procesa torque y actualiza
void posicion_servo(void);          // AJUSTA POSICION SERVO SI ES NECESARIO
bool escribe_parametro_servo(byte tipo, byte numero, long valor);   //  * ENVIA PARAMETRO AL SERVO

uint16_t checksum(uint8_t *buf, uint16_t nb);   // devuelve checksum de nb bytes del buffer

void envia_posicion_mando(uint16_t posicion_mando);  // * ENVIA POSICION DEL MANDO
void envia_estado_mando(void);    // ENVIA ESTADO A LOS MANDOS

void  procesa_posicion_timones(void);   // Pocesa activación y posición timones, llamada cada 100ms

void onSerialRxRS485(void);     // Procesa interrupcion de RX en RS485
void onSerialTxRS485(void);     // Procesa interrupcion de TX en RS485
void PrintHex8(uint8_t *data, uint8_t length);  // prints 8-bit data in hex with leading zeroes

void SerialTxRS485(void);    // Procesa envio del buffer
void udp_main(void);        // Hilo de proceso de tramas Ethernet

void actualiza_timones(void);   // ACTUALIZA TIMONES

void actualizaFechaHoraGPS(float fecha, float hora);
void getTime(void);

void procesa_estado(void);      // PROCESA CAMBIO DE ESTADO
unsigned long millis(void);     // devuelve cuenta de tiempo en ms

void procesa_pulsadores(uint8_t pulsadores);    // Procesa segun estado de los pulsadores del mando
void procesa_piloto_automatico(void);           // procesa piloto automático

long constrain(long valor, long minimo, long maximo);   // * DEVUELVE VALOR DENTRO DEL LIMITE
long map(long valor, long minimo_in, long maximo_in, long minimo_out, long maximo_out);     // * EXTRAPOLA VALOR EN UN RANGO

void sincroniza_servos(void);            // SINCRONIZA LOS DOS SERVOS SEGUN SU ESTADO

uint16_t crc16(uint8_t * buf , uint16_t len);   // CALCULO DE CRC PARA TRAMA MODBUS

void test_modbus(void);                 // TEST PARA TRAMA MODBUS
void envia_trama_modbus(void);          // ENVIA TRAMA MODBUS RS485


#endif
