/*
// RUTINAS DE COMUNICACION CON EL SERVO
// Pedro Campos 23-04-2020
// 
*/
#include "declaraciones.h"

extern Serial pc;
extern RawSerial serSERVO;

extern DigitalOut SERVO_INC_POS;     // Salida de control INC_POS del servo (J1-2)
extern DigitalOut SERVO_DEC_POS;     // Salida de control DEC_POS del servo (J1-3)

extern Timer timer;

extern volatile byte estado;
extern enEstadoServo estado_servo;
extern int16_t velocidad_servo;
extern int16_t torque_servo;
extern long desviacion_servo;
extern struct st_datos_servo datos_servo;       // ESTRUCTURA DE VARIABLES DE ESTADO DEL SERVO DE DIRECCION
extern volatile int pos_actual_servo;
extern volatile int pos_objetivo_servo;
extern volatile unsigned long tempo_espera;
extern int out_torque;                          // nivel de torque de TORQUE_MINIMO a TORQUE_MAXIMO
extern int torque_max;                          // torque máximo calculado




/*******************************************************
 * RUTINAS COMUNICACION SERVO PANASONIC MINAS-A5
 *******************************************************/


void posicion_servo(void);                      // AJUSTA POSICION SERVO SI ES NECESARIO
unsigned char calc_chk(char * buf, int nb);     // CALCULA CHK PARA COMUNICACION CON SERVO
bool read_cmd_servo(unsigned char cmd, unsigned char *buf, byte nc);    // ENVIA COMANDO DE LECTURA A SERVO 1 y devuelve nc bytes en buf
bool comunica_servo(unsigned char *buf_in, byte nb_in, unsigned char *buf_out, byte nb_out);    // ENVIA DATOS A SERVO y devuelve la respuesta
bool lee_parametro_servo(byte tipo, byte numero, long *valor);          // LEE PARAMETRO DEL SERVO
bool escribe_parametro_servo(byte tipo, byte numero, long valor);       // ESCRIBE PARAMETRO EN EL SERVO
void flush_servo(void);                         //  * Espera que el canal serie este libre de RX y TX



/**************************************************
 * lee estado del servo, procesa torque y actualiza
 **************************************************/
void procesa_servo(void){
byte buf_entrada[20];

  // LEE ESTADO SERVO
  if(read_cmd_servo(0x92, buf_entrada, 13)){   // SI RESPONDE
    estado_servo = ESTADO_SERVO_ON;
    velocidad_servo = buf_entrada[4]*256+buf_entrada[3];
    torque_servo = buf_entrada[6]*256+buf_entrada[5];
    unsigned long dato;
    dato = (unsigned long)(buf_entrada[10])*256*256*256;
    dato += (unsigned long)(buf_entrada[9])*256*256;
    dato += (unsigned long)(buf_entrada[8])*256;
    dato += (unsigned long)(buf_entrada[7]);
    desviacion_servo = (long)dato;
 
//    pc.printf("estado:%d velocidad:%d torque:%d desviacion:%d\r\n", estado_servo, velocidad_servo, torque_servo, desviacion_servo); 
//  // PROCESA ESTADO SEGUN DESVIACION_SERVO
//    if(abs(desviacion_servo) <= 50){
//      if(datos_servo.estado==MASTER){
//        datos_servo.estado = REPOSO;
//      }
//    }
//    else if(datos_servo.estado==REPOSO){
//      datos_servo.estado=MASTER;
//    }

    // CALCULA TORQUE DE SALIDA  segun velocidad GPS o PILOTO AUTOMATICO

    if(datos_servo.estado == DIR_PILOTO || datos_servo.estado == DIR_RUMBO){
      out_torque = TORQUE_PILOTO;
    }
    else{
      torque_max = constrain((long)(datos_servo.velocidad*datos_servo.velocidad), 0, 12000);
      torque_max = map(torque_max, 0, 12000, TORQUE_MINIMO, TORQUE_MAXIMO);
      out_torque = map(desviacion_servo,0,DESVIACION_MAXIMA,TORQUE_MINIMO,torque_max);
    }
    escribe_parametro_servo(0, 13, out_torque);
//    printf("out_torque: %d, velocidad: %.2f\r\n", out_torque, datos_servo.velocidad);

  }
  else
    estado_servo = ESTADO_SERVO_OFF;

}



/**************************************************
 * AJUSTA POSICION SERVO SI ES NECESARIO
 **************************************************/
void posicion_servo(void)
{
    static int index = 0;
    static bool on = false;
    int n_pulsos = 0;

//    if((datos_servo.estado == DIR_RUMBO || datos_servo.estado == DIR_PILOTO) && (millis()%10)==0)    // Si ES PILOTO AUTOMATICO solo entra cada 10ms
//        return;

//    if(datos_servo.estado == DIR_RUMBO || datos_servo.estado == DIR_PILOTO)
//        n_pulsos = 1;
//    else
//        n_pulsos = 1+((abs(pos_objetivo_servo - pos_actual_servo))/100);

//    if(pos_actual_servo != pos_objetivo_servo)
        n_pulsos = 1;

    while(n_pulsos>0) {
        if(estado_servo == ESTADO_SERVO_ON) {  // Si el servo responde
            if(tempo_espera==0) {
                SERVO_INC_POS = 1;
                SERVO_DEC_POS = 1;
                tempo_espera++;
            } else {
                if(pos_actual_servo > pos_objetivo_servo) {
                    SERVO_DEC_POS = 0;
                    wait_us(10);
                    pos_actual_servo--;
                    tempo_espera = 0;
                } else if(pos_actual_servo < pos_objetivo_servo) {
                    SERVO_INC_POS = 0;
                    wait_us(10);
                    pos_actual_servo++;
                    tempo_espera = 0;
                }
            }
        }
        SERVO_INC_POS = 1;
        SERVO_DEC_POS = 1;
        wait_us(10);
        n_pulsos--;
    }

}



/**************************************************
 * CALCULA CHK PARA COMUNICACION CON SERVO
 **************************************************/
unsigned char calc_chk(unsigned char * buf, int nb){
  unsigned char chk = 0;

  for(int i=0; i<nb; i++){
    chk += buf[i];
  }
  chk ^= 0xff;
  chk += 1;
  return chk;
}



/**************************************************
 * ENVIA COMANDO DE LECTURA A SERVO 1 y devuelve nc bytes en buf
 **************************************************/
bool read_cmd_servo(unsigned char cmd, unsigned char *buf, byte nc){
unsigned char bcmd[4];

    bcmd[0]=0x00;               // N. de datos
    bcmd[1]=0x01;               // SERVO 1
    bcmd[2]=cmd;                // COMANDO
    bcmd[3]=calc_chk(bcmd, 3);  // CHK
    return(comunica_servo(bcmd, 4, buf, nc));
}



/**************************************************
 * Espera que el canal serie este libre de RX y TX
 **************************************************/
void flush_servo(void){

    while(serSERVO.readable()){
        char c = serSERVO.getc();
    }
}

    

/**************************************************
 * Espera respuesta del servo durante un tiempo en ms
 * devuelve false si sale por time-out
 **************************************************/
bool espera_resp_servo(int ms){

    timer.reset();
    while(!serSERVO.readable()){
        if(timer.read_ms()>=ms)
          return false;             // sale por timeout de ms
    }
    return true;
}



/**************************************************
 * Envia n bytes del buffer por canal serie
 **************************************************/
void envia_buffer_servo(byte *buf, int n){

    while(n-->0){
        serSERVO.putc(*buf++);
    }
}



/**************************************************
 * Recibe n bytes por canal serie al buffer, devuelve false si timeout
 **************************************************/
bool recibe_buffer_servo(byte *buf, int n, int ms = 100){

    while(n>0){
        if(espera_resp_servo(ms)==false)
            return false;       // sale por timeout
        *buf++ = serSERVO.getc();    
        n--;
    }
    return true;
    
}



/**************************************************
 * ENVIA DATOS A SERVO y devuelve la respuesta
 **************************************************/
bool comunica_servo(unsigned char *buf_in, byte nb_in, unsigned char *buf_out, byte nb_out){
char c;

    flush_servo();
    serSERVO.putc(ENQ);
//    pc.printf("\r\n");
//    pc.printf("envia ENQ\r\n");
    if(espera_resp_servo(100)==false){    // espera respuesta durante 100ms
        pc.printf("sin respuesta\r\n");
        return false;                     // sale por timeout de 100ms
    }
    c = serSERVO.getc(); 
    if(c != EOT){             // Si no es correcta la respuesta
        pc.printf("no EOT c=%02X\r\n", c);    
        return false;                     // sale por respuesta incorrecta
    }
    envia_buffer_servo(buf_in, nb_in);    // Envia mensaje de comando
//    pc.printf("envia buf_in\r\n");
    if(espera_resp_servo(100)==false){    // espera respuesta durante 100ms
        pc.printf("sin respuesta\r\n");
        return false;                     // sale por timeout de 100ms
    }
    c=serSERVO.getc();
    if(c != ACK){             // Si no es correcta la respuesta
        pc.printf("no ACK c=%02X\r\n", c);    
        return false;                     // sale por respuesta incorrecta
    }
    if(espera_resp_servo(100)==false){    // espera respuesta durante 100ms
        pc.printf("sin respuesta\r\n");
        return false;                     // sale por timeout de 100ms
    }
    c=serSERVO.getc();
    if(c!=ENQ){             // Si no es correcta la respuesta
        pc.printf("no ENQ c=%02X\r\n", c);    
        return false;                     // sale por respuesta incorrecta
    }
    serSERVO.putc(EOT);
//    pc.printf("envia EOT\r\n");
    if(recibe_buffer_servo(buf_out, nb_out)==false){       // Espera datos nb_out
        pc.printf("sin respuesta\r\n");
        return false;                       //timeout en respuesta
    }
    byte chk = calc_chk(buf_out, nb_out-1); // CHK
    if(chk == buf_out[nb_out-1]){           // Datos recibidos
        serSERVO.putc(ACK);                 // Envia ACK
//        pc.printf("CHK OK, envia ACK\r\n");
        return true;                        // Finaliza OK
    }
    serSERVO.putc(NAK);                 // Envia NAK, error en datos 
    pc.printf("CHK NO OK, envia NAK\r\n");
    return false;                       // Finaliza con error
}



/**************************************************
 * LEE PARAMETRO DEL SERVO
 **************************************************/
bool lee_parametro_servo(byte tipo, byte numero, long *valor){
unsigned char bcmd[12];
unsigned char bresp[12];
unsigned long dato;

    bcmd[0]=0x02;               // N. de datos
    bcmd[1]=0x01;               // SERVO 1
    bcmd[2]=0x07;               // COMANDO lectura de parametro
    bcmd[3]=tipo;               // tipo de parametro
    bcmd[4]=numero;             // número de parametro
    bcmd[5]=calc_chk(bcmd, 5);  // CHK
    if(comunica_servo(bcmd, 6, bresp, 9)){
      dato = (unsigned long)(bresp[6])*256*256*256;
      dato += (unsigned long)(bresp[5])*256*256;
      dato += (unsigned long)(bresp[4])*256;
      dato += (unsigned long)(bresp[3]);
      *valor = (long)dato;
      
      return true;
    }
    else
      return false;
}



/**************************************************
 * ENVIA PARAMETRO AL SERVO
 **************************************************/
bool escribe_parametro_servo(byte tipo, byte numero, long valor){
unsigned char bcmd[12];
unsigned char bresp[12];
unsigned long dato;

    bcmd[0]=0x06;               // N. de datos
    bcmd[1]=0x01;               // SERVO 1
    bcmd[2]=0x17;               // COMANDO escritura de parametro
    bcmd[3]=tipo;               // tipo de parametro
    bcmd[4]=numero;             // número de parametro

    dato = (unsigned long)valor;

    bcmd[5]=(unsigned char)(valor&0x000000ff);             // valor low de parametro
    bcmd[6]=(unsigned char)((valor>>8)&0x000000ff);        // valor de parametro
    bcmd[7]=(unsigned char)((valor>>16)&0x000000ff);       // valor de parametro
    bcmd[8]=(unsigned char)((valor>>24)&0x000000ff);       // valor high de parametro

    bcmd[9]=calc_chk(bcmd, 9);  // CHK
    if(comunica_servo(bcmd, 10, bresp, 5)){      
      return true;
    }
    else
      return false;
}




