// Software (bit-banged) SPI Module supporting two CPHA modes and two
// chip select signals.  Yields to allow use by different threads.
//

#ifndef sw_spi_h
#define sw_spi_h

#include "mbed.h"
#include <stdint.h>

// This is the bit in the SPI address that marks it as a write
#define SPI_WRITE_MASK 0x80


/////////////////////////////////////////////////////////////////////
/// \class swspi swspi.h <swspi.h>
/// \brief bit-banged SPI

class swspi
{
public:
    /// Constructor
    /// \param[in] mosiPin pin number for MOSI
    /// \param[in] misoPin pin number for MISO
    /// \param[in] sckPin pin number for SCK
    /// \param[in] ss1Pin pin number for Slave Select 1
    /// \param[in] ss2Pin pin number for Slave Select 2
    swspi(PinName mosiPin, PinName misoPin, PinName sckPin, PinName ss1Pin, PinName ss2Pin);

    /// Initialise the Driver transport hardware and software.
    /// Make sure the Driver is properly configured before calling init().
    void           init();

    /// Reads a single register from the SPI device
    /// \param[in] reg Register number
    /// \return The value of the register
    uint8_t        spiRead(uint8_t reg, int cpha, int ss);

    /// Writes a single byte to the SPI device
    /// \param[in] reg Register number
    /// \param[in] val The value to write
    /// \return Some devices return a status byte during the first data transfer. This byte is returned.
    ///  it may or may not be meaningfule depending on the the type of device being accessed.
    uint8_t           spiWrite(uint8_t reg, uint8_t val, int cpha, int ss);

    /// Reads a number of consecutive registers from the SPI device using burst read mode
    /// \param[in] reg Register number of the first register
    /// \param[in] dest Array to write the register values to. Must be at least len bytes
    /// \param[in] len Number of bytes to read
    /// \return Some devices return a status byte during the first data transfer. This byte is returned.
    ///  it may or may not be meaningfule depending on the the type of device being accessed.
    uint8_t           spiBurstRead(uint8_t reg, uint8_t* dest, uint8_t len, int cpha, int ss);

    /// Write a number of consecutive registers using burst write mode
    /// \param[in] reg Register number of the first register
    /// \param[in] src Array of new register values to write. Must be at least len bytes
    /// \param[in] len Number of bytes to write
    /// \return Some devices return a status byte during the first data transfer. This byte is returned.
    ///  it may or may not be meaningfule depending on the the type of device being accessed.
    uint8_t           spiBurstWrite(uint8_t reg, const uint8_t* src, uint8_t len, int cpha, int ss);

protected:

    void _setSlaveSelect(int v);
    uint8_t _transfer(uint8_t v);
    
    /// Pins
    DigitalOut* mosiP;
    DigitalIn* misoP;
    DigitalOut* sckP;
    DigitalOut* ss1P;
    DigitalOut* ss2P;
    
    /// The current transfer data phase
    int                 _cpha;
    
    /// The current transfer slave select
    uint8_t             _ss;
    
    // Access control
    Mutex               _spi_mutex;
};

#endif
