/*************************************************** 
  This is a library for the Adafruit PT100/P1000 RTD Sensor w/MAX31865

  Designed specifically to work with the Adafruit RTD Sensor
  ----> https://www.adafruit.com/products/3328

  This sensor uses SPI to communicate, 4 pins are required to  
  interface
  Adafruit invests time and resources providing this open source code, 
  please support Adafruit and open-source hardware by purchasing 
  products from Adafruit!

  Written by Limor Fried/Ladyada for Adafruit Industries.  
  BSD license, all text above must be included in any redistribution
  
  Modified for mbed project - Dan Julio - 5/2017
 ****************************************************/
#include "Adafruit_MAX31865.h"

Adafruit_MAX31865::Adafruit_MAX31865(swspi& spiO, int ssNum, PinName rdyPin)
  :
  ssN(ssNum),
  spi(spiO)
{
  debug = false;
  rdyP = new DigitalIn(rdyPin);
}

bool Adafruit_MAX31865::begin(max31865_numwires_t wires) {

  /*
  for (uint8_t i=0; i<8; i++) {
    readRegister8(i);
  }
  */

  setWires(wires);
  enableBias(false);
  autoConvert(false);
  clearFault();

  if (debug) {
    printf("config: 0x%2x\r\n", readRegister8(MAX31865_CONFIG_REG));
  }
  return true;
}


uint8_t Adafruit_MAX31865::readFault(void) {
  return readRegister8(MAX31865_FAULTSTAT_REG);
}

void Adafruit_MAX31865::clearFault(void) {
  uint8_t t = readRegister8(MAX31865_CONFIG_REG);
  t &= ~0x2C;
  t |= MAX31865_CONFIG_FAULTSTAT;
  writeRegister8(MAX31865_CONFIG_REG, t);
}

void Adafruit_MAX31865::enableBias(bool b) {
  uint8_t t = readRegister8(MAX31865_CONFIG_REG);
  if (b) {
    t |= MAX31865_CONFIG_BIAS;       // enable bias
  } else {
    t &= ~MAX31865_CONFIG_BIAS;       // disable bias
  }
  writeRegister8(MAX31865_CONFIG_REG, t);
}

void Adafruit_MAX31865::autoConvert(bool b) {
  uint8_t t = readRegister8(MAX31865_CONFIG_REG);
  if (b) {
    t |= MAX31865_CONFIG_MODEAUTO;       // enable autoconvert
  } else {
    t &= ~MAX31865_CONFIG_MODEAUTO;       // disable autoconvert
  }
  writeRegister8(MAX31865_CONFIG_REG, t);
}

void Adafruit_MAX31865::setWires(max31865_numwires_t wires ) {
  uint8_t t = readRegister8(MAX31865_CONFIG_REG);
  if (wires == MAX31865_3WIRE) {
    t |= MAX31865_CONFIG_3WIRE;
  } else {
    // 2 or 4 wire
    t &= ~MAX31865_CONFIG_3WIRE;
  }
  writeRegister8(MAX31865_CONFIG_REG, t);
}

float  Adafruit_MAX31865::temperature(float RTDnominal, float refResistor, uint16_t rtdVal) {
  // http://www.analog.com/media/en/technical-documentation/application-notes/AN709_0.pdf

  float Z1, Z2, Z3, Z4, Rt, temp;

  if (rtdVal == 0) {
    Rt = readRTD();
  } else {
    Rt = rtdVal;
  }
  Rt /= 32768;
  Rt *= refResistor;
  
  if (debug) {
    printf("Resistance: %1.4f\r\n", Rt);
  }

  Z1 = -RTD_A;
  Z2 = RTD_A * RTD_A - (4 * RTD_B);
  Z3 = (4 * RTD_B) / RTDnominal;
  Z4 = 2 * RTD_B;

  temp = Z2 + (Z3 * Rt);
  temp = (sqrt(temp) + Z1) / Z4;
  
  if (temp >= 0) return temp;

  // ugh.
  float rpoly = Rt;

  temp = -242.02f;
  temp += 2.2228f * rpoly;
  rpoly *= Rt;  // square
  temp += (float) 2.5859e-3 * rpoly;
  rpoly *= Rt;  // ^3
  temp -= (float) 4.8260e-6 * rpoly;
  rpoly *= Rt;  // ^4
  temp -= (float) 2.8183e-8 * rpoly;
  rpoly *= Rt;  // ^5
  temp += (float) 1.5243e-10 * rpoly;

  return temp;
}


bool Adafruit_MAX31865::isReady(uint16_t* rtdVal) {
    if (rdyP->read() == 0) {
        *rtdVal = readRegister16(MAX31865_RTDMSB_REG);
        
        // remove fault
        *rtdVal >>= 1;
        return true;
    } else {
        return false;
    }
}


uint16_t Adafruit_MAX31865::readRTD (void) {
  clearFault();
  enableBias(true);
  wait_ms(10);
  uint8_t t = readRegister8(MAX31865_CONFIG_REG);
  t |= MAX31865_CONFIG_1SHOT;      
  writeRegister8(MAX31865_CONFIG_REG, t);
  wait_ms(65);

  uint16_t rtd = readRegister16(MAX31865_RTDMSB_REG);

  // remove fault
  rtd >>= 1;

  return rtd;
}

/**********************************************/

uint8_t Adafruit_MAX31865::readRegister8(uint8_t addr) {
  return spi.spiRead(addr, 1, ssN);
}

uint16_t Adafruit_MAX31865::readRegister16(uint8_t addr) {
  uint8_t buffer[2];
  
  spi.spiBurstRead(addr, buffer, 2, 1, ssN);

  uint16_t ret = buffer[0];
  ret <<= 8;
  ret |=  buffer[1];
  
  return ret;
}


void Adafruit_MAX31865::readRegisterN(uint8_t addr, uint8_t buffer[], uint8_t n) {
  addr &= 0x7F; // make sure top bit is not set

  spi.spiBurstRead(addr, buffer, n, 1, ssN);

  if (debug) {
    printf("$0x%2x: ", addr);
    while (n--) {
        printf(" 0x%2x", *buffer++);
    }
    printf("\n");
  }
}


void Adafruit_MAX31865::writeRegister8(uint8_t addr, uint8_t data) {
  spi.spiWrite(addr, data, 1, ssN);

  if (debug) {
    printf("$0x%2x = 0x%2x\r\n", addr, data);
  }
}

