/* 
 *
 * PicoTCP Socket interface for mbed.
 * Copyright (C) 2013 TASS Belgium NV
 * 
 * Released under GPL v2
 *
 * Other licensing models might apply at the sole discretion of the copyright holders.
 *
 *
 * This software is based on the mbed.org EthernetInterface implementation:
 * Copyright (C) 2012 mbed.org, MIT License
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software
 * and associated documentation files (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge, publish, distribute,
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
 * BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
 
#ifndef ETHERNETINTERFACE_H_
#define ETHERNETINTERFACE_H_

#if !defined(TARGET_LPC1768)
#error The Ethernet Interface library is supported only on the mbed NXP LPC1768
#endif

#include "rtos.h"

#ifdef __cplusplus
extern "C" {
#endif
    extern void (*linkCb)(uint32_t link);
#ifdef __cplusplus    
}
#endif

typedef enum
{
    INPUT_FILTER,
    OUTPUT_FILTER
}filter_type;

#define PORT_ANY    (0)
 /** Interface using Ethernet to connect to an IP-based network
 *
 */
class EthernetInterface {
public:
  /** Initialize the interface with DHCP.
  * Initialize the interface and configure it to use DHCP (no connection at this point).
  * \return 0 on success, a negative number on failure
  */
  static int init(); //With DHCP

  /** Initialize the interface with a static IP address.
  * Initialize the interface and configure it with the following static configuration (no connection at this point).
  * \param ip the IP address to use
  * \param mask the IP address mask
  * \param gateway the gateway to use
  * \return 0 on success, a negative number on failure
  */
  static int init(const char* ip, const char* mask, const char* gateway);

  /** Connect
  * Bring the interface up, start DHCP if needed.
  * \param   timeout_ms  timeout in ms (default: (10)s).
  * \return 0 on success, a negative number on failure
  */
  static int connect(unsigned int timeout_ms=15000);
  
  /** Disconnect
  * Bring the interface down
  * \return 0 on success, a negative number on failure
  */
  static int disconnect();
  
  /** Get the MAC address of your Ethernet interface
   * \return a pointer to a string containing the MAC address
   */
  static char* getMACAddress();
  
  /** Get the IP address of your Ethernet interface
   * \return a pointer to a string containing the IP address
   */
  static char* getIPAddress();
  static char* getIPGateway();
  static char* getIPNetmask();
  
  /** Register a callback to tell the status of the link.
   * \return 0 if callback was registered.
   */
  static int registerLinkStatus(void (*cb)(uint32_t linkStatus));
  
  
  /** Set/Add an ip address of a DNS server
   * \return 0 on success, a negative number on failure
   */
  static int setDnsServer(const char *ip);
  
  /**
  * Creates a custom rule for dropping packets.
  * If an incoming packet is destined for our stack and an input filter that matches
  * input_src & filter_mask == filter_src & filter_mask and the ports are the same, port = 0 means any port and also filters icmp.
  * the packet will be dropped.
  * The same thing for an output filter.
  * If we try to send a packet and 
  * output_dst & filter_mask == filter_dst & filter_mask and the ports are the same, 
  * the packet will be dropped.
  * return > 0 if success, the number is the filter identifier and it will be used later to destroy the filter.
  * return < 0 error.
  */
  static int createIpFilter(char *ipAddress, char * netmask, int port, filter_type filter);
  
  /*
  * This will create a custom filter, more complex than the previous one.
  * A packet will be dropped if 
  * packet.port(s) == filter.port(s)
  * packet.src & filter.src_mask == filter.src & filter.src_mask
  * packet.dst & filter.dst_mask == filter.dst & filter.dst_mask
  * if a value is passed null or 0.0.0.0 for address or 0 for port, will be considered as 
  * accept any.
  * return > 0 if success, the number is the filter identifier and it will be used later to destroy the filter.
  * return < 0 error.
  */
  
  static int createIpFilter(char *src, char * src_netmask, int src_port, char *dst, char * dst_netmask, int dst_port);
  /*
  * Destroy a custom filter.
  */
  static int destroyIpFilter(int filter_id);
  
};

#include "TCPSocketConnection.h"
#include "TCPSocketServer.h"

#include "Endpoint.h"
#include "UDPSocket.h"

#define htonl   long_be
#define ntohl   long_be

#endif /* ETHERNETINTERFACE_H_ */
