/** Car stereo control using 
 *  TDA7419 for input select, audio control.
 *  Adafruit 96x64 monochrome 0.96" display to show the audio selections
 *  four buttons to control selections.
 *
 *  Next steps:  take input from the car steering wheel controls!
 *
 * @Author: Dan Cohen
 *
 */



#include "mbed.h"
#include "PinDetect.h"
#include "Adafruit_SSD1306.h"
#include "PreampTDA7419.h"

#define NUM_OPTIONS 12 // how many different options does the stereo have?
char option;                               

PinDetect  PinUp    ( PTA1  );
PinDetect  PinLeft  ( PTD4  );
PinDetect  PinRight ( PTA2  );
PinDetect  PinDown  ( PTA12 );

class SPI2 : public SPI
{
public:
    SPI2(PinName mosi, PinName miso, PinName clk) : SPI(mosi,miso,clk)
    {
        format(8,3);
        frequency(2000000);
    };
};

SPI2 gSpi(PTD2,NC,PTD1);
Adafruit_SSD1306 display (gSpi, PTD0, PTD5, PTA13);  // SPI obect, DC, RST, CS
PreampTDA7419    Preamp  (PTE0, PTE1);

/////////////////////////////////////////////
// Helper functions for the serial display // 
/////////////////////////////////////////////
void testfillrect(void) {
  uint8_t color = 1;
  for (int16_t i=0; i<display.height()/2; i+=3) {
    // alternate colors
    wait(0.05);
    display.fillRect(i, i, display.width()-i*2, display.height()-i*2, color%2);
    display.display();
    color++;
  }
}

void displayWrite(char firstLine[], int value)
{
  display.clearDisplay();
  display.setCursor(0,0);
  display.setTextSize(2);
  display.printf("%s\r\n", firstLine);
  display.setCursor(0,30);
  display.setTextSize(5);
  display.printf("  %d\r\n", value);
  display.display();    
}

void displayWrite(char firstLine[], char secondLine[])
{
  display.clearDisplay();
  display.setCursor(0,0);
  display.setTextSize(2);
  display.printf("%s\r\n", firstLine);
  display.printf("%s\r\n", secondLine);  
  display.display();    
}

void processButtonPress (int button) {

  if (button == 0) {
    if (option < (NUM_OPTIONS-1)) {
      option++;
    } else {
      option = 0;
    }
  }

  if (button == 1) {
    if (option > 0) {
      option--;
    } else {
      option = (NUM_OPTIONS-1);
    }
  }

  switch (option) {
  case (0):  // if volume option is selected change volume when button 1 and 2 are pressed
    if (button == 2) {
      Preamp.decreaseVolume();
    }
    if (button == 3) {
      Preamp.increaseVolume();
    }
    displayWrite("Volume",  Preamp.readVolume() );
  break;
  case (1):  // manage the input - 1,2,3 are the standard single ended inputs
      int input = Preamp.readInput();
      if (button == 2) {
        input--;
      }
      if (button == 3) {
        input++;
      }
      Preamp.setInput(input);
      displayWrite("Input",  Preamp.readInput() );
  break;
  case (2):  // manage the treble value
    if (button == 2) {
      Preamp.decreaseTreble();
    }
    if (button == 3) {
      Preamp.increaseTreble();
    }
    displayWrite("Treble", Preamp.readTreble());
  break;
  case (3):  // manage the treble value
    if (button == 2) {
      Preamp.decreaseMiddle();
    }
    if (button == 3) {
      Preamp.increaseMiddle();
    }
    displayWrite("Middle", Preamp.readMiddle());
  break;
  case (4):  // manage the treble value
    if (button == 2) {
      Preamp.decreaseBass();
    }
    if (button == 3) {
      Preamp.increaseBass();
    }
    displayWrite("Bass", Preamp.readBass());
  break;

  // Manage the attenuators
  case (5):  // manage the atten_lf value
    if (button == 2) {
        displayWrite("LF",   Preamp.decreaseSpeaker(1) );
    } 
    else if (button == 3) {
        displayWrite("LF",   Preamp.increaseSpeaker(1) );
    } 
    else {
        displayWrite("LF",   Preamp.readSpeaker(1) );
    }
  break;
  case (6):  // manage the atten_rf value
    if (button == 2) {
        displayWrite("RF",   Preamp.decreaseSpeaker(2) );
    } 
    else if (button == 3) {
        displayWrite("RF",   Preamp.increaseSpeaker(2) );
    }
    else {
        displayWrite("RF",   Preamp.readSpeaker(2) );
    }   
  break;
  case (7):  // manage the atten_lr value
    if (button == 2) {
        displayWrite("LR",   Preamp.decreaseSpeaker(3) );
    }
    else if (button == 3) {
        displayWrite("LR",   Preamp.increaseSpeaker(3) );
    } 
    else {
        displayWrite("LR",   Preamp.readSpeaker(3) );
    }           
  break;
  case (8):  // manage the atten_rr value
    if (button == 2) {
        displayWrite("RR",   Preamp.decreaseSpeaker(4) );
    } 
    else if (button == 3) {
        displayWrite("RR",   Preamp.increaseSpeaker(4) );
    }
    else {
        displayWrite("RR",   Preamp.readSpeaker(4) );
    }     
  break;
  case (9):  // manage the atten_sub value
    if (button == 2) {
        displayWrite("SUB",  Preamp.decreaseSpeaker(5) );
    }
    else if (button == 3) {
        displayWrite("SUB",  Preamp.increaseSpeaker(5) );
    }
    else {
        displayWrite("SUB",   Preamp.readSpeaker(5) );
    }      
  break;
  case (10):  // manage the atten_mix value
    if (button == 2) {
        displayWrite("Mix Level",  Preamp.decreaseSpeaker(6) );
    }
    else if (button == 3) {
        displayWrite("Mix Level",  Preamp.increaseSpeaker(6) );
    }
    else {
        displayWrite("Mix Level",   Preamp.readSpeaker(6) );
    }         
  break;
  case (11):  // manage the atten_sub value
    if ( (button == 2) || (button == 3) ) {
      Preamp.toggleMix();
    } 
    if (Preamp.readMix()) {
      displayWrite("Mix",  "Enabled");
    }
    else {
      displayWrite("Mix", "Disabled");
    }
  break;  
  
  }
}

void UpPressed( void )
{
    processButtonPress(0);
}

void LeftPressed( void )
{
    processButtonPress(2);
}

void RightPressed( void )
{
    processButtonPress(3);
}

void DownPressed( void )
{
    processButtonPress(1);
}

int main()
{
    PinUp   .mode( PullUp );
    PinLeft .mode( PullUp );
    PinRight.mode( PullUp );
    PinDown .mode( PullUp );

    PinUp   .attach_asserted( &UpPressed    );
    PinLeft .attach_asserted( &LeftPressed  );
    PinRight.attach_asserted( &RightPressed );
    PinDown .attach_asserted( &DownPressed  );

    display.clearDisplay();
    // draw multiple rectangles
    testfillrect();
    wait(0.3);
    display.display();
    display.clearDisplay();
    display.display();
  

    // Sampling does not begin until you set a frequency.
    // The default is 20ms. If you want a different frequency
    // then pass the period in microseconds for example, for 10ms :-
    //     pin.setSampleFrequency( 10000 );
    //
    PinUp   .setSampleFrequency(10000); // Defaults to 20ms.
    PinLeft .setSampleFrequency(10000); // Defaults to 20ms.
    PinRight.setSampleFrequency(10000); // Defaults to 20ms.
    PinDown .setSampleFrequency(10000); // Defaults to 20ms.

    // option = NUM_OPTIONS;
    processButtonPress(0);

    while (1) {
        wait(0.2);
    }

}

// EOF