/*******************************************************************************
 * Copyright (C) 2018 Maxim Integrated Products, Inc., All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
 * OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 *
 * Except as contained in this notice, the name of Maxim Integrated
 * Products, Inc. shall not be used except as stated in the Maxim Integrated
 * Products, Inc. Branding Policy.
 *
 * The mere transfer of this software does not imply any licenses
 * of trade secrets, proprietary technology, copyrights, patents,
 * trademarks, maskwork rights, or any other form of intellectual
 * property whatsoever. Maxim Integrated Products, Inc. retains all
 * ownership rights.
 *******************************************************************************
 */ 
#ifndef _max8971_H_
#define _max8971_H_
 
#include "mbed.h"

class MAX8971
{
 
public:
    /**
     * @brief   Register Addresses
     * @details Enumerated max8971 register addresses
     */
    typedef enum {
        REG_CHGINT_INT = 0x0F,
        REG_CHGINT_MASK = 0x01,
        REG_CHG_STAT,
        REG_DETAILS1,
        REG_DETAILS2,
        REG_CHGCNTL1,
        REG_FCHGCRNT,
        REG_DCCRNT,
        REG_TOPOFF,
        REG_TEMPREG,
        REG_PROTCMD        
    } registers_t;
   
    /**
     * @brief   Interrupt Unmasked/Masked
     * @details Enumerated Interrupt Unmasked/Masked
     */
    typedef enum {
        VAL_INT_UNMASKED = 0x00,
        VAL_INT_MASKED
    } int_mask_t;
 
    /**
     * @brief   Interrupt Unmasked/Masked
     * @details Enumerated Interrupt Unmasked/Masked
     */
    typedef enum {
        VAL_AICL_I_BIT  = 7,
        VAL_TOP_OFF_BIT = 6,
        VAL_DC_OVP_BIT  = 5,
        VAL_DC_UVP_BIT  = 4,
        VAL_CHG_I_BIT   = 3,
        VAL_BAT_I_BIT   = 2,
        VAL_THM_I_BIT   = 1,
        VAL_POWERUP_BIT = 0,
    } int_bit_t;
    
    /**
     * @brief   ENABLE/DISABLE
     * @details Enumerated ENABLE/DISABLE
     */
    typedef enum {
        VAL_DISABLE = 0x00,
        VAL_ENABLE
    } enable_t;
    
    /**
     * @brief   Fast Charge Timer 
     * @details Enumerated Fast Charge Timer Duration
     */
    typedef enum {
        VAL_FC_TIMER_DISABLE = 0x00,
        VAL_FC_TIMER_4_HOURS,
        VAL_FC_TIMER_5_HOURS,
        VAL_FC_TIMER_6_HOURS,
        VAL_FC_TIMER_7_HOURS,
        VAL_FC_TIMER_8_HOURS,
        VAL_FC_TIMER_9_HOURS,
        VAL_FC_TIMER_10_HOURS
    } fast_charge_timer_t;
  
    /**
     * @brief   Fast Charge Restart Threshold 
     * @details Enumerated Fast Charge Restart Threshold
     */
    typedef enum {
        VAL_FC_RESTART_MINUS_150mV = 0x00,
        VAL_FC_RESTART_MINUS_100mV,
    } fast_charge_restart_threshold_t;
  
    /**
     * @brief   Topoff timer 
     * @details Enumerated Topoff Timer 
     */
    typedef enum {
        VAL_TOPOFF_TIMER_0_MIN = 0x00,
        VAL_TOPOFF_TIMER_10_MIN,
        VAL_TOPOFF_TIMER_20_MIN,
        VAL_TOPOFF_TIMER_30_MIN,
        VAL_TOPOFF_TIMER_40_MIN,
        VAL_TOPOFF_TIMER_50_MIN,
        VAL_TOPOFF_TIMER_60_MIN,
        VAL_TOPOFF_TIMER_70_MIN
    } top_off_timer_t;      
        
    /**
     * @brief   Topoff Current Threshold
     * @details Enumerated Topoff Current Threshold
     */
    typedef enum {
        VAL_TOPOFF_CURRENT_TH_50_mA = 0x00,
        VAL_TOPOFF_CURRENT_TH_100_mA,
        VAL_TOPOFF_CURRENT_TH_150_mA,
        VAL_TOPOFF_CURRENT_TH_200_mA
    } top_off_current_threshold_t;      
        
    /**
     * @brief   Charge Termination Voltage
     * @details Enumerated Charge Termination Voltage
     */
    typedef enum {
        VAL_CHARGE_TERM_4200_mV = 0x00,
        VAL_CHARGE_TERM_4100_mV,
        VAL_CHARGE_TERM_4350_mV,
        VAL_CHARGE_TERM_4150_mV
    } charger_termination_voltage_t;      
        
    /**
     * @brief   Die Temperature Thermal Regualation Loop Set Point
     * @details Enumerated Die Temperature Thermal Regualation Loop Set Point
     */
    typedef enum {
        VAL_DIE_TEMP_REG_POINT_105_DEG = 0x00,
        VAL_DIE_TEMP_REG_POINT_90_DEG,
        VAL_DIE_TEMP_REG_POINT_120_DEG,
        VAL_DIE_TEMP_REG_POINT_DISABLE,
    } die_temp_reggulation_point_t; 
                
    /**
     * @brief   Thermistor Monitor Configuration
     * @details Enumerated Thermistor Monitor Configuration
     */
    typedef enum {
        VAL_THERMISTOR_MONITOR_ENABLE = 0x00,
        VAL_THERMISTOR_MONITOR_DISABLE,
    } thermistor_monitor_config_t; 
                
    /**
     * @brief   JEITA Safety
     * @details Enumerated JEITA Safety Region Selection
     */
    typedef enum {
        VAL_JEITA_SAFETY_REGION_1 = 0x00,
        VAL_JEITA_SAFETY_REGION_2,
    } jeita_safety_selection_t; 
                  
    /**
     * @brief   Charger Setting Protection
     * @details Enumerated Charger Setting Protection
     */
    typedef enum {
        VAL_CHARGER_SETTING_LOCKED = 0x00,
        VAL_CHARGER_SETTING_UNLOCKED = 0x03
    } charger_setting_protection_t; 
        
        
    /**
      * max8971 constructor.
      *
      * @param i2c I2C object to use.
      */
    MAX8971(I2C *i2c);
 
    /**
      * max8971 destructor.
      */
    ~MAX8971();
  
    /**
      * @brief   Initialize max8971
      */
    int32_t init();
 
    /**
      * @brief   Write Register
      * @details Writes data to max8971 register
      *
      * @param   reg_addr Register to write
      * @param   reg_data Data to write
      * @returns 0 if no errors, -1 if error.
      */
    int32_t write_register(MAX8971::registers_t reg_addr, char reg_data);
 
    /**
      * @brief   Read Register
      * @details Reads data from max8971 register
      *
      * @param   reg_addr Register to read
      * @returns data if no errors, -1 if error.
      */
    int32_t read_register(MAX8971::registers_t reg_addr);
    
    /**
      * @brief   Update Register data
      * @details Update bits data of a register 
      *
      * @param   reg_no Register Number to be updated
      * @param   mask Mask Data
      * @param   reg_data bit data
      * @returns 0 if no errors, -1 if error.
      */
    int32_t update_register
    (MAX8971::registers_t reg_no, char reg_mask,  char reg_data);
       

    /**
      * @brief   Get Interrupt
      * @details Get status register data
      *          BIT7 : AICL_I DC Interrupt
      *          BIT6 : TOPOFF Topoff Interrupt
      *          BIT5 : DC Overvolatage Interrupt
      *          BIT4 : DC Undervoltage Interrupt
      *          BIT3 : Charge Current Interrupt
      *          BIT2 : Battery Interrupt
      *          BIT1 : Thermistor Interrupt
      *          BIT0 : Power-Up OK Interrupt
      * @param   None
      * @returns Interrupt register data.
      */
    int32_t get_interrupt();

    /**
      * @brief   Set Interrupt Mask
      * @details Get status register data
      *          BIT7 : AICL_I DC Interrupt
      *          BIT6 : TOPOFF Topoff Interrupt
      *          BIT5 : DC Overvolatage Interrupt
      *          BIT4 : DC Undervoltage Interrupt
      *          BIT3 : Charge Current Interrupt
      *          BIT2 : Battery Interrupt
      *          BIT1 : Thermistor Interrupt
      *          BIT0 : Reserved
      * @param   Register Number, Interrupt Bit
      * @returns Interrupt register data.
      */
    int32_t set_interrupt_mask
        (MAX8971::registers_t reg_no, MAX8971::int_bit_t interrupt_bit);


    /**
      * @brief   Set Interrupt UnMask
      * @details Get status register data
      *          BIT7 : AICL_I DC Interrupt
      *          BIT6 : TOPOFF Topoff Interrupt
      *          BIT5 : DC Overvolatage Interrupt
      *          BIT4 : DC Undervoltage Interrupt
      *          BIT3 : Charge Current Interrupt
      *          BIT2 : Battery Interrupt
      *          BIT1 : Thermistor Interrupt
      *          BIT0 : Reserved
      * @param   Register Number, Interrupt Bit
      * @returns Interrupt register data.
      */
    int32_t set_interrupt_unmask
        (MAX8971::registers_t reg_no, MAX8971::int_bit_t interrupt_bit);

    /**
      * @brief   Get status
      * @details Get status register data
      *          BIT7 : DCV_OK DC Input Voltage Status
      *          BIT6 : DCI_OK DC Input Current Status
      *          BIT5 : DCOVP_OK DC OVP Status
      *          BIT4 : DCUVP_OK DC UVP Status
      *          BIT3 : CHG_OK Charger Status
      *          BIT2 : BAT_OK Battery Status
      *          BIT1 : THM_OK Thermistor Status
      *          BIT0 : RESERVED
      * @param   None
      * @returns status register data.
      */
    int32_t get_status();

    /**
      * @brief   Get Details 1
      * @details Get Details 1 register data
      *          BIT7 : DC Details
      *          BIT6 : DC Current
      *          BIT5 : DC OVP
      *          BIT4 : DC UVP
      *          BIT3 : RESERVED
      *          BIT2-0 : Thermistor Details
      * @param   None
      * @returns status register data.
      */
    int32_t get_details1();


    /**
      * @brief   Get Details 2
      * @details Get Details 2 register data
      *          BIT7-6 : Reserved
      *          BIT5-4 : Battery Details
      *          BIT3-0 : Charger Details
      * @param   None
      * @returns status details 2 register data.
      */
    int32_t get_details2();

    /**
      * @brief   Set DCMON_DIS
      * @details Set to disable the monitoring of input voltage
      *          by the input power limiter
      * @param   None
      * @returns DCMON_DIS bit data.
      */
    int32_t set_disable_dc_monitor();

    /**
      * @brief   UnSet DCMON_DIS
      * @details UnSet to enable the monitoring of input voltage
      *          by the input power limiter
      * @param   None
      * @returns DCMON_DIS bit data.
      */
    int32_t unset_disable_dc_monitor();
/**
  * @brief   Set USB SUSPEND
  * @details Set to USB Suspend bit 
  * @param   None
  * @returns USB SUSPEND bit data.
  */
int32_t set_usb_suspend();

/**
  * @brief   UnSet USB SUSPEND
  * @details UnSet to USB Suspend bit 
  * @param   None
  * @returns USB SUSPEND bit data.
  */
int32_t unset_usb_suspend();

/**
  * @brief   Set Fast Charge Current
  * @details Set to CHGCC bit to control Fast Charge Current
  * @param   int 0-1550
  * @returns Fast Charge Current.
  */
int32_t set_fast_charge_current(int current);

/**
  * @brief   Set Fast Charge Timer Duration
  * @details Set to FCHGT bit to control Fast Charge Timer
  * @param   fast_charge_timer_t
  * @returns Fast Charge Timer Bit Data.
  */
int32_t set_fast_charge_timer(MAX8971::fast_charge_timer_t fc_time);

/**
  * @brief   Set Input Current Limit
  * @details Set to DCILMT to control Input Current Limit
  *          0x00 - 0x09 : 100mA
  *          0x10 - 1x3C : 250 mA - 1500 mA with 25m A
  * @param   int 0-1500
  * @returns DCILMT bit Data.
  */
  
/**
  * @brief   Set Fast Restart Threshold
  * @details Set to CHGRSTRT bit to control Fast Restart Threshold
  * @param   fast_charge_restart_threshold_t
  * @returns CHGRSTRT bit Data.
  */
int32_t set_fast_charge_restart_threshold
(MAX8971::fast_charge_restart_threshold_t voltage);

int32_t set_input_current_limit(int limit);

/**
  * @brief   Set Topoff Timer
  * @details Set to TOFFT bit to control Topoff Timer
  * @param   top_off_timer_t
  * @returns TOFFT bit Data.
  */
int32_t set_top_off_timer(MAX8971::top_off_timer_t time);

/**
  * @brief   Set Topoff Current Threshold
  * @details Set to TOFFS bit to control Topoff Current Threshold
  * @param   top_off_current_threshold_t
  * @returns TOFFS bit Data.
  */
int32_t set_top_off_current_threshold
(MAX8971::top_off_current_threshold_t current);

/**
  * @brief   Set Charge Termination Voltage
  * @details Set to CHGCV bit to control Charge Termination Voltage in CV Mode
  * @param   charger_termination_voltage_t
  * @returns CHGCV bit Data.
  */
int32_t set_charge_termination_voltage
(MAX8971::charger_termination_voltage_t voltage);

/**
  * @brief   Set Die-Temperature Regulation Loop Set Point
  * @details Set REGTEMP bit to control 
  *          Die-Temperature Regulation Loop Set Point
  * @param   die_temp_reggulation_point_t
  * @returns REGTEMP bit Data.
  */
int32_t set_charge_termination_voltage
(MAX8971::die_temp_reggulation_point_t degree);

/**
  * @brief   Set Thermistor Monitor Configuration
  * @details Set THM_CNFG bit to control the Thermistor Monitor Configuration
  * @param   thermistor_monitor_config_t
  * @returns REGTEMP bit Data.
  */
int32_t set_thermistor_monitor
(MAX8971::thermistor_monitor_config_t enable);

/**
  * @brief   Set JEITA Safty Region
  * @details Set SAFETYREG bit to control the JEITA Safty Region
  * @param   jeita_safety_selection_t
  * @returns SAFETYREG bit Data.
  */
int32_t set_jeita_safety_region
(MAX8971::jeita_safety_selection_t enable);

/**
  * @brief   Set Charger-Setting Protection
  * @details Set CPROT bit to control Charger-Setting Protection
  * @param   charger_setting_protection_t
  * @returns CPROT bit Data.
  */
int32_t set_charger_setting_protection
(MAX8971::charger_setting_protection_t enable);


private:
 
    I2C *i2c_;
    bool i2c_owner;
 
};
#endif /* _max8971_H_ */