/*******************************************************************************
 * Copyright (C) 2019 Maxim Integrated Products, Inc., All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
 * OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 *
 * Except as contained in this notice, the name of Maxim Integrated
 * Products, Inc. shall not be used except as stated in the Maxim Integrated
 * Products, Inc. Branding Policy.
 *
 * The mere transfer of this software does not imply any licenses
 * of trade secrets, proprietary technology, copyrights, patents,
 * trademarks, maskwork rights, or any other form of intellectual
 * property whatsoever. Maxim Integrated Products, Inc. retains all
 * ownership rights.
 *******************************************************************************
 */ 
#ifndef _MAX77950_H_
#define _MAX77950_H_
 
#include "mbed.h"

class MAX77950
{
 
public:

    /**
     * @brief   Register Addresses
     * @details Enumerated MAX77950 register addresses
     */
    typedef enum {
        REG_CHIP_ID = 0x00, 
        REG_OTP_REV = 0x05,
        REG_STATUS_L,
        REG_STATUS_H,
        REG_INT_L,
        REG_INT_H,
        REG_INT_ENABLE_L,
        REG_INT_ENABLE_H,
        REG_INT_CLEAR_L,
        REG_INT_CLEAR_H,
        REG_CHARGE_STATUS,
        REG_EPT_REASON,
        
        REG_VOUTVAL_H = 0x10,
        REG_VOUTVAL_L,
        REG_VOUTSET,
        REG_VRECT_ADJ,
        REG_VRECTVAL_H,
        REG_VRECTVAL_L,
        REG_ISENSEVAL_H,
        REG_ISENSEVAL_L,
        REG_TDIE_VALUE,
        REG_OP_FREQ_L,
        REG_OP_FREQ_H,
        REG_PING_OP_FREQ_L,
        REG_PING_OP_FREQ_H,
        REG_LDO_ILIMSET,
        REG_TX_ILIMSET,
        REG_SYS_OP_MODE,
        REG_RX_COM = 0x20,
        REG_PPP_HEADER,
        REG_RX_DATA_VALUE0, 
        REG_RX_DATA_VALUE1,
        REG_RX_DATA_VALUE2,
        REG_RX_DATA_VALUE3,
        REG_RX_DATA_VALUE4,
        REG_FSK_DATA_VALUE0,
        REG_FSK_DATA_VALUE1,
        REG_FSK_DATA_VALUE2,
        
        REG_TX_FOP_SET_L,
        REG_TX_FOP_SET_H,
        REG_TX_FOP_TON_SET_L,
        REG_TX_FOP_TON_SET_H,
        REG_TX_WPC_HEADER = 0x34,
        REG_TX_WPC_DATA0,
        REG_TX_WPC_DATA1,
        REG_TX_WPC_DATA2,
        REG_TX_WPC_DATA3,
        REG_TX_WPC_DATA4,
        REG_TX_WPC_DATA5,
        REG_TX_WPC_DATA6,
        REG_TX_WPC_DATA7,
        REG_TX_WPC_CHECKSUM,
        
        REG_FOD_X0 = 0x3E,
        REG_FOD_Y0,
        REG_FOD_X1,
        REG_FOD_Y1,
        REG_FOD_X2,
        REG_FOD_Y2,
        REG_FOD_X3,
        REG_FOD_Y3,
        REG_FOD_X4,
        REG_FOD_Y4,
        REG_FOD_X5,
        REG_FOD_Y5,
        REG_FOD_X6,
        REG_FOD_Y6,
        REG_FOD_X7,
        REG_FOD_Y7,
        REG_FOD_X8,
        REG_FOD_Y8,
        REG_FOD_X9,
        REG_FOD_Y9,
        REG_FOD_X10,
        REG_FOD_Y10,
        REG_FOD_X11,
        REG_FOD_Y11,
        REG_FOD_X12,
        REG_FOD_Y12,
        REG_FOD_X13,
        REG_FOD_Y13,
        REG_FOD_X14,
        REG_FOD_Y14,
        REG_FOD_X15,
        REG_FOD_Y15,
        
        REG_OV_CLAMP_VOLTAGE = 0x62,
        REG_TX_LAST_CEP = 0x63,
        REG_TX_LAST_RPP,
        REG_TX_LAST_PCHP,
        
        REG_LDO_WPDET_CNFG = 0x70,
        
        REG_VRECT_TARGET_X0 = 0xBA,
        REG_VRECT_TARGET_Y0,
        REG_VRECT_TARGET_PMA_Y0,
        REG_VRECT_TARGET_X1,
        REG_VRECT_TARGET_Y1,
        REG_VRECT_TARGET_PMA_Y2,
        REG_VRECT_TARGET_X3,
        REG_VRECT_TARGET_Y3,
        REG_VRECT_TARGET_PMA_Y3,
        REG_VRECT_TARGET_X4,
        REG_VRECT_TARGET_Y4,
        REG_VRECT_TARGET_PMA_Y4,
        REG_VRECT_TARGET_X5,
        REG_VRECT_TARGET_Y5,
        REG_VRECT_TARGET_PMA_Y5,
        REG_VRECT_TARGET_X6,
        REG_VRECT_TARGET_Y6,
        REG_VRECT_TARGET_PMA_Y6,
        REG_VRECT_TARGET_X7,
        REG_VRECT_TARGET_Y7,
        REG_VRECT_TARGET_PMA_Y7,
        
        REG_PMA_ADV_DATA,
        REG_PMA_ADV_CRC,
        
    } registers_t;
   
    /**
     * @brief   Interrupt Bit
     * @details Interrupt Bit Low
     */
    typedef enum {
        INT_VOUT = 0x80,
        INT_VRECT = 0x40,
        INT_WDOG = 0x20,
        INT_FSK_RCVD = 0x10,
        INT_OV = 0x02,
        INT_OC = 0x01,
    } interrupt_low_t;   
   
    /**
     * @brief   Interrupt Bit
     * @details Interrupt Bit Low
     */
    typedef enum {                       
        INT_OT = 0x80,                
        INT_TX_OC = 0x40,                   
        INT_TX_OT = 0x20,                
        INT_TX_CONN = 0x08,               
        INT_TX_DEMOD_PING = 0x04,
        INT_ASK_DEMOD_IDCF = 0x02,                
        INT_ASK_DEMOD_PT = 0x01,        
    } interrupt_high_t;
   
    /**
     * @brief   ENABLE/DISABLE
     * @details Enumerated ENABLE/DISABLE
     */
    typedef enum {
        VAL_DISABLE = 0x00,
        VAL_ENABLE
    } enable_t;
    
    /**
     * @brief   LOW/HIGH
     * @details Enumerated LOW/HIGH
     */
    typedef enum {
        VAL_LOW = 0x00,
        VAL_HIGH
    } low_high_t;
            
    /**
     * @brief   System Operation Mode
     * @details Enumerated System Operation
     */
    typedef enum {
        SYS_MODE_INITIAL = 0x00,
        SYS_MODE_WPC_RX,
        SYS_MODE_PMA_RX,
        SYS_MODE_RESERVED,
        SYS_MODE_TX_PEER_PWR,
    } system_opmode_t;
            
    /**
      * MAX77950 constructor.
      *
      * @param i2c I2C object to use.
      */
    MAX77950(I2C *i2c);
 
    /**
      * MAX77950 destructor.
      */
    ~MAX77950();
  
    /**
      * @brief   Initialize MAX77950
      */
    int32_t init();
 
    /**
      * @brief   Write Register
      * @details Writes data to MAX77950 register
      *
      * @param   reg_addr Register to write
      * @param   reg_data Data to write
      * @returns 0 if no errors, -1 if error.
      */
    int32_t write_register(MAX77950::registers_t reg_addr, char reg_data);
 
    /**
      * @brief   Read Register
      * @details Reads data from MAX77950 register
      *
      * @param   reg_addr Register to read
      * @returns data if no errors, -1 if error.
      */
    int32_t read_register(MAX77950::registers_t reg_addr);
    
    /**
      * @brief   Update Register data
      * @details Update bits data of a register 
      *
      * @param   reg_no Register Number to be updated
      * @param   mask Mask Data
      * @param   reg_data bit data
      * @returns 0 if no errors, -1 if error.
      */
      
    int32_t update_register
    (MAX77950::registers_t reg_no, char reg_mask,  char reg_data);
    
    /**
      * @brief   Get Status Low
      * @details Get status register data
      *          BIT7 : LDO is ON
      *          BIT6 : V_rect is over UVLO
      *          BIT5 : Watchdog timer
      *          BIT4 : FSK Data
      *          BIT3 : Reserved
      *          BIT2 : Reserved
      *          BIT1 : Over Voltage Status
      *          BIT0 : Over Current Status
      * @param   None
      * @returns status register data.
      */
    int32_t get_status_low();    
    
    /**
      * @brief   Get Status high
      * @details Get status register data
      *          BIT7 : Over Temp
      *          BIT6 : Over Current in TX Mode
      *          BIT5 : Over Temp in TX Mode
      *          BIT4 : Reserved
      *          BIT3 : Power Transfer Established in Tx Mode
      *          BIT2 : Packet Received in Ping Phase
      *          BIT1 : Packet Received in ID
      *          BIT0 : Packet Received in Power Transfer Phase
      * @param   None
      * @returns status register data.
      */
    int32_t get_status_high();
    
    /**
      * @brief   Get Interrupt low
      * @details Get Interrupt register data
      *          BIT7 : LDO state change Interrupt
      *          BIT6 : V_rect is over UVLO Interrupt
      *          BIT5 : Watchdog timer Interrupt
      *          BIT4 : FSK Data Interrupt
      *          BIT3 : Reserved
      *          BIT2 : Reserved
      *          BIT1 : Over Voltage Interrupt
      *          BIT0 : Over Current Interrupt
      * @param   None
      * @returns status register data.
      */
    int32_t get_interrupt_low();
        
    /**
      * @brief   Get Interrupt high
      * @details Get Interrupt register data
      *          BIT7 : Over Temp Interrupt
      *          BIT6 : Over Current in TX Mode Interrupt
      *          BIT5 : Over Temp in TX Mode Interrupt
      *          BIT4 : Reserved
      *          BIT3 : Power Transfer Established in Tx Mode Interrupt
      *          BIT2 : Packet Received in Ping Phase Interrupt
      *          BIT1 : Packet Received in ID Interrupt 
      *          BIT0 : Packet Received in Power Transfer Phase Interrupt
      * @param   None
      * @returns status register data.
      */
    int32_t get_interrupt_high();
    
    /**
      * @brief   Interrupt enable low
      * @details 
      *          BIT7 : LDO state change Interrupt
      *          BIT6 : V_rect is over UVLO Interrupt
      *          BIT5 : Watchdog timer Interrupt
      *          BIT4 : FSK Data Interrupt
      *          BIT3 : Reserved
      *          BIT2 : Reserved
      *          BIT1 : Over Voltage Interrupt
      *          BIT0 : Over Current Interrupt
      * @param   interrupt_low_t
      * @returns status register data.
      */
    int32_t set_interrupt_low(MAX77950::interrupt_low_t _int);
    
    /**
      * @brief   Interrupt enable high
      * @details 
      *          BIT7 : Over Temp Interrupt
      *          BIT6 : Over Current in TX Mode Interrupt
      *          BIT5 : Over Temp in TX Mode Interrupt
      *          BIT4 : Reserved
      *          BIT3 : Power Transfer Established in Tx Mode Interrupt
      *          BIT2 : Packet Received in Ping Phase Interrupt
      *          BIT1 : Packet Received in ID Interrupt 
      *          BIT0 : Packet Received in Power Transfer Phase Interrupt
      * @param   interrupt_high_t
      * @returns status register data.
      */
    int32_t set_interrupt_high(MAX77950::interrupt_high_t _int);
    
    /**
      * @brief   Interrupt disable low
      * @details 
      *          BIT7 : LDO state change Interrupt
      *          BIT6 : V_rect is over UVLO Interrupt
      *          BIT5 : Watchdog timer Interrupt
      *          BIT4 : FSK Data Interrupt
      *          BIT3 : Reserved
      *          BIT2 : Reserved
      *          BIT1 : Over Voltage Interrupt
      *          BIT0 : Over Current Interrupt
      * @param   interrupt_low_t
      * @returns status register data.
      */
    int32_t disable_interrupt_low(MAX77950::interrupt_low_t _int);
    
    /**
      * @brief   Interrupt disable high
      * @details 
      *          BIT7 : Over Temp Interrupt
      *          BIT6 : Over Current in TX Mode Interrupt
      *          BIT5 : Over Temp in TX Mode Interrupt
      *          BIT4 : Reserved
      *          BIT3 : Power Transfer Established in Tx Mode Interrupt
      *          BIT2 : Packet Received in Ping Phase Interrupt
      *          BIT1 : Packet Received in ID Interrupt 
      *          BIT0 : Packet Received in Power Transfer Phase Interrupt
      * @param   interrupt_high_t
      * @returns status register data.
      */
    int32_t disable_interrupt_high(MAX77950::interrupt_high_t _int);
    
    /**
      * @brief   Interrupt Clear low
      * @details 
      *          BIT7 : LDO state change Interrupt
      *          BIT6 : V_rect is over UVLO Interrupt
      *          BIT5 : Watchdog timer Interrupt
      *          BIT4 : FSK Data Interrupt
      *          BIT3 : Reserved
      *          BIT2 : Reserved
      *          BIT1 : Over Voltage Interrupt
      *          BIT0 : Over Current Interrupt
      * @param   interrupt_low_t
      * @returns status register data.
      */
    int32_t clear_interrupt_low(MAX77950::interrupt_low_t _int);
    
    /**
      * @brief   Interrupt Clear high
      * @details 
      *          BIT7 : Over Temp Interrupt
      *          BIT6 : Over Current in TX Mode Interrupt
      *          BIT5 : Over Temp in TX Mode Interrupt
      *          BIT4 : Reserved
      *          BIT3 : Power Transfer Established in Tx Mode Interrupt
      *          BIT2 : Packet Received in Ping Phase Interrupt
      *          BIT1 : Packet Received in ID Interrupt 
      *          BIT0 : Packet Received in Power Transfer Phase Interrupt
      * @param   interrupt_high_t
      * @returns status register data.
      */
    int32_t clear_interrupt_high(MAX77950::interrupt_high_t _int);
    
    /**
      * @brief   Get SOC(Charge Status);
      * @details 
      *          0x00, 0x65~0xFE : Reserved
      *          0x01 ~ 0x64 : 1 ~ 100%
      *          0xFF : No battery
      * @param   None
      * @returns -1: Communication Error
                 0x00: out of range
                 0x01 ~ 0x64: SOC 1~ 100 
                 0xff:  there is no battery to charge.
      */
    int32_t get_charge_status();
    
    /**
      * @brief   Get VOUT Value
      * @details 
      *          Vout Value 12bit data
      * @param   None
      * @returns -1: Communication Error
                 0x00~0xFFF
      */
    int32_t get_vout_value();
    
    /**
      * @brief   Get VRECT Value
      * @details 
      *          V_rect Value 12bit data
      * @param   None
      * @returns -1: Communication Error
                 0x00~0xFFF
      */
    int32_t get_vrect_value();
    
    /**
      * @brief   Get ISense Value
      * @details 
      *          ISense Value 12bit data
      * @param   None
      * @returns -1: Communication Error
                 0x00~0xFFF
      */
    int32_t get_isense_value();
    
    /**
      * @brief   Get Die Temp Value
      * @details 
      *          Die Temp
      * @param   None
      * @returns -1: Communication Error
                 0x00~0xFF
      */
    int32_t get_die_temp_value();
    
    /**
      * @brief   Get AC Operation Frequency 
      * @details 
      *          AC Operation Frequency
      * @param   None
      * @returns -1: Communication Error
                 0x00~0xFFFF
      */
    int32_t get_op_freq_value();
    
    /**
      * @brief   Get System Operation Mode
      * @details 
      *          read System Operation Mode
      * @param   None
      * @returns -1: Communication Error
                 0x00: Initial State
                 0x01: WPC Rx Mode
                 0x02: PMA Rx Mode
                 0x03: Tx Mode(PeerPower);
      */
    int32_t get_systme_opmode();
    
    /**
      * @brief   Global Interrtup Clear 
      * @details 
      *          Clears all interrupt bits. After the set, it resets automatically.
      * @param   None
      * @returns Communication status.
      */
    int32_t set_gloabl_interrupt_clear();
    
    /**
      * @brief   Send Charger Status 
      * @details 
                 Send charge status packet. After the set, it resets automatically.
      * @param   None
      * @returns Communication status.
      */
    int32_t set_send_charger_status();
    
    /**
      * @brief   Send "End Power Transfer" Packet
      * @details 
                 Send End Power Transfer Packet. 
                 After the set, it resets automatically.
      * @param   None
      * @returns Communication status.
      */
    int32_t set_send_ept_packet();
    
    /**
      * @brief   Toggle LDO output
      * @details 
                  Toggle LDO output once (on to off, off to on);. 
                  After the set, it resets automatically.
      * @param   None
      * @returns Communication status.
      */
    int32_t set_toggle_ldo();
    
    /**
      * @brief   Sed Rx Data
      * @details 
                 Send WPC proprietary packet that includes PPP_Header (0x21);, 
                 data command (0x22);, and data values (0x23 to 0x26);. 
                 After the set, it resets automatically
      * @param   None
      * @returns Communication status.
      */
    int32_t set_send_rx_data();

private:
 
    I2C *i2c_;
    bool i2c_owner;
 
};

#endif /* _MAX77950_H_ */