/*******************************************************************************
 * Copyright (C) 2018 Maxim Integrated Products, Inc., All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
 * OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 *
 * Except as contained in this notice, the name of Maxim Integrated
 * Products, Inc. shall not be used except as stated in the Maxim Integrated
 * Products, Inc. Branding Policy.
 *
 * The mere transfer of this software does not imply any licenses
 * of trade secrets, proprietary technology, copyrights, patents,
 * trademarks, maskwork rights, or any other form of intellectual
 * property whatsoever. Maxim Integrated Products, Inc. retains all
 * ownership rights.
 *******************************************************************************
 */
#include "max77816.h"
 
/***** Definitions *****/
#define I2C_ADDR            (0x18<<1)
 
//******************************************************************************
MAX77816::MAX77816(PinName sda, PinName scl)
{
    i2c_ = new I2C(sda, scl);
    i2c_owner = true;
 
    i2c_->frequency(400000);
}
 
//******************************************************************************
MAX77816::MAX77816(I2C *i2c) :
    i2c_(i2c)
{
    i2c_owner = false;
}
 
//******************************************************************************
MAX77816::~MAX77816()
{
    if(i2c_owner) {
        delete i2c_;
    }
}
 
//******************************************************************************
int32_t MAX77816::init()
{ 
    return 0;
}
 
//******************************************************************************
int32_t MAX77816::writeReg(MAX77816::registers_t reg_addr, char reg_data)
{
    char data[2];
 
    data[0] = reg_addr;
    data[1] = reg_data;
    if (i2c_->write(I2C_ADDR, data, 2) != 0) {
        return -1;
    }
 
    return 0;
}
 
//******************************************************************************
int32_t MAX77816::readReg(MAX77816::registers_t reg_addr)
{
    char data;
 
    data = reg_addr;
    if (i2c_->write(I2C_ADDR, &data, 1, true) != 0) {
        return -1;
    }
 
    if (i2c_->read(I2C_ADDR | 0x01, &data, 1) != 0) {
        return -1;
    }
 
    return (0x0 + data);
}



//*****************************************************************************
int32_t MAX77816::readBlock(MAX77816::registers_t startReg, MAX77816::registers_t stopReg, uint8_t *data)
{
    int32_t rtnVal = -1;
    int32_t numBytes = ((stopReg - startReg) + 1);
    char packet[] = {static_cast<char>(startReg)};
    
    if(i2c_->write(I2C_ADDR, packet, 1) == 0)
    {
        rtnVal = i2c_->read(I2C_ADDR | 0x01, reinterpret_cast<char *>(data), numBytes);
    }
    
    return rtnVal;
}


//*****************************************************************************
int32_t MAX77816::writeBlock(MAX77816::registers_t startReg, MAX77816::registers_t stopReg, const uint8_t *data)
{
    int32_t numBytes = ((stopReg - startReg) + 1);
    char packet[numBytes + 1];
    
    packet[0] = static_cast<char>(startReg);
    
    memcpy(packet + 1, data, numBytes);
    
    return i2c_->write(I2C_ADDR, packet, sizeof(packet));
}


//*****************************************************************************
int32_t MAX77816::getVersion()
{
    int32_t rData;
    
    rData = readReg(REG_0);
    if(rData< 0)
        return rData;
        
    return ((rData >> 4) & 0x7);
}


//*****************************************************************************
int32_t MAX77816::getChipRevision()
{
    int32_t rData;
    
    rData = readReg(REG_0);
    if(rData< 0)
        return rData;
        
    return ((rData >> 0) & 0xf);
}

//*****************************************************************************
int32_t MAX77816::getStatus()
{
    int32_t rData;
    
    rData = readReg(REG_1);
    if(rData< 0)
        return rData;
        
    return ((rData >> 0) & 0xf);
}

//*****************************************************************************
int32_t MAX77816::setVout(uint8_t reg_data)
{
    int32_t rData;
    
    rData = writeReg(REG_4, (reg_data & 0x7f));
    return rData;
}

//*****************************************************************************
int32_t MAX77816::setVoutH(uint8_t reg_data)
{
    int32_t rData;
    
    rData = writeReg(REG_5, (reg_data & 0x7f));
    return rData;
}

//*****************************************************************************
int32_t MAX77816::setIntMask(uint8_t reg_data)
{
    int32_t rData;
    
    rData = writeReg(REG_6, (reg_data & 0x7f));
    return rData;
}

int32_t MAX77816::setIntMask()
{
    int32_t rData;
    
    rData = readReg(REG_6);
    if(rData< 0)
        return rData;
        
    return ((rData >> 0) & 0xf);
}
//*****************************************************************************
int32_t MAX77816::getInt()
{
    int32_t rData;
    
    rData = readReg(REG_7);
    if(rData< 0)
        return rData;
        
    return ((rData >> 0) & 0xf);
}
