/*******************************************************************************
 * Copyright (C) 2018 Maxim Integrated Products, Inc., All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
 * OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 *
 * Except as contained in this notice, the name of Maxim Integrated
 * Products, Inc. shall not be used except as stated in the Maxim Integrated
 * Products, Inc. Branding Policy.
 *
 * The mere transfer of this software does not imply any licenses
 * of trade secrets, proprietary technology, copyrights, patents,
 * trademarks, maskwork rights, or any other form of intellectual
 * property whatsoever. Maxim Integrated Products, Inc. retains all
 * ownership rights.
 *******************************************************************************
 */
 #include "max8971.h"
 
/***** Definitions *****/
#define I2C_ADDR            (0x35<<1)
 
/**
  * max8971 constructor.
  *
  * @param i2c I2C object to use.
  */
MAX8971::MAX8971(I2C *i2c) :
    i2c_(i2c)
{
    i2c_owner = false;
}
 
/**
  * max8971 destructor.
  */
MAX8971::~MAX8971()
{
    if(i2c_owner) {
        delete i2c_;
    }
}

/**
  * @brief   Initialize max8971
  */
int32_t MAX8971::init()
{     
    // insert initial setting functions
    return 0;
}
 
/**
  * @brief   Read Register
  * @details Reads data from max8971 register
  *
  * @param   reg_addr Register to read
  * @returns data if no errors, -1 if error.
  */
int32_t MAX8971::read_register(MAX8971::registers_t reg_no)
{
    char data;
 
    data = reg_no;
    if (i2c_->write(I2C_ADDR, &data, 1, true) != 0) {
        return -1;
    }
 
    if (i2c_->read(I2C_ADDR | 0x01, &data, 1) != 0) {
        return -1;
    }
 
    return (0x0 + data);
}
 
/**
  * @brief   Write Register
  * @details Writes data to MAX77756 register
  *
  * @param   reg_addr Register to write
  * @param   reg_data Data to write
  * @returns 0 if no errors, -1 if error.
  */
int32_t MAX8971::write_register(MAX8971::registers_t reg_no, char reg_data)
{
    char data[2];
 
    data[0] = reg_no;
    data[1] = reg_data;
    if (i2c_->write(I2C_ADDR, data, 2) != 0) {
        return -1;
    }
 
    return 0;
}

/**
  * @brief   Update Register data
  * @details Update bits data of a register 
  *
  * @param   reg_no Register Number to be updated
  * @param   mask Mask Data
  * @param   reg_data bit data
  * @returns 0 if no errors, -1 if error.
  */
int32_t MAX8971::update_register
(MAX8971::registers_t reg_no, char reg_mask,  char reg_data)
{
    int32_t data;
 
    data = read_register(reg_no);
    if(data < 0)
        return -1;
    
    data &= ~reg_mask;
    data |= reg_data;
    
    data = write_register(reg_no, (char)(data & 0xff));
    if(data < 0)
        return -1;
    return 0;
}

/**
  * @brief   Get Interrupt
  * @details Get status register data
  *          BIT7 : AICL_I DC Interrupt
  *          BIT6 : TOPOFF Topoff Interrupt
  *          BIT5 : DC Overvolatage Interrupt
  *          BIT4 : DC Undervoltage Interrupt
  *          BIT3 : Charge Current Interrupt
  *          BIT2 : Battery Interrupt
  *          BIT1 : Thermistor Interrupt
  *          BIT0 : Power-Up OK Interrupt
  * @param   None
  * @returns Interrupt register data.
  */
int32_t MAX8971::get_interrupt()
{    
    int32_t data;
    
    data = read_register(REG_CHG_STAT);
    if(data < 0)
        return -1;
    return (data & 0xFF);
}

/**
  * @brief   Set Interrupt Mask
  * @details Get status register data
  *          BIT7 : AICL_I DC Interrupt
  *          BIT6 : TOPOFF Topoff Interrupt
  *          BIT5 : DC Overvolatage Interrupt
  *          BIT4 : DC Undervoltage Interrupt
  *          BIT3 : Charge Current Interrupt
  *          BIT2 : Battery Interrupt
  *          BIT1 : Thermistor Interrupt
  *          BIT0 : Reserved
  * @param   Register Number, Interrupt Bit
  * @returns Interrupt register data.
  */
int32_t MAX8971::set_interrupt_mask
(MAX8971::registers_t reg_no, MAX8971::int_bit_t interrupt_bit)
{   
    char mask_bit;
    int32_t data;
    
    if(interrupt_bit <VAL_THM_I_BIT)
        return -1;
        
    mask_bit = 0x01 << interrupt_bit;
    data = update_register(REG_CHGINT_MASK, mask_bit, mask_bit);
    if(data < 0)
        return -1;
        
    data = read_register(REG_CHGINT_MASK);
    if(data < 0)
        return -1;
    return (data & 0xFF);
}

/**
  * @brief   Set Interrupt UnMask
  * @details Get status register data
  *          BIT7 : AICL_I DC Interrupt
  *          BIT6 : TOPOFF Topoff Interrupt
  *          BIT5 : DC Overvolatage Interrupt
  *          BIT4 : DC Undervoltage Interrupt
  *          BIT3 : Charge Current Interrupt
  *          BIT2 : Battery Interrupt
  *          BIT1 : Thermistor Interrupt
  *          BIT0 : Reserved
  * @param   Register Number, Interrupt Bit
  * @returns Interrupt register data.
  */
int32_t MAX8971::set_interrupt_unmask
(MAX8971::registers_t reg_no, MAX8971::int_bit_t interrupt_bit)
{   
    char mask_bit;
    int32_t data;
    
    if(interrupt_bit <VAL_THM_I_BIT)
        return -1;
        
    mask_bit = 0x01 << interrupt_bit;
    data = update_register(REG_CHGINT_MASK, mask_bit, 0);
    if(data < 0)
        return -1;
        
    data = read_register(REG_CHGINT_MASK);
    if(data < 0)
        return -1;
    return (data & 0xFF);
}

/**
  * @brief   Get status
  * @details Get status register data
  *          BIT7 : DCV_OK DC Input Voltage Status
  *          BIT6 : DCI_OK DC Input Current Status
  *          BIT5 : DCOVP_OK DC OVP Status
  *          BIT4 : DCUVP_OK DC UVP Status
  *          BIT3 : CHG_OK Charger Status
  *          BIT2 : BAT_OK Battery Status
  *          BIT1 : THM_OK Thermistor Status
  *          BIT0 : RESERVED
  * @param   None
  * @returns status register data.
  */
int32_t MAX8971::get_status()
{    
    int32_t data;
    
    data = read_register(REG_CHG_STAT);
    if(data < 0)
        return -1;
    return (data & 0xFE);
}


/**
  * @brief   Get Details 1
  * @details Get Details 1 register data
  *          BIT7 : DC Details
  *          BIT6 : DC Current
  *          BIT5 : DC OVP
  *          BIT4 : DC UVP
  *          BIT3 : RESERVED
  *          BIT2-0 : Thermistor Details
  * @param   None
  * @returns status details 1 register data.
  */
int32_t MAX8971::get_details1()
{    
    int32_t data;
    
    data = read_register(REG_DETAILS1);
    if(data < 0)
        return -1;
    return (data & 0xF7);
}

/**
  * @brief   Get Details 2
  * @details Get Details 2 register data
  *          BIT7-6 : Reserved
  *          BIT5-4 : Battery Details
  *          BIT3-0 : Charger Details
  * @param   None
  * @returns status details 2 register data.
  */
int32_t MAX8971::get_details2()
{    
    int32_t data;
    
    data = read_register(REG_DETAILS2);
    if(data < 0)
        return -1;
    return (data & 0x3F);
}

/**
  * @brief   Set DCMON_DIS
  * @details Set to disable the monitoring of input voltage
  *          by the input power limiter
  * @param   None
  * @returns DCMON_DIS bit data.
  */
int32_t MAX8971::set_disable_dc_monitor()
{    
    int32_t shift = 1;   
    char mask_bit = 0x01 << shift;
    int32_t data;
    
    data = update_register(REG_CHGCNTL1, mask_bit, 1);
    if(data < 0)
        return -1;
        
    data = read_register(REG_CHGCNTL1);
    if(data < 0)
        return -1;
    return ( (data & mask_bit) >> shift);
}

/**
  * @brief   UnSet DCMON_DIS
  * @details UnSet to enable the monitoring of input voltage
  *          by the input power limiter
  * @param   None
  * @returns DCMON_DIS bit data.
  */
int32_t MAX8971::unset_disable_dc_monitor()
{       
    int32_t shift = 1;   
    char mask_bit = 0x01 << shift;
    int32_t data;
    
    data = update_register(REG_CHGCNTL1, mask_bit, 0);
    if(data < 0)
        return -1;
        
    data = read_register(REG_CHGCNTL1);
    if(data < 0)
        return -1;
    return ( (data & mask_bit) >> shift);
}

/**
  * @brief   Set USB SUSPEND
  * @details Set to USB Suspend bit 
  * @param   None
  * @returns USB SUSPEND bit data.
  */
int32_t MAX8971::set_usb_suspend()
{    
    int32_t data;
    
    data = update_register(REG_CHGCNTL1, 0x01, 1);
    if(data < 0)
        return -1;
        
    data = read_register(REG_CHGCNTL1);
    if(data < 0)
        return -1;
    return ( data & 0x01);
}

/**
  * @brief   UnSet USB SUSPEND
  * @details UnSet to USB Suspend bit 
  * @param   None
  * @returns USB SUSPEND bit data.
  */
int32_t MAX8971::unset_usb_suspend()
{    
    int32_t data;
        
    data = update_register(REG_CHGCNTL1, 0x01, 0);
    if(data < 0)
        return -1;
        
    data = read_register(REG_CHGCNTL1);
    if(data < 0)
        return -1;
    return ( data & 0x01);
}


/**
  * @brief   Set Fast Charge Current
  * @details Set to CHGCC bit to control Fast Charge Current
  * @param   int 0-1550
  * @returns Fast Charge Current Bit Data.
  */
int32_t MAX8971::set_fast_charge_current(int current)
{    
    char mask_bit = 0x1f;
    char value_bit = 0x0;
    int32_t data;
    
    switch(current)
    {
        case 0 ... 249:
            value_bit = 0x0;
            break;
        case 250 ... 1550:
            value_bit = current/50;
            break;
        default: 
            value_bit = 0x1f;
            break;         
    }    
    
    data = update_register(REG_FCHGCRNT, mask_bit, value_bit);
    if(data < 0)
        return -1;
        
    data = read_register(REG_FCHGCRNT);
    if(data < 0)
        return -1;
    return ( data & mask_bit);
}

/**
  * @brief   Set Fast Charge Timer Duration
  * @details Set to FCHGT bit to control Fast Charge Timer
  * @param   fast_charge_timer_t
  * @returns Fast Charge Timer Bit Data.
  */
int32_t MAX8971::set_fast_charge_timer(MAX8971::fast_charge_timer_t fc_time)
{    
    char mask_bit = 0xE0;    
    int32_t shift = 5;   
    int32_t data;       
    
    data = update_register(REG_FCHGCRNT, mask_bit, fc_time<<shift);
    if(data < 0)
        return -1;
        
    data = read_register(REG_FCHGCRNT);
    if(data < 0)
        return -1;
    return ( (data & mask_bit) >> shift);
}

/**
  * @brief   Set Fast Restart Threshold
  * @details Set to CHGRSTRT bit to control Fast Restart Threshold
  * @param   fast_charge_restart_threshold_t
  * @returns CHGRSTRT bit Data.
  */
int32_t MAX8971::set_fast_charge_restart_threshold
(MAX8971::fast_charge_restart_threshold_t voltage)
{    
    char mask_bit = 0x40;    
    int32_t shift = 6;   
    int32_t data;       
    
    data = update_register(REG_DCCRNT, mask_bit, voltage<<shift);
    if(data < 0)
        return -1;
        
    data = read_register(REG_DCCRNT);
    if(data < 0)
        return -1;
    return ( (data & mask_bit) >> shift);
}

/**
  * @brief   Set Input Current Limit
  * @details Set to DCILMT to control Input Current Limit
  *          0x00 - 0x09 : 100mA
  *          0x10 - 1x3C : 250 mA - 1500 mA with 25m A
  * @param   int 0-1500
  * @returns DCILMT bit Data.
  */
int32_t MAX8971::set_input_current_limit(int limit)
{    
    char mask_bit = 0x3F;    
    char value_bit = 0;   
    int32_t data; 
        
    switch(limit)
    {
        case 0 ... 99:
            value_bit = 0x3f; // disable
            break;
        case 100 ... 249:
            value_bit = 0x09;
            break;
        case 250 ... 1500:
            value_bit = limit/25;
            break;
        default: 
            value_bit = 0x3c; //maximum current is 1500mA(0x3C)
            break;         
    }    
    
    data = update_register(REG_DCCRNT, mask_bit, value_bit);
    if(data < 0)
        return -1;
        
    data = read_register(REG_DCCRNT);
    if(data < 0)
        return -1;
    return ( data & mask_bit);
}

/**
  * @brief   Set Topoff Timer
  * @details Set to TOFFT bit to control Topoff Timer
  * @param   top_off_timer_t
  * @returns TOFFT bit Data.
  */
int32_t MAX8971::set_top_off_timer(MAX8971::top_off_timer_t time)
{    
    char mask_bit = 0xE0;    
    int32_t shift = 5;   
    int32_t data;       
    
    data = update_register(REG_TOPOFF, mask_bit, time<<shift);
    if(data < 0)
        return -1;
        
    data = read_register(REG_TOPOFF);
    if(data < 0)
        return -1;
    return ( (data & mask_bit) >> shift);
}


/**
  * @brief   Set Topoff Current Threshold
  * @details Set to TOFFS bit to control Topoff Current Threshold
  * @param   top_off_current_threshold_t
  * @returns TOFFS bit Data.
  */
int32_t MAX8971::set_top_off_current_threshold
(MAX8971::top_off_current_threshold_t current)
{    
    char mask_bit = 0x0C;    
    int32_t shift = 2;   
    int32_t data;       
    
    data = update_register(REG_TOPOFF, mask_bit, current<<shift);
    if(data < 0)
        return -1;
        
    data = read_register(REG_TOPOFF);
    if(data < 0)
        return -1;
    return ( (data & mask_bit) >> shift);
}


/**
  * @brief   Set Charge Termination Voltage
  * @details Set to CHGCV bit to control Charge Termination Voltage in CV Mode
  * @param   charger_termination_voltage_t
  * @returns CHGCV bit Data.
  */
int32_t MAX8971::set_charge_termination_voltage
(MAX8971::charger_termination_voltage_t voltage)
{    
    char mask_bit = 0x03;    
    int32_t shift = 0;   
    int32_t data;       
    
    data = update_register(REG_TOPOFF, mask_bit, voltage<<shift);
    if(data < 0)
        return -1;
        
    data = read_register(REG_TOPOFF);
    if(data < 0)
        return -1;
    return ( (data & mask_bit) >> shift);
}

/**
  * @brief   Set Die-Temperature Regulation Loop Set Point
  * @details Set to REGTEMP bit to control 
  *          Die-Temperature Regulation Loop Set Point
  * @param   die_temp_reggulation_point_t
  * @returns REGTEMP bit Data.
  */
int32_t MAX8971::set_charge_termination_voltage
(MAX8971::die_temp_reggulation_point_t degree)
{    
    char mask_bit = 0xC0;    
    int32_t shift = 6;   
    int32_t data;       
    
    data = update_register(REG_TEMPREG, mask_bit, degree<<shift);
    if(data < 0)
        return -1;
        
    data = read_register(REG_TEMPREG);
    if(data < 0)
        return -1;
    return ( (data & mask_bit) >> shift);
}

/**
  * @brief   Set Thermistor Monitor Configuration
  * @details Set to THM_CNFG bit to control Thermistor Monitor Configuration
  * @param   thermistor_monitor_config_t
  * @returns REGTEMP bit Data.
  */
int32_t MAX8971::set_thermistor_monitor
(MAX8971::thermistor_monitor_config_t enable)
{    
    char mask_bit = 0x08;    
    int32_t shift = 3;   
    int32_t data;       
    
    data = update_register(REG_TEMPREG, mask_bit, enable<<shift);
    if(data < 0)
        return -1;
        
    data = read_register(REG_TEMPREG);
    if(data < 0)
        return -1;
    return ( (data & mask_bit) >> shift);
}

/**
  * @brief   Set JEITA Safty Region
  * @details Set SAFETYREG bit to control JEITA Safty Region
  * @param   jeita_safety_selection_t
  * @returns SAFETYREG bit Data.
  */
int32_t MAX8971::set_jeita_safety_region
(MAX8971::jeita_safety_selection_t enable)
{    
    char mask_bit = 0x01;    
    int32_t shift = 0;   
    int32_t data;       
    
    data = update_register(REG_TEMPREG, mask_bit, enable<<shift);
    if(data < 0)
        return -1;
        
    data = read_register(REG_TEMPREG);
    if(data < 0)
        return -1;
    return ( (data & mask_bit) >> shift);
}

/**
  * @brief   Set Charger-Setting Protection
  * @details Set CPROT bit to control Charger-Setting Protection
  * @param   charger_setting_protection_t
  * @returns CPROT bit Data.
  */
int32_t MAX8971::set_charger_setting_protection
(MAX8971::charger_setting_protection_t enable)
{    
    char mask_bit = 0x0C;    
    int32_t shift = 2;   
    int32_t data;       
    
    data = update_register(REG_PROTCMD, mask_bit, enable<<shift);
    if(data < 0)
        return -1;
        
    data = read_register(REG_PROTCMD);
    if(data < 0)
        return -1;
    return ( (data & mask_bit) >> shift);
}
