/*******************************************************************************
 * Copyright (C) 2017 Maxim Integrated Products, Inc., All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
 * OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 *
 * Except as contained in this notice, the name of Maxim Integrated
 * Products, Inc. shall not be used except as stated in the Maxim Integrated
 * Products, Inc. Branding Policy.
 *
 * The mere transfer of this software does not imply any licenses
 * of trade secrets, proprietary technology, copyrights, patents,
 * trademarks, maskwork rights, or any other form of intellectual
 * property whatsoever. Maxim Integrated Products, Inc. retains all
 * ownership rights.
 *******************************************************************************
 */
 
#ifndef _MAX77756_H_
#define _MAX77756_H_
 
#include "mbed.h"
 
class MAX77756
{
 
public:
 
    /**
     * @brief   Register Addresses
     * @details Enumerated MAX77756 register addresses
     */
    typedef enum {
        REG_CONFIGA = 0x00,
        REG_CONFIGB 
    } registers_t;
 
    /**
     * @brief   Spread Spectrum Control
     * @details Enumerated Spread Spectrum Module
     */
    typedef enum {
        S_SPECTRUM_OFF = 0x00,
        S_SPECTRUM_ON
    } SpreadSpectrum_t;
 
    /**
     * @brief   Soft Start Control
     * @details Enumerated Start-Up Ramp Time for the Regulator
     */
    typedef enum {
        S_START_8_ms = 0x00,
        S_START_4ms 
    } SoftStart_t;

    /**
     * @brief   Peak Current Limit
     * @details Enumerated High Side DMOS Peak Current Limit
     */
    typedef enum {
        I_PEAK_LIMIT_700_mA = 0x00,
        I_PEAK_LIMIT_800_mA,
        I_PEAK_LIMIT_900_mA,
        I_PEAK_LIMIT_1000_mA
    } PeakCurrent_t;
 
    /**
     * @brief   Enable Logic
     * @details Enumerated Enable Logic Control between EN_BIT and EN Pin
     */
    typedef enum {
        EN_LOGICAL_OR = 0x00,
        EN_LOGICAL_AND
    } EnableLogic_t;
  
     /**
     * @brief   Enable Bit
     * @details Enumerated Enable for Regulator
     */
    typedef enum {
        ENABLE = 0x00,
        DISABLE
    } Enable_t;
 
    /**
     * MAX77756 constructor.
     *
     * @param sda mbed pin to use for SDA line of I2C interface.
     * @param scl mbed pin to use for SCL line of I2C interface.
     */
    MAX77756(PinName sda, PinName scl);
 
    /**
     * MAX77756 constructor.
     *
     * @param i2c I2C object to use.
     */
    MAX77756(I2C *i2c);
 
    /**
     * MAX44000 destructor.
     */
    ~MAX77756();
 
    /**
     * @brief   Initialize MAX77756
     */
    int32_t init();
 
    /**
     * @brief   Write Register
     * @details Writes data to MAX77756 register
     *
     * @param   reg_addr Register to write
     * @param   reg_data Data to write
     * @returns 0 if no errors, -1 if error.
     */
    int32_t writeReg(MAX77756::registers_t reg_addr, char reg_data);
 
    /**
     * @brief   Read Register
     * @details Reads data from MAX77756 register
     *
     * @param   reg_addr Register to read
     * @returns data if no errors, -1 if error.
     */
    int32_t readReg(MAX77756::registers_t reg_addr);
    
    /**
     * @brief   Regultor Enable
     * @details Control EN_BIT to Enable/Disable Regulator 
     *
     * @param   enable value to control
     * @returns 0 if no errors, -1 if error.
     */
    int32_t outEnable(MAX77756::Enable_t enable);
    
    /**
     * @brief   Set Output Voltage
     * @details Control Output Voltage.  
     *
     * @param   Output Voltage from 1500 to 7500 (mV)
     * @returns 0 if no errors, -1 if error.
     */
    int32_t setVout(uint32_t outVoltage);
    
    /**
     * @brief   Config MAX77756
     * @details Set Spread-Spectrum, SoftStart, Peak Current and Enable Logic.  
     *
     * @param   Spread Spectrum Configuration Value
     * @param   Soft Start Configuration Value
     * @param   High Side DMOS Peak Current Configuration Value
     * @param   Enable Logic Configuration Value
     * @returns 0 if no errors, -1 if error.
     */
    int32_t config(MAX77756::SpreadSpectrum_t sspectrum, 
                   MAX77756::SoftStart_t sstart,
                   MAX77756::PeakCurrent_t ipeak, 
                   MAX77756::EnableLogic_t enlogic);
    
private:
 
    I2C *i2c_;
    bool i2c_owner;
 
};
 
#endif /* _MAX77756_H_ */
 